package org.gcube.data.analysis.tabulardata.operation.data.transformation;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.model.column.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.column.ColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.AnnotationColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.AttributeColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.CodeColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.CodeDescriptionColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.CodeNameColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.DimensionColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.MeasureColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.TimeDimensionColumnType;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ImmutableLocalizedText;
import org.gcube.data.analysis.tabulardata.model.table.TableId;
import org.gcube.data.analysis.tabulardata.model.table.TableType;
import org.gcube.data.analysis.tabulardata.model.table.type.CodelistTableType;
import org.gcube.data.analysis.tabulardata.model.table.type.DatasetTableType;
import org.gcube.data.analysis.tabulardata.model.table.type.DatasetViewTableType;
import org.gcube.data.analysis.tabulardata.model.table.type.GenericTableType;
import org.gcube.data.analysis.tabulardata.model.table.type.TimeCodelistTableType;
import org.gcube.data.analysis.tabulardata.operation.OperationId;
import org.gcube.data.analysis.tabulardata.operation.factories.types.TableTransformationWorkerFactory;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.parameters.Cardinality;
import org.gcube.data.analysis.tabulardata.operation.parameters.CompositeParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.LocalizedTextChoiceParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.TargetColumnParameter;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;

@Singleton
public class GroupByFactory extends TableTransformationWorkerFactory{
	
	private static final OperationId OPERATION_ID = new OperationId(3006);
	
	CubeManager cubeManager;
	
	DatabaseConnectionProvider connectionProvider;
	SQLExpressionEvaluatorFactory sqlExpressionEvaluatorFactory;
	
	@Override
	public Worker createWorker(OperationInvocation arg0)
			throws InvalidInvocationException {
		performBaseChecks(arg0);	
		checkParameters(arg0);
		return new GroupBy(arg0, cubeManager, connectionProvider,sqlExpressionEvaluatorFactory);
	}
	
	
	private void checkParameters(OperationInvocation invocation)throws InvalidInvocationException{
		TableId targetTableId=invocation.getTargetTableId();
		
		Object toAggregateObj=invocation.getParameterInstances().get(GroupByFactory.GROUPBY_COLUMNS.getIdentifier());
		if(toAggregateObj instanceof Iterable<?>){
			Iterable<ColumnReference> cols=(Iterable<ColumnReference>) toAggregateObj;
			for(ColumnReference ref:cols)
				if(!ref.getTableId().equals(targetTableId)) throw new InvalidInvocationException(invocation, "Inconsistent target table and aggregation column references");			
		}else{
			if(!((ColumnReference)toAggregateObj).getTableId().equals(targetTableId))throw new InvalidInvocationException(invocation, "Inconsistent target table and aggregation column references");
		}
		
		Object compositeObj=invocation.getParameterInstances().get(GroupByFactory.AGGREGATE_FUNCTION_TO_APPLY.getIdentifier());
		if(compositeObj instanceof Iterable<?>)
			for(Object mapObj:(Iterable<?>)compositeObj){
				if(!((ColumnReference)((Map<String, Object>) mapObj).get(GroupByFactory.TO_AGGREGATE_COLUMNS.getIdentifier())).getTableId().equals(targetTableId)) 
					throw new InvalidInvocationException(invocation, "Inconsistent target table and to aggregate values column references");
			}
		else if(!((ColumnReference)((Map<String, Object>) compositeObj).get(GroupByFactory.TO_AGGREGATE_COLUMNS.getIdentifier())).getTableId().equals(targetTableId)) 
			throw new InvalidInvocationException(invocation, "Inconsistent target table and to aggregate values column references");
	}


	
	
	
	@Inject
	public GroupByFactory(CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider,
			SQLExpressionEvaluatorFactory sqlExpressionEvaluatorFactory) {
		super();
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		this.sqlExpressionEvaluatorFactory=sqlExpressionEvaluatorFactory;
	}







	public static final TargetColumnParameter GROUPBY_COLUMNS= new TargetColumnParameter("groupByColumns", "Group By Columns", "Columns to use as a key of grouping", 
																		new Cardinality(1, Integer.MAX_VALUE), 
																		Arrays.asList(new TableType[]{
																		new CodelistTableType(),
																		new DatasetTableType(),
																		new DatasetViewTableType(),
																		new GenericTableType(),
																		new TimeCodelistTableType()}),
																		Arrays.asList(new ColumnType[]{
																				new AnnotationColumnType(),
																				new AttributeColumnType(),
																				new CodeColumnType(),
																				new CodeDescriptionColumnType(),
																				new CodeNameColumnType(),
																				new DimensionColumnType(),
																				new MeasureColumnType(),
																				new TimeDimensionColumnType()}));

	public static final LocalizedTextChoiceParameter FUNCTION_PARAMETER=new LocalizedTextChoiceParameter("functionParameter", "Function Parameter", "Aggregation function to apply",
			Cardinality.ONE, Arrays.asList(new ImmutableLocalizedText[]{
				new ImmutableLocalizedText(AggregationFunction.AVG+""),
				new ImmutableLocalizedText(AggregationFunction.COUNT+""),
				new ImmutableLocalizedText(AggregationFunction.FIRST+""),
				new ImmutableLocalizedText(AggregationFunction.LAST+""),
				new ImmutableLocalizedText(AggregationFunction.MAX+""),
				new ImmutableLocalizedText(AggregationFunction.MIN+""),
				new ImmutableLocalizedText(AggregationFunction.SUM+""),
			}));
	
	public static final TargetColumnParameter TO_AGGREGATE_COLUMNS=new TargetColumnParameter("functionMember", "To Aggregate values", "Aggregation function member",
			Cardinality.ONE,
			Arrays.asList(new TableType[]{
					new CodelistTableType(),
					new DatasetTableType(),
					new DatasetViewTableType(),
					new GenericTableType(),
					new TimeCodelistTableType()}),
					Arrays.asList(new ColumnType[]{
							new AnnotationColumnType(),
							new AttributeColumnType(),
							new CodeColumnType(),
							new CodeDescriptionColumnType(),
							new CodeNameColumnType(),
							new DimensionColumnType(),
							new MeasureColumnType(),
							new TimeDimensionColumnType()}));

	
	public static final CompositeParameter AGGREGATE_FUNCTION_TO_APPLY= new CompositeParameter("aggregationFunctions", "Aggregation Functions", "Aggregation Function to apply",
			new Cardinality(1, Integer.MAX_VALUE),
			Arrays.asList(new Parameter[]{
					FUNCTION_PARAMETER,
					TO_AGGREGATE_COLUMNS
			}));
			
	
	
	private List<Parameter> parameters = Arrays.asList(
			GROUPBY_COLUMNS,
			AGGREGATE_FUNCTION_TO_APPLY
			);
	
	
	
	
	@Override
	protected String getOperationDescription() {
		return "Aggregate selected columns values, applying the specified functions to relative selected member columns";
	}


	@Override
	protected String getOperationName() {
		return "Aggregation";
	}


	@Override
	protected List<Parameter> getParameters() {		
		return parameters;
	}
	
	@Override
	protected OperationId getOperationId() {		
		return OPERATION_ID;
	}
}
