package org.gcube.data.analysis.tabulardata.operation.data.add;

import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.expression.functions.Cast;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.column.type.IdColumnType;
import org.gcube.data.analysis.tabulardata.model.datatype.DataType;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.type.GenericTableType;
import org.gcube.data.analysis.tabulardata.operation.OperationHelper;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.ImmutableWorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;

public class UnionWorker extends Worker {

	private CubeManager cubeManager;
	private DatabaseConnectionProvider connectionProvider;
	private SQLExpressionEvaluatorFactory evaluatorFactory;
	public UnionWorker(OperationInvocation sourceInvocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider,SQLExpressionEvaluatorFactory evaluatorFactory) {
		super(sourceInvocation);
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		this.evaluatorFactory= evaluatorFactory;
	}


	private Table targetTable=null;
	private Table sourceTable=null;
	private Table resultTable=null;
	private Table diffTable=null;
	private Map<Column,Column> colMappings=new HashMap<>();

	private String insertQuery=null;
	private String defaultsQuery=null;

	@Override
	protected WorkerResult execute() throws WorkerException {
		init();
		updateProgress(0.1f,"Initialized process");
		resultTable=cubeManager.createTable(targetTable.getTableType()).like(targetTable, true).create();
		diffTable=cubeManager.createTable(new GenericTableType()).create();
		updateProgress(0.5f,"Importing data");
		formQueries();
		try{
			SQLHelper.executeSQLCommand(insertQuery, connectionProvider);
			updateProgress(0.8f,"Finalizing data");
			if(defaultsQuery!=null)SQLHelper.executeSQLCommand(defaultsQuery, connectionProvider);
			updateProgress(0.9f,"Completing");
			return new ImmutableWorkerResult(resultTable, diffTable);
		}catch(SQLException e){
			throw new WorkerException("Unable to execute queries", e);
		}
	}


	private void init(){
		targetTable =cubeManager.getTable(getSourceInvocation().getTargetTableId());
		List<Map<String,Object>> mappings=UnionFactory.getMappings(getSourceInvocation());		
		for(Map<String,Object> mapping:mappings){
			// all source columns must belong to same table
			ColumnReference sourceRef=(ColumnReference) mapping.get(UnionFactory.SOURCE_COLUMN_PARAMETER.getIdentifier());
			if(sourceTable==null)sourceTable=cubeManager.getTable(sourceRef.getTableId());

			// all target columns must belong to target table
			ColumnReference targetRef=(ColumnReference) mapping.get(UnionFactory.TARGET_COLUMN_PARAMETER.getIdentifier());
			// all source-target columns must have compatible data types

			Column sourceCol=sourceTable.getColumnById(sourceRef.getColumnId());		
			Column targetCol=targetTable.getColumnById(targetRef.getColumnId());

			colMappings.put(targetCol, sourceCol);
		}
	}

	private void formQueries(){
		StringBuilder castedSelection=new StringBuilder();
		for(Entry<Column,Column> mapping:colMappings.entrySet()){			
			DataType targetDataType=mapping.getKey().getDataType();
			castedSelection.append(evaluatorFactory.getEvaluator(
					new Cast(sourceTable.getColumnReference(mapping.getValue()), targetDataType)).evaluate()+",");
		}

		castedSelection.deleteCharAt(castedSelection.lastIndexOf(","));

		insertQuery=String.format("WITH inserted AS (INSERT INTO %s(%s) SELECT %s from %s RETURNING *) "+
				"INSERT INTO %s(id) SELECT id from inserted",
				resultTable.getName(),
				OperationHelper.getColumnNamesSnippet(colMappings.keySet()),
				castedSelection.toString()
				,sourceTable.getName(),
				diffTable.getName());


		StringBuilder defaultValuesSnippet=new StringBuilder();
		boolean setDefaults=false;
		for(Column col:resultTable.getColumnsExceptTypes(IdColumnType.class)){
			if(!colMappings.containsKey(col)){
				setDefaults=true;
				String defaultValue=evaluatorFactory.getEvaluator(col.getDataType().getDefaultValue()).evaluate();
				defaultValuesSnippet.append(col.getName()+" = "+defaultValue+",");
			}
		}

		defaultValuesSnippet.deleteCharAt(defaultValuesSnippet.lastIndexOf(","));
		if(setDefaults)
			defaultsQuery=String.format("UPDATE %s set %s where id IN (SELECT id from %s)",
					resultTable.getName(),
					defaultValuesSnippet,
					diffTable.getName());
	}

}
