package org.gcube.data.analysis.tabulardata.operation.data.transformation;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map.Entry;

import org.apache.commons.dbutils.DbUtils;
import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableCreator;
import org.gcube.data.analysis.tabulardata.expression.composite.aggregation.Sum;
import org.gcube.data.analysis.tabulardata.expression.composite.comparable.Equals;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.expression.functions.Cast;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.factories.MeasureColumnFactory;
import org.gcube.data.analysis.tabulardata.model.column.type.IdColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.ValidationColumnType;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDText;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ImmutableLocalizedText;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.OperationHelper;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.gcube.data.analysis.tabulardata.operation.worker.results.ImmutableWorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.results.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.types.DataWorker;

public class DenormalizationWorker extends DataWorker{

	private CubeManager cubeManager;
	private SQLExpressionEvaluatorFactory evaluatorFactory;
	private DatabaseConnectionProvider connProvider;


	public DenormalizationWorker(OperationInvocation sourceInvocation,
			CubeManager cubeManager,
			SQLExpressionEvaluatorFactory evaluatorFactory,DatabaseConnectionProvider connProvider) {
		super(sourceInvocation);
		this.cubeManager = cubeManager;
		this.evaluatorFactory = evaluatorFactory;
		this.connProvider=connProvider;
	}

	private Table targetTable;
	private Table newTable;
	private Column valueColumn;
	private Column attributeColumn;

	private ArrayList<String> values=new ArrayList<>();

	private HashMap<String,Column> newColumnsByValue=new HashMap<>();


	@Override
	protected WorkerResult execute() throws WorkerException {
		init();
		updateProgress(0.1f, String.format("Getting values in ",OperationHelper.retrieveColumnLabel(attributeColumn)));
		getAttributeValues();
		updateProgress(0.2f, "Forming new table structure");
		createTable();
		updateProgress(0.3f,"Transforming table...");
		executeScripts();
		updateProgress(0.9f,"Finalizing");
		return new ImmutableWorkerResult(newTable);
	}

	private void init(){
		targetTable=cubeManager.getTable(getSourceInvocation().getTargetTableId());
		valueColumn=targetTable.getColumnById(OperationHelper.getParameter(DenormalizationFactory.VALUE_COLUMN, getSourceInvocation()).getColumnId());
		attributeColumn=targetTable.getColumnById(OperationHelper.getParameter(DenormalizationFactory.ATTRIBUTE_COLUMN, getSourceInvocation()).getColumnId());
	}


	private void getAttributeValues()throws WorkerException{
		Connection conn=null;		
		Statement stmt=null;
		ResultSet rs=null;
		try{
			conn=connProvider.getConnection();
			stmt=conn.createStatement();
			rs=stmt.executeQuery(String.format("SELECT DISTINCT(%s) FROM %s",attributeColumn.getName(),targetTable.getName()));
			while(rs.next()){
				values.add(rs.getObject(1).toString());
			}
		}catch(Exception e){
			throw new WorkerException("Unable to anlyze attribute column", e);
		}finally{
			DbUtils.closeQuietly(rs);
			DbUtils.closeQuietly(stmt);
			DbUtils.closeQuietly(conn);
		}
	}


	private void createTable(){		
		MeasureColumnFactory factory=new MeasureColumnFactory();
		for(String value:values){
			Column toAdd=factory.create(new ImmutableLocalizedText(value), valueColumn.getDataType());
			newColumnsByValue.put(value, toAdd);			
		}
		TableCreator creator=cubeManager.createTable(targetTable.getTableType()).like(targetTable, false).removeColumn(valueColumn).removeColumn(attributeColumn);
		for(Column toAdd:newColumnsByValue.values()){
			creator.addColumn(toAdd);
		}
		newTable=creator.create();
	}

	private void executeScripts() throws WorkerException{
		ArrayList<String> scripts=getScripts();
		try{
			SQLHelper.executeSQLBatchCommands(connProvider, scripts.toArray(new String[scripts.size()]));
		}catch(Exception e){
			throw new WorkerException("Unable to execute transformation scripts",e);
		}
	}




	private ArrayList<String> getScripts(){

		String aggregationExpression=evaluatorFactory.getEvaluator(new Sum(targetTable.getColumnReference(valueColumn))).evaluate();

		StringBuilder existingCondition= new StringBuilder();

		StringBuilder rValues=new StringBuilder();

		ArrayList<Column> toGroupByColumns=new ArrayList<>();
		ArrayList<Column> groupedColumns=new ArrayList<>();		

		//Iterate over other columns then attribute and value to get snippets
		for(Column col:targetTable.getColumnsExceptTypes(IdColumnType.class,ValidationColumnType.class)){
			if(!col.getLocalId().equals(attributeColumn.getLocalId())&&!col.getLocalId().equals(valueColumn.getLocalId())) {								
				Column newColumn = newTable.getColumnById(col.getLocalId());


				toGroupByColumns.add(col); // source col snippet
				groupedColumns.add(newColumn); // new cols snippet

				existingCondition.append(newColumn.getName()+"="+"r."+newColumn.getName()+" AND ");
				rValues.append("r."+newColumn.getName()+",");

			}
		}


		existingCondition.replace(existingCondition.lastIndexOf("AND"), existingCondition.length()-1, "");
		rValues.deleteCharAt(rValues.lastIndexOf(","));
		String sourceColSnippet=OperationHelper.getColumnNamesSnippet(toGroupByColumns);
		String targetColSnippet=OperationHelper.getColumnNamesSnippet(groupedColumns);

		ArrayList<String> scripts=new ArrayList<>();
		for(Entry<String,Column> entry:newColumnsByValue.entrySet()){
			String attCondition = evaluatorFactory.getEvaluator(new Equals(targetTable.getColumnReference(attributeColumn), new Cast(new TDText(entry.getKey()), attributeColumn.getDataType()))).evaluate();
			String columnName=entry.getValue().getName();

			scripts.add("DO $$DECLARE r record;" +
					"BEGIN" +
					"	FOR r in Select "+sourceColSnippet+", "+aggregationExpression+" as agg from "+targetTable.getName()+" WHERE "+attCondition+" GROUP BY " +sourceColSnippet+
					"	LOOP " +
					" 		IF EXISTS (SELECT * FROM "+newTable.getName()+" WHERE "+existingCondition+")"+
					" 		THEN UPDATE "+newTable.getName()+" SET "+columnName+" = r.agg WHERE "+existingCondition+";" +					
					" 		ELSE INSERT INTO "+newTable.getName()+" ("+targetColSnippet+","+columnName+") VALUES ("+rValues+", r.agg);" +
					" 		END IF;"+
					"	END LOOP;" +
					"END$$");				
		}

		return scripts;
	}


}
