package org.gcube.data.analysis.tabulardata.operation.validation;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import org.apache.commons.dbutils.DbUtils;
import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableCreator;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableMetaCreator;
import org.gcube.data.analysis.tabulardata.expression.Expression;
import org.gcube.data.analysis.tabulardata.expression.evaluator.EvaluatorException;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.CubeManagerEvaluatorUtil;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.type.ValidationColumnType;
import org.gcube.data.analysis.tabulardata.model.datatype.BooleanType;
import org.gcube.data.analysis.tabulardata.model.metadata.column.DataValidationMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.worker.BaseWorker;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.OperationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ValidateDataWithExpression extends BaseWorker {
	
	private static final Logger log = LoggerFactory.getLogger(ValidateDataWithExpression.class);
	
	private CubeManager cubeManager;
	
	private DatabaseConnectionProvider connectionProvider;
	
	private Table targetTable;
	
	private Expression validationExpression; 
	
	private Table newTable;
	
	private Column validationColumn;

	private Table tempTable;

	private SQLExpressionEvaluatorFactory evaluatorFactory;
	
	public ValidateDataWithExpression(OperationInvocation invocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider) {
		super(invocation);
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		targetTable = cubeManager.getTable(invocation.getTargetTableId());
		validationExpression = (Expression) invocation.getParameterInstances().get(ValidateDataWithExpressionFactory.EXPRESSION_ID);
		evaluatorFactory = new SQLExpressionEvaluatorFactory(new CubeManagerEvaluatorUtil(cubeManager));
	}

	@Override
	public void run() {
		String errorMsg = "Unable to complete data validation";
		try {
			inProgress(0.1f);
			createNewTable();
			inProgress(0.4f);
			fillValidationColumn();
			inProgress(0.7f);
			evaluateValidityAndUpdateTableMeta();
			succeed(newTable);
		} catch (OperationException e) {
			log.error(e.getMessage());
			fail(e);
		} catch (Exception e) {
			log.error(errorMsg,e);
			new OperationException(errorMsg, e);
		} finally {
//			if (tempTable!=null && tempTable.getId()!=null) cubeManager.removeTable(tempTable.getId());
		}
	}

	private void evaluateValidityAndUpdateTableMeta() throws OperationException{
		boolean validationResult = getValidationResult();
		
		TableMetaCreator tmc = cubeManager.modifyTableMeta(tempTable.getId());
		tmc.setColumnMetadata(validationColumn.getLocalId(), new DataValidationMetadata(validationExpression, "", validationResult));
		
	}
	
	private boolean getValidationResult() throws OperationException{
		Connection conn = null;
		Statement stmt = null;
		ResultSet rs = null;
		try {
			conn = connectionProvider.getConnection();
			stmt = conn.createStatement();
			rs = stmt.executeQuery(generateGetValidationResultSQLQuery());
			
			while(rs.next()){
				return rs.getBoolean(0);
			}
			throw new OperationException("Unable to obtain validation result");
		} catch (SQLException e) {
			String msg = "Unable to query table";
			log.error(msg,e);
			throw new OperationException(msg, e);
		} finally {
			DbUtils.closeQuietly(conn);
			DbUtils.closeQuietly(stmt);
			DbUtils.closeQuietly(rs);
		}
	}

	private String generateGetValidationResultSQLQuery() {
		return String.format("SELECT bool_and(%s) from %s", validationColumn.getName(), tempTable.getName());
	}

	private void fillValidationColumn() throws OperationException{
		executeSQLBatchCommands(connectionProvider, generateSetAllFalseSqlCommand(), generateValidationSqlCommand());
	}

	private String generateSetAllFalseSqlCommand() {
		return String.format("UPDATE %s SET %s = false;",tempTable.getName(), validationColumn.getName());
	}

	private String generateValidationSqlCommand() throws OperationException{
		try {
			return String.format("UPDATE %s SET %s = true WHERE %s;", tempTable.getName(), validationColumn.getName(), evaluatorFactory.getEvaluator(validationExpression).evaluate());
		} catch (EvaluatorException e) {
			throw new OperationException("Unable to evaluate epression", e);
		}
	}

	private void createNewTable() {
		TableCreator tc = cubeManager.createTable(targetTable.getTableType());
		tc.like(targetTable, true);
		validationColumn = new Column(new BooleanType(), new ValidationColumnType());
		tc.addColumn(validationColumn);
		tempTable = tc.create();
		log.debug("Created temporary table:\n"+tempTable);
	}

}
