package org.gcube.data.analysis.tabulardata.operation.validation;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.Expression;
import org.gcube.data.analysis.tabulardata.expression.MalformedExpressionException;
import org.gcube.data.analysis.tabulardata.expression.TableReferenceReplacer;
import org.gcube.data.analysis.tabulardata.expression.evaluator.EvaluatorException;
import org.gcube.data.analysis.tabulardata.expression.evaluator.description.DescriptionExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.column.factories.ValidationColumnFactory;
import org.gcube.data.analysis.tabulardata.model.metadata.column.DataValidationMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.column.ValidationReferencesMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ImmutableLocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.table.GlobalDataValidationReportMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableId;
import org.gcube.data.analysis.tabulardata.operation.OperationHelper;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.ValidationHelper;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.ImmutableWorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ValidateDataWithExpression extends Worker {

	private static final Logger log = LoggerFactory.getLogger(ValidateDataWithExpression.class);

	private CubeManager cubeManager;

	private DatabaseConnectionProvider connectionProvider;

	private Table targetTable;

	private Expression validationExpression;

	private String expressionDescription = null;

	private String validationTitle= null;
	
	private Column validationColumn;

	private SQLExpressionEvaluatorFactory sqlEvaluatorFactory;

	private DescriptionExpressionEvaluatorFactory descriptionEvaluatorFactory;


	private List<Column> evaluatedColumns=new ArrayList<>();

	public ValidateDataWithExpression(OperationInvocation invocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider, SQLExpressionEvaluatorFactory sqlEvaluatorFactory, DescriptionExpressionEvaluatorFactory descriptionEvaluatorFactory) {
		super(invocation);
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		
		this.sqlEvaluatorFactory = sqlEvaluatorFactory;
		this.descriptionEvaluatorFactory = descriptionEvaluatorFactory;
	}

	
	
	
	
	@Override
	protected WorkerResult execute() throws WorkerException {
		updateProgress(0.1f);
		initiate();
		addValidationColumn();
		updateProgress(0.4f);
		fillValidationColumn();
		updateProgress(0.7f);
		evaluateValidityAndUpdateTableMeta();
		return new ImmutableWorkerResult(targetTable);
	}

	
	private void initiate() throws WorkerException{
		try{
		OperationInvocation invocation=getSourceInvocation();
		targetTable = cubeManager.getTable(invocation.getTargetTableId());
		validationExpression = (Expression) invocation.getParameterInstances().get(
				ValidateDataWithExpressionFactory.EXPRESSION_PARAMETER.getIdentifier());
	
		// Getting info from columns in expression
		TableReferenceReplacer replacer=new TableReferenceReplacer(validationExpression);
		for(TableId tableId:replacer.getTableIds()){			
			if(tableId.equals(targetTable.getId())){
				//internal
				for(ColumnReference ref:replacer.getReferences(tableId)){
					Column col=targetTable.getColumnById(ref.getColumnId());
					evaluatedColumns.add(col);			
				}
			}
		}
		
		
		if (invocation.getParameterInstances().containsKey(ValidateDataWithExpressionFactory.DESCRIPTION_PARAMETER))
			expressionDescription = (String) invocation.getParameterInstances().get(
					ValidateDataWithExpressionFactory.DESCRIPTION_PARAMETER.getIdentifier());
		else expressionDescription= descriptionEvaluatorFactory.getEvaluator(validationExpression).evaluate();
		
		if (invocation.getParameterInstances().containsKey(ValidateDataWithExpressionFactory.VALIDATION_TITLE_PARAMETER))
		validationTitle = (String) invocation.getParameterInstances().get(
				ValidateDataWithExpressionFactory.VALIDATION_TITLE_PARAMETER.getIdentifier());
		else validationTitle= "Expression on "+OperationHelper.getColumnLabelsSnippet(evaluatedColumns);
		
		}catch(Exception e){
			throw new WorkerException("Unable to initiate, wrong parameters configuration",e);
		}
	}
	
	
	private void evaluateValidityAndUpdateTableMeta() throws WorkerException {
		
		try{
			int invalidCount=ValidationHelper.getErrorCount(connectionProvider, targetTable, validationColumn, sqlEvaluatorFactory);
			GlobalDataValidationReportMetadata globalMeta=ValidationHelper.createDataValidationReport(validationColumn);
			DataValidationMetadata validationMeta = new DataValidationMetadata(new ImmutableLocalizedText(expressionDescription),invalidCount);
			targetTable = cubeManager.modifyTableMeta(targetTable.getId())
					.setColumnMetadata(validationColumn.getLocalId(), 
							validationMeta).
							setTableMetadata(globalMeta).create();
			}catch(Exception e){
				throw new WorkerException("Unable to evaluate global validation",e);
			}
		
	}


	private void fillValidationColumn() throws WorkerException {
		try {
			SQLHelper.executeSQLBatchCommands(connectionProvider, generateSetAllFalseSqlCommand(),
					generateValidationSqlCommand());
		} catch (SQLException e) {
			e.getNextException().printStackTrace();
			throw new WorkerException("Error occurred while executing SQL command", e);
		}
	}

	private String generateSetAllFalseSqlCommand() {
		return String.format("UPDATE %s SET %s = false;", targetTable.getName(), validationColumn.getName());
	}

	private String generateValidationSqlCommand() throws WorkerException {
		try {
			//Change table references in given expression
			//			log.debug("Passed expression is "+sqlEvaluatorFactory.getEvaluator(validationExpression).evaluate());
			//			Expression toUseExpression =new TableReferenceReplacer(validationExpression).replaceTableId(targetTable.getId(), tempTable.getId()).getExpression();
			validationExpression.validate();
			return String.format("UPDATE %s SET %s = true WHERE %s;", targetTable.getName(), validationColumn.getName(),
					sqlEvaluatorFactory.getEvaluator(validationExpression).evaluate());
		} catch (EvaluatorException e) {
			throw new WorkerException("Unable to evaluate epression", e);
		} catch (MalformedExpressionException e) {
			throw new WorkerException("Unable to evaluate epression", e);
		}
	}

	private void addValidationColumn() throws WorkerException {	
		
		DataValidationMetadata validationMeta = new DataValidationMetadata(new ImmutableLocalizedText(expressionDescription),0);
		validationColumn = new ValidationColumnFactory().create(new ImmutableLocalizedText(validationTitle),validationMeta);
		targetTable=cubeManager.addValidations(targetTable.getId(),validationColumn);	
		log.debug("Added validation column:\n" + targetTable);
		
		ValidationReferencesMetadata meta=new ValidationReferencesMetadata(evaluatedColumns.toArray(new Column[evaluatedColumns.size()]));
		
		targetTable=cubeManager.modifyTableMeta(targetTable.getId()).setColumnMetadata(validationColumn.getLocalId(), meta).create();
	}

}
