package org.gcube.data.analysis.tabulardata.operation.validation;

import java.util.ArrayList;
import java.util.List;

import javax.inject.Inject;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.datatype.DataType;
import org.gcube.data.analysis.tabulardata.operation.OperationHelper;
import org.gcube.data.analysis.tabulardata.operation.OperationId;
import org.gcube.data.analysis.tabulardata.operation.factories.types.ColumnValidatorFactory;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.parameters.Cardinality;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.DataTypeParameter;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;
import org.gcube.data.analysis.tabulardata.operation.worker.types.ValidationWorker;

public class ColumnTypeCastValidatorFactory extends ColumnValidatorFactory {
	
	private final static List<Parameter> parameters = new ArrayList<Parameter>();
	
	private static final OperationId OPERATION_ID = new OperationId(5002);
	
	public static final DataTypeParameter TARGET_TYPE_PARAMETER = new DataTypeParameter("targetDataType", "Target type", "The data type on which the cast should be checked", Cardinality.ONE);
	
	static{
		parameters.add(TARGET_TYPE_PARAMETER);
	}

	CubeManager cubeManager;
	
	DatabaseConnectionProvider connectionProvider;

	SQLExpressionEvaluatorFactory evalutorFactory;
	
	@Inject
	public ColumnTypeCastValidatorFactory(CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider,
			SQLExpressionEvaluatorFactory evalutorFactory) {
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		this.evalutorFactory=evalutorFactory;
	}

	@Override
	public ValidationWorker createWorker(OperationInvocation invocation) throws InvalidInvocationException {
		performBaseChecks(invocation,cubeManager);
		return new ColumnTypeCastValidator(invocation, cubeManager, connectionProvider,evalutorFactory);
		
	}

	@Override
	protected String getOperationName() {
		return "Column type cast check";
	}

	@Override
	protected String getOperationDescription() {
		return "Check if a column type can be casted to another one";
	}

	@Override
	protected List<Parameter> getParameters() {
		return parameters;
	}

	@Override
	protected OperationId getOperationId() {
		return OPERATION_ID;
	}
	
	@Override
	public String describeInvocation(OperationInvocation invocation)
			throws InvalidInvocationException {
		performBaseChecks(invocation,cubeManager);
		Column targetCol=cubeManager.getTable(invocation.getTargetTableId()).getColumnById(invocation.getTargetColumnId());
		DataType dataType=OperationHelper.getParameter(TARGET_TYPE_PARAMETER, invocation);
		return String.format("Check if %s can be converted to %s",OperationHelper.retrieveColumnLabel(targetCol),dataType.getName());
	}
}
