package org.gcube.data.analysis.tabulardata.operation.sdmx.datastructuredefinition;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.metadata.NoSuchMetadataException;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnLocalId;
import org.gcube.data.analysis.tabulardata.model.column.type.AttributeColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.DimensionColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.MeasureColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.TimeDimensionColumnType;
import org.gcube.data.analysis.tabulardata.model.metadata.common.LocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.common.NamesMetadata;
import org.gcube.data.analysis.tabulardata.model.relationship.ColumnRelationship;
import org.gcube.data.analysis.tabulardata.model.resources.ResourceType;
import org.gcube.data.analysis.tabulardata.model.resources.SDMXResource;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.sdmx.WorkerUtils;
import org.gcube.data.analysis.tabulardata.operation.sdmx.codelist.SDMXCodelistGenerator;
import org.gcube.data.analysis.tabulardata.operation.sdmx.conceptscheme.SDMXConceptSchemeGenerator;
import org.gcube.data.analysis.tabulardata.operation.sdmx.configuration.ConfigurationManager;
import org.gcube.data.analysis.tabulardata.operation.sdmx.datastructuredefinition.beans.DataStructureBean;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.gcube.data.analysis.tabulardata.operation.worker.results.ResourcesResult;
import org.gcube.data.analysis.tabulardata.operation.worker.results.resources.ImmutableSDMXResource;
import org.gcube.data.analysis.tabulardata.operation.worker.types.ResourceCreatorWorker;
import org.gcube.datapublishing.sdmx.RegistryInformationProvider;
import org.gcube.datapublishing.sdmx.api.registry.SDMXRegistryClient;
import org.gcube.datapublishing.sdmx.impl.exceptions.SDMXRegistryClientException;
import org.gcube.datapublishing.sdmx.impl.exceptions.SDMXVersionException;
import org.gcube.datapublishing.sdmx.security.model.impl.BasicCredentials;
import org.sdmxsource.sdmx.api.constants.ATTRIBUTE_ATTACHMENT_LEVEL;
import org.sdmxsource.sdmx.api.constants.SDMX_STRUCTURE_TYPE;
import org.sdmxsource.sdmx.api.model.beans.codelist.CodelistBean;
import org.sdmxsource.sdmx.api.model.beans.datastructure.DimensionBean;
import org.sdmxsource.sdmx.api.model.beans.datastructure.PrimaryMeasureBean;
import org.sdmxsource.sdmx.api.model.beans.reference.StructureReferenceBean;
import org.sdmxsource.sdmx.api.model.mutable.base.RepresentationMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.base.TextTypeWrapperMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.codelist.CodelistMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.conceptscheme.ConceptMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.conceptscheme.ConceptSchemeMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.datastructure.AttributeMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.datastructure.DataStructureMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.datastructure.DataflowMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.datastructure.DimensionMutableBean;
import org.sdmxsource.sdmx.api.model.mutable.datastructure.PrimaryMeasureMutableBean;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.base.RepresentationMutableBeanImpl;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.base.TextTypeWrapperMutableBeanImpl;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.datastructure.AttributeMutableBeanImpl;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.datastructure.DimensionMutableBeanImpl;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.datastructure.PrimaryMeasureMutableBeanImpl;
import org.sdmxsource.sdmx.sdmxbeans.model.mutable.metadatastructure.DataflowMutableBeanImpl;
import org.sdmxsource.sdmx.util.beans.reference.StructureReferenceBeanImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

public abstract class SDMXDataStructureDefinitionAbstractExporter extends ResourceCreatorWorker {

	
	
	private Logger log;
	private Table table;
	private OperationInvocation invocation;
	private CubeManager cubeManager;
//	private ;
	private final String ATTRIBUTE_ASSIGNMENT_STATUS_MANDATORY = "Mandatory";
	private static String errorMessage = "Unable to complete export procedure";;
	private DatabaseConnectionProvider connectionProvider;
	private SDMXConceptSchemeGenerator conceptSchemeGenerator;

	
	private final String 	CODELIST_VERSION_ACTION_PROPERTY = "codelist.version.action",
							//CODELIST_VERSION_ACTION_IGNORE = "ignore",
							CODELIST_VERSION_ACTION_BLOCK = "block",
							CODELIST_VERSION_ACTION_WARNING = "warning";
	
	private class Parameters
	{
		String 		registryUrl,
					targetAgency, 
					targetId,
					targetVersion;
		Column primaryMeasure;
		BasicCredentials credentials;
	}
	
	
	public SDMXDataStructureDefinitionAbstractExporter(Table table, DatabaseConnectionProvider connectionProvider, OperationInvocation invocation,CubeManager cubeManager) {
		super(invocation);
		this.log = LoggerFactory.getLogger(this.getClass());
		this.table = table;
		this.invocation = invocation;
		this.connectionProvider = connectionProvider;
		this.cubeManager = cubeManager;

	}
	
	/**
	 * 
	 */
	@Override
	protected ResourcesResult execute() throws WorkerException 
	{		

			this.log.debug("Init parameters and functionalities");
			Parameters parameters = getParameters(this.invocation);
			this.conceptSchemeGenerator = new SDMXConceptSchemeGenerator(this.table, parameters.targetId,parameters.targetAgency, parameters.targetVersion);
			List<LocalizedText> tableNamesMetadata =loadMetadata ();
			this.log.debug("Parameters and functionalities initialized");
			try 
			{
				updateProgress(0.1f,"Creating beans");
				DataStructureBean dataStructure = createDataStructureBean(this.table, parameters.targetAgency, parameters.targetVersion, parameters.targetId, tableNamesMetadata);
				ConceptSchemeMutableBean conceptScheme = conceptSchemeGenerator.createConceptSchemeBean();
				dataStructure.setConcepts(conceptScheme);
				DataflowMutableBean dataFlow = createDataFlowBean(dataStructure.getDsd(),tableNamesMetadata,parameters );
				dataStructure.setDataFlow(dataFlow);			
				updateProgress(0.2f,"Populating data structure");
				populateDataStructure(dataStructure,parameters);
				updateProgress(0.6f,"Publishing");
				String warning = publishData(dataStructure,parameters);
				
				// Warning message
				this.log.warn(warning);
				
				extraOperation(dataStructure);
				updateProgress(0.9f,"Finalizing");
				SDMXResource sdmxResource = new SDMXResource(new URL(WorkerUtils.getResourceURI(parameters.registryUrl)), dataStructure.getDsd().getId(), parameters.targetVersion, parameters.targetAgency,  SDMXResource.TYPE.DATA_STRUCTURE);
				return new ResourcesResult(new ImmutableSDMXResource(sdmxResource, "Data Structure SDMX export" , 
						sdmxResource.toString(), ResourceType.SDMX));
	
			} catch (RuntimeException e) {
				log.error(errorMessage, e);
				throw new WorkerException(errorMessage, e);
			} catch (MalformedURLException e) {
				throw new WorkerException(String.format("exported url %s not valid",parameters.registryUrl),e);
			}
	}

	private int checkCodelistVersionAction ()
	{
		//0 = ignore 1 = warning 2 = block
		String actionParameter = ConfigurationManager.getInstance().getValue(CODELIST_VERSION_ACTION_PROPERTY);
		this.log.debug("Codelist action parameter "+actionParameter);
		
		if (actionParameter == null || actionParameter.equals(CODELIST_VERSION_ACTION_BLOCK)) return 2;
		else if (actionParameter.equals(CODELIST_VERSION_ACTION_WARNING)) return 1;
		else return 0;
		
	}
	
	protected abstract void extraOperation (DataStructureBean dataStructure);

	/**
	 * 
	 * @param invocation
	 */
	private Parameters getParameters (OperationInvocation invocation) 
	{
		Parameters parameters = new Parameters();
		parameters.registryUrl = (String) invocation.getParameterInstances().get(DataStructureDefinitionWorkerUtils.REGISTRY_BASE_URL);
		parameters.targetAgency = (String) invocation.getParameterInstances().get(DataStructureDefinitionWorkerUtils.AGENCY);
		parameters.targetId = (String) invocation.getParameterInstances().get(DataStructureDefinitionWorkerUtils.ID);
		parameters.targetVersion = (String) invocation.getParameterInstances().get(DataStructureDefinitionWorkerUtils.VERSION);
		parameters.credentials = RegistryInformationProvider.retrieveCredentials(parameters.registryUrl);
		String observationValue = (String) invocation.getParameterInstances().get(DataStructureDefinitionWorkerUtils.OBS_VALUE_COLUMN);
		parameters.primaryMeasure = this.table.getColumnById(new ColumnLocalId(observationValue));
		return parameters;

	}


	/**
	 * 
	 * @param dsd
	 * @param concepts
	 * @param dataFlow
	 * @throws WorkerException
	 */
	
	private String publishData(DataStructureBean dataStructure, Parameters parameters) throws WorkerException 
	{
		String url = (String) invocation.getParameterInstances().get(DataStructureDefinitionWorkerUtils.REGISTRY_BASE_URL);
		SDMXRegistryClient registryClient = DataStructureDefinitionWorkerUtils.initSDMXClient(url,parameters.credentials.getUsername(), parameters.credentials.getPassword());
		String currentType = null;
		String warning = "";
		
		try {
			
			this.log.debug("Publishing associated codelists...");
			currentType = "codelists";
			
			for (CodelistBean codelist : dataStructure.getAllCodelists())
			{
				try
				{
					this.log.debug("Publishing codelist "+codelist.getId());
					registryClient.publish(codelist);
					this.log.debug("Codelist published");
				} catch (SDMXVersionException e)
				{
					switch (checkCodelistVersionAction())
					{
					case 2:
						throw e;
					case 1:
						warning = warning + e.getMessage()+" ";
					default:
							this.log.warn(e.getMessage(),e);
					}
				}

			}
		
			this.log.debug("Codelists published");
			currentType = "concepts";
			this.log.debug("Publishing concepts...");
			registryClient.publish(dataStructure.getConcepts().getImmutableInstance());
			this.log.debug("Concepts published");
			currentType = "data structure definitions";
			this.log.debug("Publishing dsd...");
			registryClient.publish(dataStructure.getDsd().getImmutableInstance());
			this.log.debug("DSD published");
			currentType = "data flow";
			this.log.debug("Publishing data flow...");
			registryClient.publish(dataStructure.getDataFlow().getImmutableInstance());
			this.log.debug("Data flow published");
			this.log.debug("Warning message "+warning);
			return warning;
		} 		
		catch (SDMXVersionException e) {
			log.error("Error in the pubblication",e);
			throw new WorkerException(e.getMessage(), e);
		}
		 
		catch (SDMXRegistryClientException e) {
			log.error("Error in the SDMX client",e);
			throw new WorkerException("Unable to publish dsd on registry: error in the"+currentType, e);
		}
		

	}



	/**
	 * 
	 * @param dataStructure
	 * @param conceptScheme
	 * @param dataSourceConfigurationBean
	 * @param codelists
	 * @throws WorkerException
	 */
	@SuppressWarnings("unchecked")
	private void populateDataStructure (DataStructureBean dataStructure,Parameters parameters ) throws WorkerException 
	{
		List<Column> columnsList = new ArrayList<>();
		columnsList.add(parameters.primaryMeasure);
		this.log.debug("Pupulating data structure");
		List<Column> measureColumns = new ArrayList<>();
		getMeasures(measureColumns, parameters);
		columnsList.addAll(measureColumns);
		this.log.debug("Measure columns loaded");
		addPrimaryMeasure(dataStructure, parameters);
		this.log.debug("Primary measure added");
		addMeasureDimensions(dataStructure, measureColumns, parameters);
		this.log.debug("Measure dimensions added");
		this.log.debug("Adding generic dimension bean");
		List<Column> dimensionColumns = this.table.getColumnsByType(DimensionColumnType.class);
		addGenericDimensions(dataStructure,dimensionColumns,parameters);
		columnsList.addAll(dimensionColumns);
		this.log.debug("Dimension columns added");
		Column timeDimensionColumn = this.table.getColumnsByType(TimeDimensionColumnType.class).get(0);
		addTimeDimension(dataStructure,timeDimensionColumn,parameters);
		columnsList.add(timeDimensionColumn);
		List<Column> attributeColumns = this.table.getColumnsByType(AttributeColumnType.class);
		addAttributes(dataStructure,attributeColumns,parameters);
		columnsList.addAll(attributeColumns);
		registerData(dataStructure, table, columnsList, connectionProvider);

	}
	
	/**
	 * 
	 * @param dataStructure
	 */
	private void addTimeDimension (DataStructureBean dataStructure,Column timeDimensionColumn,Parameters parameters )
	{
		this.log.debug("Adding time dimension");
		ConceptSchemeMutableBean concepts = dataStructure.getConcepts();
		DimensionMutableBean timeDimensionBean = new DimensionMutableBeanImpl();
		timeDimensionBean.setId(DimensionBean.TIME_DIMENSION_FIXED_ID);
		ConceptMutableBean timeDimensionConcept = this.conceptSchemeGenerator.createConceptBean(timeDimensionColumn);
		concepts.addItem(timeDimensionConcept);
		timeDimensionBean.setConceptRef(getConceptReference(concepts, timeDimensionConcept,parameters.targetVersion));
		timeDimensionBean.setTimeDimension(true);
		dataStructure.getDsd().addDimension(timeDimensionBean);
		registerTimeDimensionColumn(timeDimensionColumn, timeDimensionConcept,dataStructure);
	}
	
	
	protected abstract void registerTimeDimensionColumn (Column column, ConceptMutableBean concept,DataStructureBean dataStructure);


	/**
	 * 
	 * @param dimensionListBean
	 * @param dimensionColumns
	 * @param concepts
	 */

	private void addGenericDimensions (DataStructureBean dataStructure,List<Column> dimensionColumns,Parameters parameters )
	{
		ConceptSchemeMutableBean concepts = dataStructure.getConcepts();
		
		for (Column column : dimensionColumns)
		{
			ConceptMutableBean columnConcept = this.conceptSchemeGenerator.createConceptBean(column);
			concepts.addItem(columnConcept);
			CodelistBean immutableCodelist = createCodeListRepresentation(column,parameters);
			
			if (immutableCodelist != null) columnConcept.setCoreRepresentation(getCoreRepresentation (immutableCodelist));

		
			StructureReferenceBean conceptsReference = getConceptReference(concepts, columnConcept,parameters.targetVersion);		
			DimensionMutableBean dimension = new DimensionMutableBeanImpl();
			dimension.setId(column.getName()+"_DSD");
			dimension.setConceptRef(conceptsReference);
			dataStructure.getDsd().addDimension(dimension);
			registerDimensionColumn(column, columnConcept, immutableCodelist,dataStructure);
		}	
	}
	
	
	/**
	 * 
	 * @param dataStructure
	 * @param attributeColumns
	 * @param concepts
	 */

	private void addAttributes (DataStructureBean dataStructure,List<Column> attributeColumns,Parameters parameters )
	{
		log.debug("Adding attribute list bean");
		ConceptSchemeMutableBean concepts = dataStructure.getConcepts();
		
		for (Column column : attributeColumns)
		{
			ConceptMutableBean columnConcept = this.conceptSchemeGenerator.createConceptBean(column);
			concepts.addItem(columnConcept);
			CodelistBean immutableCodelist = createCodeListRepresentation(column,parameters);	
			
			if (immutableCodelist != null) columnConcept.setCoreRepresentation(getCoreRepresentation (immutableCodelist));
			
			StructureReferenceBean conceptsReference = getConceptReference(concepts, columnConcept,parameters.targetVersion);
			AttributeMutableBean attributeBean = new AttributeMutableBeanImpl();
			attributeBean.setAttachmentLevel(ATTRIBUTE_ATTACHMENT_LEVEL.OBSERVATION);
			attributeBean.setAssignmentStatus(ATTRIBUTE_ASSIGNMENT_STATUS_MANDATORY);
			attributeBean.setConceptRef(conceptsReference);
			dataStructure.getDsd().addAttribute(attributeBean);
			registerAttributeColumn(column, columnConcept, immutableCodelist,dataStructure);

		}	
	}
	
	
	protected abstract void registerDimensionColumn (Column column, ConceptMutableBean concept,CodelistBean immutableCodelist, DataStructureBean dataStructure);


	
	protected abstract void registerAttributeColumn (Column column, ConceptMutableBean concept,CodelistBean immutableCodelist, DataStructureBean dataStructure);
	
	
	private RepresentationMutableBean getCoreRepresentation (CodelistBean immutableCodelist)
	{
		RepresentationMutableBean coreRepresentation = new RepresentationMutableBeanImpl();
		coreRepresentation.setRepresentation(new StructureReferenceBeanImpl(immutableCodelist));
		log.debug("Found a code list representation");
		return coreRepresentation;
	}
	
	@SuppressWarnings("unchecked")
	private void getMeasures (List<Column> measures, Parameters parameters)
	{
		this.log.debug("Loading measures");
		measures.addAll(this.table.getColumnsByType(MeasureColumnType.class));
		measures.remove(parameters.primaryMeasure);
		
	}
	
	private void addPrimaryMeasure (DataStructureBean dataStructure, Parameters parameters)
	{
		ConceptMutableBean primaryMeasureConcept = this.conceptSchemeGenerator.createConceptBean(parameters.primaryMeasure);
		dataStructure.getConcepts().addItem(primaryMeasureConcept);
		StructureReferenceBean conceptReferenceBean = new StructureReferenceBeanImpl (primaryMeasureConcept.getParentAgency(),dataStructure.getConcepts().getId(),
				parameters.targetVersion,SDMX_STRUCTURE_TYPE.CONCEPT,primaryMeasureConcept.getId());
		
		PrimaryMeasureMutableBean primaryMeasureBean = new PrimaryMeasureMutableBeanImpl();
		primaryMeasureBean.setConceptRef(conceptReferenceBean);
		primaryMeasureBean.setId(PrimaryMeasureBean.FIXED_ID);
		dataStructure.getDsd().setPrimaryMeasure(primaryMeasureBean);
		registerPrimaryMeasure(parameters.primaryMeasure, primaryMeasureConcept, dataStructure);
		this.log.debug("Primary measure added");

	}
	
	/**
	 * 
	 * @param dimensionListBean
	 * @param measureColumns
	 * @param concepts
	 */

	private void addMeasureDimensions (DataStructureBean dataStructure,List<Column> measures,Parameters parameters )
	{

		log.debug("Adding measure dimension list bean");
		ConceptSchemeMutableBean concepts = dataStructure.getConcepts();
		
		for (Column column : measures)
		{
			ConceptMutableBean columnConcept = this.conceptSchemeGenerator.createConceptBean(column);
			concepts.addItem(columnConcept);
			DimensionMutableBean dimensionBean = new DimensionMutableBeanImpl();
			dimensionBean.setMeasureDimension(true);
			dimensionBean.setConceptRef(getConceptReference(concepts, columnConcept,parameters.targetVersion));
			dataStructure.getDsd().addDimension(dimensionBean);
			registerMeasureColumn(column, columnConcept,dataStructure);
		}	
	}
	
	protected abstract void registerPrimaryMeasure (Column column, ConceptMutableBean concept,DataStructureBean dataStructure);
	
	protected abstract void registerMeasureColumn (Column column, ConceptMutableBean concept,DataStructureBean dataStructure);
	
	
	/**
	 * 
	 * @param conceptScheme
	 * @param concept
	 * @return
	 */
	private StructureReferenceBean getConceptReference (ConceptSchemeMutableBean conceptScheme, ConceptMutableBean concept,String conceptVersion)
	{
		log.debug("Creating reference for concept "+concept.getId());
		return new StructureReferenceBeanImpl (concept.getParentAgency(),conceptScheme.getId(),
				conceptVersion,SDMX_STRUCTURE_TYPE.CONCEPT,concept.getId());
	}

	/**
	 * 
	 * @param column
	 * @param codelists
	 * @return
	 */
	private CodelistBean createCodeListRepresentation (Column column,Parameters parameters )
	{
	
		log.debug("Creating codelist bean");
		CodelistBean response = null;
		Table codelist = getAssociatedCodelist(column);
		
		if (codelist != null)
		{
			log.debug("Table found "+codelist.getName());
			SDMXCodelistGenerator codeListGenerator = new SDMXCodelistGenerator(codelist, this.connectionProvider, parameters.targetAgency, codelist.getName()+"_CL", parameters.targetVersion);
			CodelistMutableBean codeListBean =codeListGenerator.createBaseCodelistBean();

			try
			{
				codeListGenerator.populateCodelistWithCodes(codeListBean);
				response = codeListBean.getImmutableInstance();
				
				
			} catch (Exception e)
			{
				log.warn("Codelist not loaded",e);
				
			}
		}
		
		return response;
	}
	

	
	/**
	 * 
	 * @return
	 */
	protected abstract DataStructureBean createDataStructureBean(Table table,String targetAgency, String targetVersion, String targetId,List<LocalizedText> tableNamesMetadata);
	
	/**
	 * 
	 * @param dataStructure
	 * @return
	 */
	private DataflowMutableBean createDataFlowBean (DataStructureMutableBean dataStructure,List<LocalizedText> tableNamesMetadata,Parameters parameters )
	{
		DataflowMutableBean dataFlow = new DataflowMutableBeanImpl();
		log.debug("Populating data flow bean");
		dataFlow.setAgencyId(parameters.targetAgency);
		dataFlow.setDataStructureRef(new StructureReferenceBeanImpl (dataStructure.getAgencyId(),dataStructure.getId(), parameters.targetVersion,SDMX_STRUCTURE_TYPE.DSD));
		dataFlow.setId(parameters.targetId+"_dataFlow");
		dataFlow.setVersion(parameters.targetVersion);
		dataFlow.setNames(getNamesMetadata(tableNamesMetadata,parameters.targetId+" Data Flow", "en"));
		return dataFlow;
	}



	/**
	 * 
	 */
	private List<LocalizedText>  loadMetadata ()
	{
		List<LocalizedText> tableNamesMetadata = null;
		
		try
		{
			tableNamesMetadata = this.table.getMetadata(NamesMetadata.class).getTexts();
		} catch (NoSuchMetadataException e)
		{
			tableNamesMetadata = Lists.newArrayList();
		}

		return tableNamesMetadata;
	}



	/**
	 * 
	 * @param metadataValues
	 * @param defaultValue
	 * @param defaultLocale
	 * @return
	 */
	protected List<TextTypeWrapperMutableBean> getNamesMetadata (final List<LocalizedText> metadataValues,String defaultValue, String defaultLocale)
	{
		List<TextTypeWrapperMutableBean> response = Lists.newArrayList();
		
		if (metadataValues.size() == 0 && defaultValue != null) 
		{
			log.warn("Names Metadata: using default value "+defaultValue);
			response.add(new TextTypeWrapperMutableBeanImpl(defaultLocale, defaultValue));
		}
		else
		{
			for (LocalizedText text : metadataValues) 
			{
				log.debug("Adding metadata value "+text.getValue()+" "+text.getLocale());
				response.add(new TextTypeWrapperMutableBeanImpl(text.getLocale(), text.getValue()));
			}
		}
		
		return response;

	}
	
	private Table getAssociatedCodelist (Column column)
	{

		log.debug("Looking for table associated to column "+column.getLocalId());
    	ColumnRelationship cr = column.getRelationship();
    	log.debug("Relationship " +cr);
    	Table response = null;
    	
    	if (cr != null)
    	{
    		log.debug("Loading referenced table...");
			Table relatedTable = this.cubeManager.getTable(cr.getTargetTableId());
			
			if (relatedTable.getTableType().getCode().equals("CODELIST"))
			{
				log.debug("Table found "+relatedTable.getName());
				response = relatedTable;
			}
			else
			{
				log.debug("Referenced table is not a codelist");
			}
	        
    	}
    	else log.debug("No related table found");
    	
    	return response;
	}
	
	
	
	protected abstract void registerData (DataStructureBean dataStructure,Table table, List<Column> columns,DatabaseConnectionProvider connectionProvider);
	


}