package org.gcube.data.analysis.tabulardata.service;

import java.io.InputStream;
import java.io.ObjectInputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.parameters.Cardinality;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.BooleanParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.MultivaluedStringParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.RegexpStringParameter;
import org.gcube.data.analysis.tabulardata.operation.worker.EligibleOperation;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationDescriptor;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationDescriptor.OperationId;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationDescriptor.OperationScope;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationDescriptor.OperationType;
import org.gcube.data.analysis.tabulardata.service.exception.NoSuchTabularResourceException;
import org.gcube.data.analysis.tabulardata.service.operation.Task;
import org.gcube.data.analysis.tabulardata.service.operation.TaskJobClassifier;
import org.gcube.data.analysis.tabulardata.service.query.QueryInterfaceMock;
import org.gcube.data.analysis.tabulardata.service.tabular.HistoryStep;
import org.gcube.data.analysis.tabulardata.service.tabular.HistoryStepImpl;
import org.gcube.data.analysis.tabulardata.service.tabular.ImmutableTabularResource;
import org.gcube.data.analysis.tabulardata.service.tabular.TabularResource;
import org.gcube.data.analysis.tabulardata.service.tabular.TabularResourceId;
import org.gcube.data.analysis.tabulardata.service.tabular.metadata.AgencyMetadata;
import org.gcube.data.analysis.tabulardata.service.tabular.metadata.CreationDateMetadata;
import org.gcube.data.analysis.tabulardata.service.tabular.metadata.NameMetadata;
import org.gcube.data.analysis.tabulardata.service.tabular.metadata.TabularResourceMetadata;
import org.json.JSONObject;

import com.google.common.collect.Lists;

public class ServiceState {

	public static List<Table> tables = Lists.newArrayList();

	public static JSONObject codelistData;

	public static List<TabularResource> tabularResources = Lists.newArrayList();
	
	public static long lastTabularResourceId = 1;

	public static List<EligibleOperation> voidCapabilities = Lists.newArrayList();

	public static List<EligibleOperation> tableCapabilities = Lists.newArrayList();

	public static List<EligibleOperation> columnCapabilities = Lists.newArrayList();
	
	public static Map<TabularResourceId,List<Task>> operations = new HashMap<TabularResourceId, List<Task>>(); 

	static {
		loadTable();
		loadData();
		createTabularResource();
		createImportCapabilities();
	}

	private static Table loadTable() {
		try {
			InputStream is = QueryInterfaceMock.class.getClassLoader().getResourceAsStream("TestCodelistTable.obj");
			ObjectInputStream ois = new ObjectInputStream(is);
			Table codelistTable = (Table) ois.readObject();
			tables.add(codelistTable);
			ois.close();
			return codelistTable;
		} catch (Exception e) {
			System.err.println("Unable to initialize mock table");
			return null;
		}
	}

	private static void createImportCapabilities() {
		OperationDescriptor csvDescriptor = new OperationDescriptor(new OperationId(100), "CSV Import", "", OperationScope.VOID, OperationType.IMPORT);
		List<Parameter> csvParameters = Lists.newArrayList();
		csvParameters.add(new RegexpStringParameter("documentURL", "Document URL",
				"URL that points to a location where the document can be downloaded.", Cardinality.ONE,"^https?://[-a-zA-Z0-9+&@#/%?=~_|!:,.;]*[-a-zA-Z0-9+&@#/%=~_|]"));
		csvParameters.add(new RegexpStringParameter("separator", "Separator", "Char separator", Cardinality.ONE,"^.$"));
		csvParameters.add(new MultivaluedStringParameter("encoding", "Encoding", "Document Encoding", Cardinality.ONE,Lists.newArrayList("Encoding1","Encoding2","Encoding3")));
		csvParameters.add(new BooleanParameter("hasHeader", "Header", "Tells if the document has header or not", Cardinality.ONE));		
		EligibleOperation csvImport = new EligibleOperation(csvDescriptor, csvParameters);
		
		voidCapabilities.add(csvImport);
		OperationDescriptor sdmxDescriptor = new OperationDescriptor(new OperationId(200), "SDMX Codelist Import", "Import a codelist from a sdmx registry", OperationScope.VOID, OperationType.IMPORT);
		List<Parameter> sdmxParameters = Lists.newArrayList();
		sdmxParameters.add(new RegexpStringParameter("registryBaseUrl", "Registry Base URL", "URL that points to the registry REST base endpoint", Cardinality.ONE, "^https?://[-a-zA-Z0-9+&@#/%?=~_|!:,.;]*[-a-zA-Z0-9+&@#/%=~_|]"));
		sdmxParameters.add(new RegexpStringParameter("agency", "Agency", "SDMX Agency", Cardinality.ONE, "[A-z0-9_-]+"));
		sdmxParameters.add(new RegexpStringParameter("id", "Id", "SDMX Codelist id", Cardinality.ONE, "[A-z0-9_-]+"));
		sdmxParameters.add(new RegexpStringParameter("version", "Version", "SDMX Codelist version", Cardinality.ONE, "[0-9]+\\.[0-9]+"));
		EligibleOperation sdmxImport = new EligibleOperation(sdmxDescriptor, sdmxParameters);
		voidCapabilities.add(sdmxImport);
	}

	private static void createTabularResource() {
		TabularResource mockedTabularResource = new ImmutableTabularResource(createMockedHistory(), createMockedMetadata());
		tabularResources.add(mockedTabularResource);
	}

	private static List<TabularResourceMetadata> createMockedMetadata() {
		List<TabularResourceMetadata> result=new ArrayList<TabularResourceMetadata>();
		result.add(new NameMetadata("My codelist"));
		result.add(new AgencyMetadata("Test user"));
		result.add(new CreationDateMetadata(new Date()));
		return result;
	}

	private static List<HistoryStep> createMockedHistory() {
		List<HistoryStep> result = Lists.newArrayList();
		HistoryStep mockedHistoryStep = new HistoryStepImpl(null,tables.get(0),TaskJobClassifier.PROCESSING);
		result.add(mockedHistoryStep);
		return result;

	}

	private static void loadData() {
		try {
			InputStream is = QueryInterfaceMock.class.getClassLoader().getResourceAsStream("TestCodelistData.json");
			codelistData = new JSONObject(IOUtils.toString(is));
		} catch (Exception e) {
			System.err.println("Unable to load data from file");
		}
	}
	
	public static TabularResource getTabularResourceById(TabularResourceId id) throws NoSuchTabularResourceException{
		for (TabularResource tabularResource : tabularResources ) {
			if (tabularResource.getId().equals(id)) return tabularResource;
		}
		throw new NoSuchTabularResourceException(id);
	}

	public static List<Task> getOperations(TabularResourceId tabularResourceId) {
		List<Task> result = operations.get(tabularResourceId);
		if (result==null) {
			result = Lists.newArrayList();
			operations.put(tabularResourceId, result);
		}
		return result;
	}

}
