package org.gcube.data.analysis.tabulardata.service.operation;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.gcube.data.analysis.tabulardata.model.column.ColumnLocalId;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableId;
import org.gcube.data.analysis.tabulardata.model.table.type.CodelistTableType;
import org.gcube.data.analysis.tabulardata.operation.parameters.Cardinality;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.RegexpStringParameter;
import org.gcube.data.analysis.tabulardata.operation.worker.ActivityStatus;
import org.gcube.data.analysis.tabulardata.operation.worker.EligibleOperation;
import org.gcube.data.analysis.tabulardata.operation.worker.ImmutableJob;
import org.gcube.data.analysis.tabulardata.operation.worker.ImmutableJobResult;
import org.gcube.data.analysis.tabulardata.operation.worker.Job;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationDescriptor;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationDescriptor.OperationId;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationDescriptor.OperationScope;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationDescriptor.OperationType;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.service.ServiceState;
import org.gcube.data.analysis.tabulardata.service.exception.InvalidInvocationException;
import org.gcube.data.analysis.tabulardata.service.exception.NoSuchColumnException;
import org.gcube.data.analysis.tabulardata.service.exception.NoSuchTableException;
import org.gcube.data.analysis.tabulardata.service.exception.NoSuchTabularResourceException;
import org.gcube.data.analysis.tabulardata.service.exception.NoSuchTaskException;
import org.gcube.data.analysis.tabulardata.service.operation.Task.TaskId;
import org.gcube.data.analysis.tabulardata.service.tabular.HistoryStepImpl;
import org.gcube.data.analysis.tabulardata.service.tabular.TabularResource;
import org.gcube.data.analysis.tabulardata.service.tabular.TabularResourceId;

import com.google.common.collect.Lists;

public class OperationInterfaceMock implements OperationInterface {

	private static long operationCounter = 0;

	@Override
	public List<EligibleOperation> getCapabilities() {
		return ServiceState.voidCapabilities;
	}

	static EligibleOperation sdmxCodelistExportCapability;

	private EligibleOperation getSDMXCodelistExportCapability(TableId tableId) {
		OperationDescriptor operationDescriptor = new OperationDescriptor(new OperationId(201), "Sdmx codelist export",
				"SDMX codelist export", OperationScope.TABLE, OperationType.EXPORT);
		List<Parameter> parameters = new ArrayList<Parameter>();
		parameters.add(new RegexpStringParameter("registryBaseUrl", "Registry REST URL",
				"Target SDMX Registry REST Service base URL", Cardinality.ONE,
				"^https?://[-a-zA-Z0-9+&@#/%?=~_|!:,.;]*[-a-zA-Z0-9+&@#/%=~_|]"));
		parameters.add(new RegexpStringParameter("agency", "Agency", "SDMX Agency", Cardinality.ONE, "[A-z0-9_-]+"));
		parameters.add(new RegexpStringParameter("id", "Id", "SDMX Codelist id", Cardinality.ONE, "[A-z0-9_-]+"));
		parameters.add(new RegexpStringParameter("version", "Version", "SDMX Codelist version", Cardinality.ONE,
				"[0-9]+\\.[0-9]+"));
		return new EligibleOperation(operationDescriptor, parameters, tableId);
	}

	@Override
	public List<EligibleOperation> getCapabilities(TabularResourceId tabularResourceId)
			throws NoSuchTabularResourceException {
		List<EligibleOperation> result = Lists.newArrayList();
		TabularResource tr = ServiceState.getTabularResourceById(tabularResourceId);
		Table lastStepTable = tr.getHistory().get(tr.getHistory().size() - 1).getResultTable();
		if (lastStepTable.getTableType().equals(new CodelistTableType())) {
			result.add(getSDMXCodelistExportCapability(lastStepTable.getId()));
		}
		return result;
	}

	@Override
	public List<EligibleOperation> getCapabilities(TabularResourceId tabularResourceId, ColumnLocalId columnId)
			throws NoSuchTabularResourceException, NoSuchColumnException {
		return ServiceState.columnCapabilities;
	}

	@Override
	public synchronized Task execute(OperationInvocation invocation, TabularResourceId targetTabularResourceId)
			throws NoSuchTabularResourceException, InvalidInvocationException {

		ServiceState.getTabularResourceById(targetTabularResourceId);
		List<Task> operations = getTasks(targetTabularResourceId);

		checkOperations(operations);

		Worker worker = new FakeWorker(invocation);

		DelegateTask delegateOperation = new DelegateTask(new ImmutableTask(new TaskId(String.valueOf(operationCounter++)), invocation,
				null, null, 0, ActivityStatus.INITIALIZING, new Date(), null, "user"));
		ServiceState.operations.get(targetTabularResourceId).add(delegateOperation);
		new Thread(new OperationUpdater(worker, delegateOperation, targetTabularResourceId, invocation)).start();

		return delegateOperation;

	}

	private void checkOperations(List<Task> operations) throws InvalidInvocationException {
		if (operations.isEmpty())
			return;
		Task lastOperation = operations.get(operations.size() - 1);

		switch (lastOperation.getStatus()) {
		case IN_PROGRESS:
		case INITIALIZING:
		case WAITING:
			throw new InvalidInvocationException("Cannot start operation while one is still in progress");
		default:
		}
	}

	@Override
	public Table rollbackToTable(TabularResourceId tabularResourceId, TableId toTableId)
			throws NoSuchTabularResourceException, NoSuchTableException {
		throw new UnsupportedOperationException("Rollback is not supported yet");
	}

	@Override
	public List<Task> getTasks(TabularResourceId tabularResourceId) throws NoSuchTabularResourceException {
		return ServiceState.getOperations(tabularResourceId);
	}

	@Override
	public Task getTask(TaskId operationId, TabularResourceId tabularResourceId) throws NoSuchTaskException,
			NoSuchTabularResourceException {
		if (operationId == null)
			throw new IllegalArgumentException("Must provide a non null operation id");
		if (tabularResourceId == null)
			throw new IllegalArgumentException("Must provide a non null tabular resource id");

		List<Task> operations = ServiceState.getOperations(tabularResourceId);

		for (Task operation : operations) {
			if (operation.getId().equals(operationId))
				return operation;
		}
		throw new NoSuchTaskException(tabularResourceId, operationId);
	}

	public class OperationUpdater implements Runnable {

		Worker worker;
		DelegateTask delegateOperation;
		TabularResourceId tabularResourceId;
		OperationInvocation invocation;

		public OperationUpdater(Worker worker, DelegateTask delegateOperation, TabularResourceId tabularResourceId,
				OperationInvocation invocation) {
			this.worker = worker;
			this.delegateOperation = delegateOperation;
			this.tabularResourceId = tabularResourceId;
			this.invocation = invocation;
		}

		@Override
		public void run() {

			new Thread(worker).start();
			Job job = null;
			do {
				try {
					job = worker.getJob();
					TaskJob taskStep = new ImmutableTaskJob(job, TaskJobClassifier.PROCESSING);
					Task newTask = new ImmutableTask(delegateOperation.getId(),
							delegateOperation.getSourceInvocation(), Lists.newArrayList(taskStep), null,
							taskStep.getProgress(), ActivityStatus.IN_PROGRESS, delegateOperation.getStartTime(),
							null, delegateOperation.getInvokerUsername());
					delegateOperation.setDelegate(newTask);
					// System.out.println("New operation status: " +
					// delegateOperation);
					Thread.sleep(1000);
				} catch (Exception e) {
					System.err.println("Error during execution of operation updater");
					// TODO-LF: handle exception
				}
			} while (job.getStatus() == ActivityStatus.INITIALIZING || job.getStatus() == ActivityStatus.IN_PROGRESS
					|| job.getStatus() == ActivityStatus.WAITING);
			TaskJob taskJob = new ImmutableTaskJob(worker.getJob(), TaskJobClassifier.PROCESSING);
			try {
				TabularResource tr = ServiceState.getTabularResourceById(tabularResourceId);
				HistoryStepImpl historyStep = new HistoryStepImpl(invocation, job.getResult().getOutput(), TaskJobClassifier.PROCESSING);
				tr.getHistory().add(historyStep);
			} catch (Exception e) {
				throw new RuntimeException("Tabular Resource not found! This should not happen, check your code.");
			}
			Task newTask = new ImmutableTask(delegateOperation.getId(), delegateOperation.getSourceInvocation(),
					Lists.newArrayList(taskJob), taskJob.getResult(), taskJob.getProgress(),
					ActivityStatus.SUCCEDED, delegateOperation.getStartTime(), new Date(),
					delegateOperation.getInvokerUsername());
			delegateOperation.setDelegate(newTask);
		}

	}

	public class FakeWorker implements Worker {

		private ImmutableJob job;
		private OperationInvocation invocation;

		public FakeWorker(OperationInvocation invocation) {
			this.invocation = invocation;
			job = new ImmutableJob(0f, ActivityStatus.INITIALIZING, invocation);
		}

		@Override
		public void run() {
			try {
				for (int i = 0; i < 9; i++) {
					job = new ImmutableJob(job.getProgress() + 0.1f, ActivityStatus.IN_PROGRESS, invocation);
					// System.out.println("New Job status: " + job);
					Thread.sleep(1000);
				}
				job = new ImmutableJob(1f, ActivityStatus.SUCCEDED, invocation, new ImmutableJobResult(
						ServiceState.tables.get(0)));
			} catch (Exception e) {
				// TODO-LF handle in a better way
				System.err.println("Error during execution of fakeworker");
			}
		}

		@Override
		public ImmutableJob getJob() {
			return job;
		}

	}

}
