package org.gcube.data.analysis.tabulardata.model.datatype;

import com.google.common.base.Optional;

/**
 * The type numeric can store numbers with up to 1000 digits of precision and
 * perform calculations exactly. 
 * A Numeric is defined  with a precision and a scale.
 * The scale of a numeric is the count of decimal
 * digits in the fractional part, to the right of the decimal point. The
 * precision of a numeric is the total count of significant digits in the whole
 * number, that is, the number of digits to both sides of the decimal point.
 * 
 * @author "Luigi Fortunati"
 * 
 */
public class Numeric implements DataType {

	/**
	 * 
	 */
	private static final long serialVersionUID = -1778127136791503220L;

	/**
	 * The precision is the total count of significant digits in the whole
	 * number.
	 */
	Optional<java.lang.Integer> precision;

	/**
	 * The scale of a numeric is the count of decimal digits in the fractional
	 * part.
	 */
	Optional<java.lang.Integer> scale;

	/**
	 * Define a Numeric type with default precision and scale
	 */
	public Numeric() {
		scale = Optional.absent();
		precision = Optional.absent();
	}

	/**
	 * Creates a Numeric type with a given precision
	 * @param precision total count of significant digits in the whole number
	 */
	public Numeric(int precision) {
		if (precision <= 0)
			throw new IllegalArgumentException("Precision must be positive");
		this.precision = Optional.of(precision);
	}

	/**
	 * Creates a Numeric type with a given precision and scale.
	 * @param precision total count of significant digits in the whole number
	 * @param scale count of decimal digits in the fractional part
	 * @throws IllegalArgumentException
	 */
	public Numeric(int precision, int scale) {
		if (precision <= 0)
			throw new IllegalArgumentException("Precision must be positive.");
		if (scale < 0 || scale > precision )
			throw new IllegalArgumentException("Scale must be 0 or less than precision.");
		this.precision = Optional.of(precision);
		this.scale = Optional.of(scale);
	}

	public DataTypeEnum getDataType() {
		return DataTypeEnum.NUMERIC;
	}

	public boolean hasPrecision() {
		return precision.isPresent();
	}

	public int getPrecision() {
		return precision.get();
	}

	public boolean hasScale() {
		return scale.isPresent();
	}

	public int getScale() {
		return scale.get();
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((precision == null) ? 0 : precision.hashCode());
		result = prime * result + ((scale == null) ? 0 : scale.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Numeric other = (Numeric) obj;
		if (precision == null) {
			if (other.precision != null)
				return false;
		} else if (!precision.equals(other.precision))
			return false;
		if (scale == null) {
			if (other.scale != null)
				return false;
		} else if (!scale.equals(other.scale))
			return false;
		return true;
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("Numeric(");
		builder.append(precision);
		builder.append(",");
		builder.append(scale);
		builder.append(")");
		return builder.toString();
	}

}
