package org.gcube.data.analysis.tabulardata.query;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Iterator;

import org.apache.commons.dbutils.DbUtils;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sqlfilter.SQLFilterExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.query.params.Filter;
import org.gcube.data.analysis.tabulardata.query.params.Ordering;
import org.gcube.data.analysis.tabulardata.query.params.Page;
import org.gcube.data.analysis.tabulardata.query.sql.SQLResultSetIterator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TabularQueryImpl implements TabularQuery {

	private Logger log = LoggerFactory.getLogger(TabularQueryImpl.class);

	private TabularQueryUtils queryUtils;

	private Table table;

	private Filter filter = null;

	private Ordering ordering = null;

	private Page page;
	
	private QueryBuilder queryBuilder = new QueryBuilder();

	public TabularQueryImpl(TabularQueryUtils queryUtils, Table table) {
		super();
		this.queryUtils = queryUtils;
		this.table = table;
	}

	@Override
	public TabularQuery setFilter(Filter filter) {
		this.filter = filter;
		return this;
	}

	@Override
	public TabularQuery setOrdering(Ordering ordering) {
		this.ordering = ordering;
		return this;
	}

	@Override
	public int getTotalTuples() {
		String sql = queryBuilder.buildCountTuplesQuery();
		ResultSet rs = queryUtils.executeSQLQuery(sql);
		int totalTuples = parseGetTotalTuplesQueryResult(rs); 
		return totalTuples;
	}

	@Override
	public Iterator<Object[]> getPage(Page page) {
		this.page = page;
		return executeQuery();
	}

	@Override
	public Iterator<Object[]> getAll() {
		this.page=null;
		return executeQuery();
	}

	private Collection<Column> getColumnsToShow() {
		return table.getColumns();
	}

	private Iterator<Object[]> executeQuery(){
		checkIfOrderingColumnExists();
	
		String query = queryBuilder.buildQuery();
	
		return new SQLResultSetIterator(queryUtils.executeSQLQuery(query));
	}

	private void checkIfOrderingColumnExists() {
		if (ordering==null)return;
		try {
			table.getColumnByName(ordering.getOrderingColumnName());
		} catch (Exception e) {
			log.error(String.format("Provided ordering column name '%s' is not valid within table %s.",
					ordering.getOrderingColumnName(), table));
			throw new IllegalArgumentException("Given ordering column with name " + ordering.getOrderingColumnName()
					+ " does not exists within the table with id " + table.getId());
		}
	}

	private int parseGetTotalTuplesQueryResult(ResultSet rs) {
		try {
			rs.next();
			int result = rs.getInt(1);
			DbUtils.closeQuietly(rs);
			return result;
		} catch (SQLException e) {
			log.error("An error occurred while querying the database.", e);
			throw new RuntimeException("An error occurred while querying the database. Check server logs");
		}
	}

	private class QueryBuilder {

		private String buildQuery() {
			StringBuilder queryBuilder = new StringBuilder();

			queryBuilder.append(getQuerySelectPart());
			queryBuilder.append(getQueryFilterPart());
			queryBuilder.append(getOrderQueryPart());
			queryBuilder.append(getLimitQueryPart());
			queryBuilder.append(";");

			return queryBuilder.toString();
		}

		private String getQuerySelectPart() {
			String columns = buildColumnCommaSeparatedList(getColumnsToShow());
			return String.format("SELECT %s FROM %s ", columns, table.getName());
		}

		private String getQueryFilterPart() {
			if (filter==null) return "";
			SQLFilterExpressionEvaluatorFactory evaluatorFactory = SQLFilterExpressionEvaluatorFactory.getInstance();
			String whereCondition = evaluatorFactory.getEvaluator(filter.getFilterExpression()).evaluate();
			return String.format(" WHERE %s ", whereCondition);
		}

		private String getOrderQueryPart() {
			if (ordering==null) return "";
			return String.format(" ORDER BY %s %s ", ordering.getOrderingColumnName(), ordering
					.getOrderingDirection().getSQLKeyword());
		}

		private String getLimitQueryPart() {
			if (page==null) return "";
			return String.format(" LIMIT %s OFFSET %s ", page.getPageSize(), page.getOffset());
		}

		private String buildColumnCommaSeparatedList(Collection<Column> columns) {
			String columnNames = "";
			int i = 1;
			for (Column c : columns) {
				columnNames += c.getName();
				if (i++ < columns.size())
					columnNames += ",";
			}
			return columnNames;
		}
		
		private String buildCountTuplesQuery() {
			StringBuilder stringBuilder = new StringBuilder();
			stringBuilder.append(buildSelectCountFirstPart());
			stringBuilder.append(getQueryFilterPart());
			stringBuilder.append(";");
			return stringBuilder.toString();
			
		}
		
		private String buildSelectCountFirstPart(){
			return String.format("SELECT COUNT(*) FROM %s ", table.getName());
		}

	}

}
