package org.gcube.data.analysis.tabulardata.query;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Collection;
import java.util.List;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.apache.commons.dbutils.DbUtils;
import org.apache.commons.dbutils.QueryRunner;
import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.cube.data.connection.unprivileged.Unprivileged;
import org.gcube.data.analysis.tabulardata.cube.exceptions.NoSuchTableException;
import org.gcube.data.analysis.tabulardata.cube.exceptions.TableCreationException;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableCreator;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnType;
import org.gcube.data.analysis.tabulardata.model.metadata.levels.Level;
import org.gcube.data.analysis.tabulardata.model.metadata.levels.Levels;
import org.gcube.data.analysis.tabulardata.model.relationship.TableRelationship;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

@Singleton
public class TabularQueryUtils {

	private DatabaseConnectionProvider connectionProvider;

	private CubeManager cm;

	private Logger log = LoggerFactory.getLogger(TabularQueryUtils.class);

	@Inject
	public TabularQueryUtils(@Unprivileged DatabaseConnectionProvider connectionProvider, CubeManager cm) {
		super();
		this.connectionProvider = connectionProvider;
		this.cm = cm;
	}

	public Table generateDatasetView(long tableId) {
		Table dataset = getTable(tableId);

		checkIfTableIsDataset(dataset);

		TableCreator ttc = cm.createTable(TableType.VIEWTABLE);
		try {
			ttc.like(dataset, false);
			Collection<Column> crCols = dataset.getColumns(ColumnType.CODELISTREF);
			for (Column column : crCols) {
				Table refTable = cm.getTable(column.getRelationship().getTargetTableRef().getTableId());
				List<Level> levels = refTable.getMetadataObject(Levels.class);
				int i = 0;
				for (String viewColumnName : levels.get(0).getViewColumnNames()) {
					Column refColumn = refTable.getColumnByName(viewColumnName);
					refColumn.setName(column.getName() + "_view" + i);
					refColumn.setLabel(column.getName() + "_view" + i);
					i++;
					ttc.addColumn(refColumn);
				}
			}
			Table viewTable = ttc.create();
			log.debug("Created view: " + viewTable);
			// Fill with data

			// Create SELECT statement
			Collection<Table> refTables = Lists.newArrayList();
			for (TableRelationship rel : dataset.getRelationships()) {
				refTables.add(cm.getTable(rel.getTargetTableRef().getTableId()));
			}
			StringBuilder sqlSelect = new StringBuilder();
			sqlSelect.append("SELECT ");
			for (Column column : dataset.getColumns()) {
				if (column.getColumnType() == ColumnType.ID)
					continue;
				sqlSelect.append("d." + column.getName() + " ,");
			}
			int i = 0;
			for (Table table : refTables) {
				List<Level> levels = table.getMetadataObject(Levels.class);
				// TODO for now handle single level only. Multilevel management
				// with hcl and such will be done in the future
				Level level = levels.get(0);
				for (String levelColumnName : level.getViewColumnNames()) {
					sqlSelect.append("c" + i + "." + levelColumnName + " ,");
				}
				i++;
			}
			sqlSelect.deleteCharAt(sqlSelect.length() - 1); // Remove last comma
			sqlSelect.append("FROM " + dataset.getName() + " AS d ");
			i = 0;
			for (TableRelationship rel : dataset.getRelationships()) {

				Table refTable = cm.getTable(rel.getTargetTableRef().getTableId());
				String refTableAlias = "c" + i;
				sqlSelect.append(" LEFT JOIN " + refTable.getName() + " AS " + refTableAlias + " ON d."
						+ rel.getSourceTableFKRef().getColumnName() + " = " + refTableAlias + ".id");
				i++;
			}
			// Create INSERT statement
			StringBuilder orderedColumnNames = new StringBuilder();
			for (Column column : viewTable.getColumns()) {
				orderedColumnNames.append(column.getName() + " ,");
			}
			orderedColumnNames.deleteCharAt(orderedColumnNames.length() - 1); // Remove
																				// last
																				// comma
			String sqlInsert = String.format("INSERT INTO %s (%s) ", viewTable.getName(), orderedColumnNames);

			String sql = sqlInsert + sqlSelect + ";";
			log.debug("Executing view filling SQL statement: " + sql);
			executeSQLCommand(sql);

			return viewTable;
		} catch (TableCreationException e) {
			log.error("Unable to create table", e);
			throw new RuntimeException("An error occurred during table creation. Check server logs.");
		} catch (NoSuchTableException e) {
			log.error("Unable to find referenced table.", e);
			throw new RuntimeException("An error occurred during table creation. Check server logs.");
		} catch (Exception e) {
			log.error("Generic exception detected.", e);
			throw new RuntimeException("An error occurred during table creation. Check server logs.");
		}
	}

	private Connection getConnection() throws SQLException {
		return connectionProvider.getConnection();
	}

	public ResultSet executeSQLQuery(String sql) {
		QueryRunner run = new QueryRunner();
		
		log.debug("Executing SQL Query: " + sql);
		Connection connection = null;
		Statement statement = null;
		ResultSet rs;
		try {
			connection = getConnection();
			statement = connection.createStatement();
			rs = statement.executeQuery(sql);
		} catch (SQLException e) {
			log.error("Unable to execute SQL query.", e);
			throw new RuntimeException("Unable to query the DB.");
		} finally {
			DbUtils.closeQuietly(connection);
		}
		return rs;
	}

	public void executeSQLCommand(String sql) {
		log.debug("Exceuting SQL command: " + sql);
		Connection connection = null;
		Statement statement = null;
		try {
			connection = getConnection();
			statement = connection.createStatement();
			statement.execute(sql);
		} catch (SQLException e) {
			log.error("Unable to execute SQL query.", e);
			throw new RuntimeException("Unable to query the DB.");
		} finally {
			DbUtils.closeQuietly(connection);
			DbUtils.closeQuietly(statement);
		}
	}

	private void checkIfTableIsDataset(Table dataset) {
		if (dataset.getTableType() != TableType.DATASET)
			throw new RuntimeException("The table with the given id is not a dataset.");
	}

	private Table getTable(long tableId) {
		try {
			return cm.getTable(tableId);
		} catch (NoSuchTableException e) {
			log.error("Error occurred while querying table", e);
			throw new RuntimeException(String.format(
					"Unable to retrieve metadata for the given table id '%s' . Check server logs.", tableId));
		}
	}

}
