package org.gcube.data.oai.tmplugin.repository;

import static org.gcube.data.streams.dsl.Streams.pipe;

import java.io.IOException;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.gcube.data.oai.tmplugin.OAIReader;
import org.gcube.data.oai.tmplugin.binders.OAIDCBinder;
import org.gcube.data.oai.tmplugin.repository.iterators.RepositoryIterator;
import org.gcube.data.oai.tmplugin.repository.iterators.SetIterator;
import org.gcube.data.oai.tmplugin.requests.Request;
import org.gcube.data.oai.tmplugin.utils.Constants;
import org.gcube.data.streams.Stream;
import org.gcube.data.streams.delegates.PipedStream;
import org.gcube.data.streams.exceptions.StreamException;
import org.gcube.data.streams.exceptions.StreamSkipSignal;
import org.gcube.data.streams.generators.Generator;
import org.gcube.data.tmf.api.exceptions.UnknownTreeException;
import org.gcube.data.trees.data.Tree;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.gcube.common.OAIException;
import org.gcube.common.OAIRecord;
import org.gcube.common.OAIRecordList;
import org.gcube.common.OAIRepository;
import org.gcube.common.OAISet;
import org.gcube.common.OAISetList;

/**
 * Default {@link Repository} implementation
 * 
 */
public class BaseRepository implements Repository{

	private static final long serialVersionUID = 1L;

	private static final Logger log = LoggerFactory.getLogger(BaseRepository.class);

	transient private OAIRepository connection;

	private final String url;
	private final String metadataFormat;
	transient private String description;
	transient private String name;
	transient private int cardinality;

	Stream<OAIRecord> records;

	private OAIDCBinder binder;


	// sees repository as a whole, no set boundaries
	public BaseRepository(Request request) throws Exception {
		this.url = request.getRepositoryUrl();
		this.metadataFormat = request.getMetadataFormat();
		this.description = request.getDescription();
		this.name = request.getName();
		//		log.info("url: " + this.url + " - metadataFormat: " + this.metadataFormat);
		connect(this.url);
		binder = new OAIDCBinder(request);
	}

	/**
	 * {@inheritDoc}
	 */
	public String url() {
		return url;
	}

	/**
	 * {@inheritDoc}
	 */
	public String metadataFormat() {
		return metadataFormat;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String name() {
		return name;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String description() {
		return description;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Tree get(String id, List<Set> sets) throws UnknownTreeException, Exception {

		try {

			OAIRecord record=connection.getRecord(id,metadataFormat);

			//check record is in one of sets
			if (!sets.isEmpty()) {

				//check membership to set
				boolean inSomeSet= false;
				loop: 
					for (int i=0; i<record.getSetSpecCount(); i++)
						for (Set set: sets)
							if(record.getSetSpec(i).equals(set.id())){
								inSomeSet= true;
								break loop;
							}

				if(!inSomeSet) 
					throw new UnknownTreeException();

			}
			//			log.info("record.getIdentifier() " + record.getIdentifier());
			return binder.bind(record);
		}
		catch(OAIException e) {
			throw new UnknownTreeException(id,e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Stream<Tree> getAllIn(List<Set> sets) {
		if (sets.isEmpty()){
			records = new RepositoryIterator() {
				@Override 
				protected OAIRecordList fetchRecords() throws OAIException {
					return connection.listRecords(metadataFormat);
				}


				@Override
				public boolean isClosed() {
					// TODO Auto-generated method stub
					return false;
				}			
			};
		}

		else
			records = new SetIterator(sets) {
			@Override 
			protected OAIRecordList fetchRecords(Set set) throws OAIException {
				return connection.listRecords(metadataFormat,null,null,set.id());
			}

			@Override
			public boolean isClosed() {
				// TODO Auto-generated method stub
				return false;
			}
		};



		Generator<OAIRecord,Tree> parser = new Generator<OAIRecord,Tree>() {
			public Tree yield(OAIRecord record) {
				try {
					return binder.bind(record);
				}
				catch(Exception e) {
					//					throw new StreamException();
					throw new StreamSkipSignal();
				}				
			}
		};

		PipedStream<OAIRecord, Tree> rec = pipe(records).through(parser);
		return rec;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Summary summary(List<Set> sets) throws Exception {

		Stream<OAIRecord> records = null;
		if (sets.isEmpty())
			records = new RepositoryIterator() {
			@Override 
			protected OAIRecordList fetchRecords() throws OAIException {
				return connection.listIdentifiers(null, null, null,metadataFormat);
			}

			@Override
			public boolean isClosed() {
				// TODO Auto-generated method stub
				return false;
			}
		};
		else
			records = new SetIterator(sets) {

			@Override 
			protected OAIRecordList fetchRecords(Set set) throws OAIException {
				return connection.listIdentifiers(null,null,set.id(),metadataFormat);
			}

			@Override
			public boolean isClosed() {
				// TODO Auto-generated method stub
				return false;
			}

		};

		Calendar lastUpdate=null;
		long cardinality=0;

		while (records.hasNext()){
			try {
				OAIRecord record= records.next();
				if (record==null || record.deleted() || !record.isRecordValid() || record.getMetadata()==null)
					//				if (record ==null || !record.isRecordValid() || record.deleted()) 
					continue;
				cardinality++;
				Calendar tempUpdate= Constants.getDate(record.getDatestamp());
				if (lastUpdate==null || tempUpdate.after(lastUpdate)) 
					lastUpdate= tempUpdate;
			}
			catch(Exception e) {
				log.error("could not count record",e);
			}
		}

		//attempts to recognise retrieval failures
		if(lastUpdate==null)
			throw new Exception();

		return new Summary(lastUpdate, cardinality);		
	}



	// used internally to establish a connection at instantiation and load time
	private void connect(String url) throws Exception {

		log.info("connecting to repository @ " + url);

		connection = new OAIRepository();

		try {
			connection.setBaseURL(url); // causes remote connection
			name = connection.getRepositoryName();

			if (connection.getDescriptionCount() > 0)

				description = connection.getDescription(0).getTextContent()
				.trim();
			else
				log.info("0 records");

		} catch (Exception e) {
			//			System.out.println("exc");
			//			log.error("could not connect to repository @ " + url);
			throw new Exception("could not connect to repository @ " + url, e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Set> getSetsWith(List<String> setIds) {

		boolean doAll = setIds.isEmpty();

		log.info(doAll ? "retrieving all sets" : "retrieving sets {}", setIds);

		List<String> done = new ArrayList<String>(setIds);
		List<Set> output = null;

		if (!setIds.isEmpty()){
			output = new ArrayList<Set>();
			try {

				OAISetList OAISets = connection.listSets(); // remote connection

				while (OAISets.moreItems()) {

					OAISet set = OAISets.getCurrentItem();
					//				log.info(set.getSetName() + " " + set.getSetSpec() + " " + set.getSetDescription());

					String setId = set.getSetSpec();
					//				log.info(setId);
					// create only if client did not specify sets or specified this
					// set
					if (doAll || done.contains(setId)) {
						String description = set.getSetDescriptionCount() > 0 ? set
								.getSetDescription(0).getTextContent() : null;
								output.add(new Set(setId, set.getSetName(), description));

								done.remove(setId);

								if (!doAll && done.isEmpty())
									break;
					}

					OAISets.moveNext();
				}


			} catch (OAIException e) {
				log.error("could not process sets ", e);
				throw new RuntimeException("could not process sets ", e);
			}
		}

		if (!done.isEmpty()){
			log.error("unkwnon sets " + done);
			throw new RuntimeException("unkwnon sets " + done);
		}

		return output;
	}



	private void writeObject(ObjectOutputStream out) throws IOException {
		out.defaultWriteObject();
	}

	// invoked upon deserialisation, resets non-serializable defaults
	private void readObject(java.io.ObjectInputStream in) throws IOException,
	ClassNotFoundException {

		in.defaultReadObject();

		// check invariants
		if (url == null)
			throw new IOException(
					"invalid serialisation, missing respository url");

		// recreate connection connection
		try {
			connect(url);
		} catch (Exception e) {
			throw new IOException(e);
		}
	}

	@Override
	public long getCardinality() {
		long cardinality = 0;
		try{
			cardinality = (((RepositoryIterator) records).getCardinality());
		}catch (Exception e) {
			cardinality = (((SetIterator) records).getCardinality());
		}

		return cardinality;
	}


}
