package org.gcube.data.publishing.gis.geoserver;

import java.util.ArrayList;
import java.util.List;

import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.informationsystem.client.AtomicCondition;
import org.gcube.common.core.informationsystem.client.ISClient;
import org.gcube.common.core.informationsystem.client.queries.GCUBERuntimeResourceQuery;
import org.gcube.common.core.resources.GCUBERuntimeResource;
import org.gcube.common.core.resources.runtime.AccessPoint;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.common.geoserverinterface.GeoCaller;
import org.gcube.common.geoserverinterface.GeonetworkCommonResourceInterface.GeoserverMethodResearch;
import org.gcube.data.publishing.gis.geoserver.model.DBDescriptor;
import org.gcube.data.publishing.gis.geoserver.model.DBDescriptor.DBType;
import org.gcube.data.publishing.gis.geoserver.model.DataSourceDescriptor;
import org.gcube.data.publishing.gis.geoserver.model.GeoServerDescriptor;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.CSquareConverter;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.GISInteraction;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.GISInteractionFactory;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.types.GISDataType;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.types.GISFileType;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.types.GISRepositoryType;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.types.GISStreamedType;

public class GeoServerInteractionFactory implements GISInteractionFactory {

	
	private static GCUBELog logger= new GCUBELog(GeoServerInteractionFactory.class);
	protected static ISClient isClient;
	
	
	private static final List<GISDataType> supportedTypes=new ArrayList<GISDataType>();
	
	static{
		supportedTypes.add(GISStreamedType.COORDINATE_POINTS);
		supportedTypes.add(GISStreamedType.CSQUARE_POINTS);
		supportedTypes.add(GISStreamedType.GEOMETRIES);
		supportedTypes.add(GISFileType.GEOTIFF);
		supportedTypes.add(GISFileType.NETCDF);
		
		try {
			isClient = GHNContext.getImplementation(ISClient.class);
		} catch (Exception e) {
			logger.warn("Unable to get ISClient implementation",e);
		}
	}
	
	
	private List<DBDescriptor> databases=new ArrayList<DBDescriptor>();
	private List<GeoServerDescriptor> geoservers=new ArrayList<GeoServerDescriptor>();
	private List<DataSourceDescriptor> geonetworks=new ArrayList<DataSourceDescriptor>();
	private GCUBEScope scope;
	
	@Override
	public boolean init(GCUBEScope scope) {
		this.scope=scope;
		try{
			databases=getGISDataBase(scope);		
			geoservers=getGeoServers(scope);
			geonetworks=getGeoNetwork(scope);
			if(databases.size()==0) throw new Exception("No Gis DataBase found in scope");
			if(geoservers.size()==0) throw new Exception("No GeoServer found in scope");
			if(geonetworks.size()==0) throw new Exception("No Geonetwork found in scope");
			return true;
		}catch(Exception e){
			logger.error("Unable to gather informations under scope "+scope,e);
			return false;
		}
	}

	@Override
	public GISInteraction createInteraction(CSquareConverter converter) throws Exception {
		DataSourceDescriptor geoNetwork=geonetworks.get(0);
		GeoServerDescriptor geoServer=geoservers.get(0);
		GeoCaller caller=new GeoCaller(geoNetwork.getEntryPoint(), geoNetwork.getUser(), geoNetwork.getPassword(), 
				geoServer.getEntryPoint(),geoServer.getUser(),geoServer.getPassword()	,
				GeoserverMethodResearch.MOSTUNLOAD);		
		return new GeoServerPlugin(converter,caller,databases.get(0), geoServer);
	}

	@Override
	public void close() {
		// TODO Auto-generated method stub
		
	}

	@Override
	public GISRepositoryType getHandledRepositoryType() {
		return GISRepositoryType.GEOSERVER;
	}

	@Override
	public List<GISDataType> getHandledDataType() {
		return supportedTypes;
	}

	
	//************************************ IS QUERIES
	
	
	/**
	 * Query passed scope for RuntimeResources describing valid geoServers
	 * AtomicConditions : 	//Profile/Category = Constants.get().getGeoServerCategoryName()
	 * 						//Profile/Platform/Name = Constants.get().getGeoServerPlatformName()
	 * 
	 * Valid Access Points : 
	 * 				EntryName = Constants.get().getGeoServerEntryName()
	 * 		Mandatory properties :  
	 * 				Constants.get().getGeoServerAquaMapsWorkspace()
	 * 				Constants.get().getGeoServerAquaMapsDataStore()
	 * 				Constants.get().getGeoServerAquaMapsDefaultDistributionStyle()
	 * @param scope
	 * @return
	 * @throws Exception
	 */
	protected static List<GeoServerDescriptor> getGeoServers(GCUBEScope scope)throws Exception{
		logger.debug("Checking geoServers..");
		List<GeoServerDescriptor> toReturn=new ArrayList<GeoServerDescriptor>();
		GCUBERuntimeResourceQuery runtimeQuery=isClient.getQuery(GCUBERuntimeResourceQuery.class);
		runtimeQuery.addAtomicConditions(new AtomicCondition("//Profile/Category", Constants.get().getGeoServerCategoryName()));
		runtimeQuery.addAtomicConditions(new AtomicCondition("//Profile/Platform/Name", Constants.get().getGeoServerPlatformName()));
		List<GCUBERuntimeResource> list=isClient.execute(runtimeQuery, scope);
		for(GCUBERuntimeResource rr:list){
			try{
				for(AccessPoint access:rr.getAccessPoints()){
					if(access.getEntryname().equals(Constants.get().getGeoServerEntryName())){
						logger.debug("Found Access point "+access.getEntryname()+" in "+rr.getName()+" [ ID : "+rr.getID()+"]");
						toReturn.add(new GeoServerDescriptor(
								access.getEndpoint(), 
								access.getUsername(), 
								access.getPassword(), 
								access.getProperty(Constants.get().getGeoServerAquaMapsWorkspace()), 
								access.getProperty(Constants.get().getGeoServerAquaMapsDataStore()), 
								access.getProperty(Constants.get().getGeoServerAquaMapsDefaultDistributionStyle())));
					}
				}
			}catch(Exception e){
				logger.warn("Unable to parse resource [ID :"+rr.getID()+"]",e);
			}
		}
		return toReturn;
	}
	
	
	/**
	 * Query passed scope for RuntimeResources describing valid geoNetworks
	 * AtomicConditions : 	//Profile/Category = Constants.get().getGeoNetworkCategoryName()
	 * 						//Profile/Platform/Name = Constants.get().getGeoNetworkPlatformName()
	 * 
	 * Valid Access Points : 
	 * 				EntryName = Constants.get().getGeoNetworkEntryName()
	 * 
	 * @param scope
	 * @return
	 * @throws Exception
	 */
	protected static List<DataSourceDescriptor> getGeoNetwork(GCUBEScope scope)throws Exception{
		logger.debug("Checking geoNetwork..");
		ArrayList<DataSourceDescriptor> toReturn=new ArrayList<DataSourceDescriptor>();
		GCUBERuntimeResourceQuery runtimeQuery=isClient.getQuery(GCUBERuntimeResourceQuery.class);
		runtimeQuery.addAtomicConditions(new AtomicCondition("//Profile/Category", Constants.get().getGeoNetworkCategoryName()));
		runtimeQuery.addAtomicConditions(new AtomicCondition("//Profile/Platform/Name", Constants.get().getGeoNetworkPlatformName()));
		List<GCUBERuntimeResource> list=isClient.execute(runtimeQuery, scope);
		for(GCUBERuntimeResource rr:list){
			try{
				for(AccessPoint access:rr.getAccessPoints())
					if(access.getEntryname().equals(Constants.get().getGeoNetworkEntryName())){	
						logger.debug("Found Access point "+access.getEntryname()+" in "+rr.getName()+" [ ID : "+rr.getID()+"]");
						toReturn.add(new DataSourceDescriptor(
								access.getEndpoint(), 
								access.getUsername(),
								access.getPassword()));
					}
			}catch(Exception e){logger.warn("Unable to parse resource [ID :"+rr.getID()+"]",e);}
		}
		return toReturn;
	}

	/**
	 * Query passed scope for RuntimeResources describing valid GIS Database
	 * AtomicConditions : 	//Profile/Category = Constants.get().getGeoServerDBCategory()
	 * 						//Profile/Platform/Name = Constants.get().getGeoServerDBPlatformName()
	 * 
	 * Valid Access Points : 
	 * 				EntryName = Constants.get().getGeoServerDBEntryName()
	 * 
	 * 		Mandatory properties: 
	 * 				Constants.get().getGeoServerDBAquaMapsDataStore() : boolean = true
	 * 				Constants.get().getDBMaxConnection() : integer
	 * 				DBDescriptor.AQUAMAPS_WORLD_TABLE
	 * 
	 * @param scope
	 * @return
	 * @throws Exception
	 */
	protected static ArrayList<DBDescriptor> getGISDataBase(GCUBEScope scope)throws Exception{
		logger.debug("Checking gis DBs");		
		ArrayList<DBDescriptor> toReturn=new ArrayList<DBDescriptor>();
		GCUBERuntimeResourceQuery runtimeQuery=isClient.getQuery(GCUBERuntimeResourceQuery.class);
		runtimeQuery.addAtomicConditions(new AtomicCondition("//Profile/Category", Constants.get().getGeoServerDBCategory()));
		runtimeQuery.addAtomicConditions(new AtomicCondition("//Profile/Platform/Name", Constants.get().getGeoServerDBPlatformName()));
		List<GCUBERuntimeResource> list=isClient.execute(runtimeQuery, scope);
		for(GCUBERuntimeResource rr:list){
			try{
				for(AccessPoint access: rr.getAccessPoints()){			
					if(access.getEntryname().equals(Constants.get().getGeoServerDBEntryName())&&
							access.getAllPropertyNames().contains(Constants.get().getGeoServerDBAquaMapsDataStore())&&
							Boolean.parseBoolean(access.getProperty(Constants.get().getGeoServerDBAquaMapsDataStore()))){
						logger.debug("Found Access point "+access.getEntryname()+" in "+rr.getName()+" [ ID : "+rr.getID()+"]");
						DBDescriptor toAdd=new DBDescriptor(
								"jdbc:postgresql:"+access.getEndpoint(), 
								access.getUsername(), 
								access.getPassword(), 
								DBType.postgres,
								Integer.parseInt(access.getProperty(Constants.get().getDBMaxConnection())));
						toReturn.add(toAdd);
					}
				}
			}catch(Exception e){logger.warn("Unable to parse resource [ID :"+rr.getID()+"]",e);}

		}
		return toReturn;
	}
	
}
