package org.gcube.data.publishing.gis.publisher.csquare;

import static org.gcube.data.streams.dsl.Streams.convert;
import static org.gcube.data.streams.dsl.Streams.pipe;
import gr.uoa.di.madgik.grs.record.GenericRecord;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.net.URI;
import java.sql.Connection;
import java.sql.Date;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.HashMap;
import java.util.Map.Entry;
import java.util.concurrent.atomic.AtomicInteger;

import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.publishing.gis.publisher.common.Utils;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.CSquarePoint;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.GeometryPoint;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.utils.ResultGenerator;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.writers.rswrapper.ResultWrapper;
import org.gcube.data.streams.Stream;
import org.postgis.PGgeometry;

public class CSquarePolygonDBImpl implements CSquarePolygonsDBInterface {

	private static final Character CSV_DELIMITER=',';
	private static final String csquareCodeField="csquarecode";

	private static final String GEOM_FIELD="the_geom";


	private static final GCUBELog logger=new GCUBELog(CSquarePolygonDBImpl.class);

	private Connection connection=null;
	private CSQuarePolygonDBDescriptor dbDesc;
	private AtomicInteger openedResultSets=new AtomicInteger(0);


	public CSquarePolygonDBImpl(CSQuarePolygonDBDescriptor desc) {
		this.dbDesc=desc;
	}


	@Override
	public Table streamToTable(Stream<CSquarePoint> points) throws SQLException {
		Table toReturn=null;
		PreparedStatement ps=null;
		long count=0;
		while(points.hasNext()){
			CSquarePoint point=points.next();
			//			logger.debug("Read Point : "+point);
			if(toReturn==null) {
				toReturn=createTableFromPoint(point);
				ps=prepareStatement(toReturn,point);
			}
			if(insertPoint(ps,point)==0){
				//point not inserted
				logger.warn("Unable to insert point into table "+toReturn.getTableName());
			}else{
				count++;
			}
		}
		toReturn.setRowCount(count);
		return toReturn;
	}

	@Override
	public Table joinToWorld(Table toJoin) throws SQLException {
		WorldTable world=dbDesc.getWorldTables().get(0);
		String newTableName="t"+Utils.getUUID();
		String joinQuery="CREATE TABLE "+newTableName+" AS ( "+getJoinQuery(world, toJoin)+" )";
		logger.debug("Gonna execute join query : "+joinQuery);
		Statement stmt = null;
		try{
			stmt=connection.createStatement();
			stmt.execute(joinQuery);
			long actualCount=getTableCount(newTableName, connection);
			if(toJoin.getRowCount()!=actualCount) logger.warn("JOINED TABLE "+newTableName+" has unexpected row count "+actualCount+" (exp. "+toJoin.getRowCount()+")");
			return new Table(actualCount, getTableFields(newTableName, connection), newTableName);
		}finally{
			if(stmt!=null)stmt.close();
		}
	}

	@Override
	public File exportCSV(Table toExport) throws IOException, SQLException  {
		Statement stmt = null;
		try{
			File out=File.createTempFile("csquare", ".csv");
			stmt=connection.createStatement(ResultSet.TYPE_SCROLL_INSENSITIVE,ResultSet.CONCUR_READ_ONLY);
			//			String copyString ="COPY "+toExport.getTableName()+" TO '"+out.getAbsolutePath()+"' WITH DELIMITER '"+CSV_DELIMITER+"' CSV HEADER";
			logger.debug("Gonna exporting to  : "+out.getAbsolutePath());
			long count=Utils.resultSetToCSVFile(stmt.executeQuery("SELECT * FROM "+toExport.getTableName()), out.getAbsolutePath(), true);
			if(count!=toExport.getRowCount())logger.warn("Exported "+count+" out of "+toExport.getRowCount());
			return out;
		}finally{
			if(stmt!=null)stmt.close();
		}
	}



	private Connection getConnection() throws SQLException, ClassNotFoundException{
		Connection toReturn=DriverManager.getConnection(dbDesc.getEndpoint(),dbDesc.getUser(),dbDesc.getPassword());
		toReturn.setAutoCommit(false);
		((org.postgresql.PGConnection)toReturn).addDataType("geometry",Class.forName("org.postgis.PGgeometry"));
		return toReturn;
	}


	@Override
	public void openSession() throws SQLException, ClassNotFoundException {
		closeSession();
		this.connection=getConnection();
	}

	@Override
	public synchronized void closeSession() throws SQLException {
		if(connection!=null){
			Thread t=new Thread(){
				@Override
				public void run() {
					int i=0;
					while((i=openedResultSets.get())>0){
						logger.trace("Opened ResultSet count "+i);
						try{
							Thread.sleep(1000);
						}catch(InterruptedException e){}
					}
					logger.trace("All ResultSet closed, closing connection");
					try {
						connection.close();
					} catch (SQLException e) {
						logger.warn("Unable to close connection ",e);
					}
				}
			};
			t.setName("CSQ_CONN_"+t.getName());
			t.start();
		}
	}

	private Table createTableFromPoint(CSquarePoint point) throws SQLException{
		String toCreateTable="t"+Utils.getUUID();
		StringBuilder createQuery=new StringBuilder("CREATE TABLE "+toCreateTable+" ( "+csquareCodeField +" VARCHAR PRIMARY KEY,");
		for(Entry<String,Serializable> field:point.getAttributes().entrySet())
			createQuery.append(" "+field.getKey()+" "+getTypeDefinition(getSQLType(field.getValue()))+",");
		createQuery.deleteCharAt(createQuery.lastIndexOf(","));
		createQuery.append(")");
		String query=createQuery.toString();
		Statement stmt=null;
		try{
			stmt=connection.createStatement();
			logger.debug("Goin to execute "+query);
			stmt.execute(query);
			return new Table(0,getSQLTypeByPoint(point),toCreateTable);
		}finally{
			if(stmt!=null)stmt.close();
		}
	}

	private PreparedStatement prepareStatement(Table theTable,CSquarePoint point) throws SQLException{
		StringBuilder fieldsName=new StringBuilder("("+csquareCodeField+",");
		StringBuilder fieldsValues=new StringBuilder("(?,");
		for (String f: point.getAttributes().keySet()){
			fieldsValues.append("?,");
			fieldsName.append(f+",");
		}

		fieldsValues.deleteCharAt(fieldsValues.length()-1);
		fieldsValues.append(")");
		fieldsName.deleteCharAt(fieldsName.length()-1);
		fieldsName.append(")");

		String query="INSERT INTO "+theTable.getTableName()+" "+fieldsName+" VALUES "+fieldsValues;
		logger.debug("the prepared statement is :"+ query);
		PreparedStatement ps= connection.prepareStatement(query,Statement.NO_GENERATED_KEYS);
		return ps;
	}

	private int insertPoint(PreparedStatement ps,CSquarePoint point) throws SQLException{
		ps.setString(1, point.getcSquareCode());
		Serializable[] values=point.getAttributes().values().toArray(new Serializable[point.getAttributes().size()]);

		for(int i=0;i<values.length;i++){
			ps.setObject(i+2,values[i]);
		}
		return ps.executeUpdate();
	}

	private static final long getTableCount(String table,Connection conn) throws SQLException{
		Statement stmt=null;
		try{
			stmt=conn.createStatement();
			ResultSet rs=stmt.executeQuery("SELECT COUNT(*) FROM "+table);
			rs.next();
			return rs.getLong(1);
		}finally{
			if(stmt!=null)stmt.close();
		}
	}

	private static final HashMap<String,SQLType> getTableFields(String table,Connection conn)throws SQLException{
		Statement stmt=null;
		try{
			stmt=conn.createStatement();
			ResultSet rs=stmt.executeQuery("SELECT * FROM "+table+" LIMIT 1 OFFSET 0");
			rs.next();
			ResultSetMetaData meta=rs.getMetaData();
			HashMap<String,SQLType> toReturn=new HashMap<String, SQLType>();
			for(int i=1;i<=meta.getColumnCount();i++)
				toReturn.put(meta.getColumnName(i),getSQLType(rs.getObject(i)));
			return toReturn;
		}finally{
			if(stmt!=null)stmt.close();
		}
	}

	
	private static final HashMap<String,SQLType> getSQLTypeByPoint(CSquarePoint point){
		HashMap<String,SQLType> toReturn=new HashMap<String, SQLType>();
		toReturn.put("csquarecode", SQLType.STRING);
		for(Entry<String,Serializable> entry : point.getAttributes().entrySet())
			toReturn.put(entry.getKey().toLowerCase(), getSQLType(entry.getValue()));
		return toReturn;
	}
	
	private static final String getJoinQuery(WorldTable world,Table t){
		StringBuilder toReturn=new StringBuilder("SELECT T.*,");
		for(Entry<String, SQLType> worldField:world.getFields().entrySet()){
			if(!t.getFields().containsKey(worldField.getKey())&&!worldField.getKey().equalsIgnoreCase(csquareCodeField)) 
				toReturn.append(" W."+worldField.getKey()+",");
		}
		toReturn.deleteCharAt(toReturn.lastIndexOf(","));
		toReturn.append(" FROM "+t.getTableName()+" AS T INNER JOIN "+world.getTableName()+" AS W ");
		toReturn.append(" ON T."+csquareCodeField+" = W."+csquareCodeField);
		return toReturn.toString();
	}


	@Override
	public Stream<GeometryPoint> streamTableRows(final Table toStreamTable,GCUBEScope scope)
	throws Exception {
			final ResultWrapper<GeometryPoint> rs=new ResultWrapper<GeometryPoint>(scope);
			final Statement stmt =connection.createStatement(ResultSet.TYPE_SCROLL_INSENSITIVE,ResultSet.CONCUR_READ_ONLY);
			openedResultSets.incrementAndGet();
			final ResultSet rSql=stmt.executeQuery("SELECT * FROM "+toStreamTable.getTableName());
			Thread t=new Thread(){
				@Override
				public void run() {
					try{
						logger.debug("Streaming geometries from table "+toStreamTable.getTableName());
						ResultSetMetaData rsMeta=rSql.getMetaData();
						long count=0;
						long rowCount=0;
						while(rSql.next()){
							rowCount++;
							try{
								GeometryPoint point=new GeometryPoint();
								for(int i=1;i<=rsMeta.getColumnCount();i++){
									String colName=rsMeta.getColumnName(i);							
									if(colName.equalsIgnoreCase(GEOM_FIELD))point.setTheGeometry((PGgeometry) rSql.getObject(i));
									else point.getAttributes().put(colName, (Serializable) rSql.getObject(i));							
								}
								rs.add(point);
								count++;
							}catch(Throwable t){
								logger.warn("Unable to stream object",t);
							}
						}
					logger.debug("Streamed "+count+" geometries out of "+rowCount+" rows ");	
					}catch(Throwable t){
						logger.error("Unable to stream data",t);
					}finally{
						try {
							rs.close();							
						} catch (Exception e) {
							logger.error("Cannot close Result wrapper ",e);
						}
						try {
							rSql.close();
							stmt.close();
						} catch (SQLException e) {
							logger.error("Cannot close ResultSet",e);							
						}
						logger.debug("Closing rs, ramaining count "+openedResultSets.decrementAndGet());
					}
				}
			};
			t.setName("STREAMER"+t.getId());
				t.start();		
			return pipe(convert(new URI(rs.getLocator())).of(GenericRecord.class).withDefaults()).through(new ResultGenerator<GeometryPoint>());
	}

	
	///**************************** FIELD TYPE LOGIC 
	
	private static SQLType getSQLType(Object obj){
		if(obj.getClass().isAssignableFrom(Integer.class)) return SQLType.INTEGER;
		if(obj.getClass().isAssignableFrom(Long.class)) return SQLType.LONG;
		if(obj.getClass().isAssignableFrom(Float.class)) return SQLType.INTEGER;
		if(obj.getClass().isAssignableFrom(Date.class)||obj.getClass().isAssignableFrom(java.util.Date.class)) return SQLType.DATE;
		if(obj.getClass().isAssignableFrom(Time.class)) return SQLType.TIME;
		if(obj.getClass().isAssignableFrom(Timestamp.class)) return SQLType.TIMESTAMP;
		if(obj.getClass().isAssignableFrom(Boolean.class)) return SQLType.BOOLEAN;
		if(obj.getClass().isAssignableFrom(PGgeometry.class))return SQLType.GEOMETRY;
		return SQLType.TEXT;
	}

	private static int getType(SQLType type){
		switch(type){
		case BOOLEAN : return Types.BOOLEAN;
		case DATE : return Types.DATE;
		case FLOAT : return Types.DECIMAL;
		case GEOMETRY : return Types.OTHER;
		case INTEGER : return Types.INTEGER;
		case LONG : return Types.BIGINT;
		case STRING : return Types.VARCHAR;
		case TEXT : return Types.LONGNVARCHAR;
		case TIME : return Types.TIME;
		case TIMESTAMP : return Types.TIMESTAMP;
		case SERIAL : return Types.BIGINT;
		default : return Types.NULL;
		}
	}


	private static String getTypeDefinition(SQLType type){
		switch(type){
		case BOOLEAN : return "boolean";
		case DATE : return "date";
		case FLOAT : return "decimal";
		case GEOMETRY : return "geometry";
		case INTEGER : return "integer";
		case LONG : return "bigint";
		case STRING : return "varchar (200)";
		case TEXT : return "text";
		case TIME : return "time";
		case TIMESTAMP : return "timestamp";
		case SERIAL : return "serial";
		default : return "";
		}
	}
	
}
