package org.gcube.data.spd.irmng;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.spd.irmng.dbconnection.ConnectionPool;
import org.gcube.data.spd.irmng.dbconnection.ConnectionPoolException;
import org.gcube.data.spd.irmng.dbconnection.RSItem;
import org.gcube.data.spd.plugin.fwk.Properties;
import org.gcube.data.spd.plugin.fwk.Property;
import org.gcube.data.spd.plugin.fwk.capabilities.ClassificationCapability;
import org.gcube.data.spd.plugin.fwk.exceptions.ExternalRepositoryException;
import org.gcube.data.spd.plugin.fwk.exceptions.IdNotValidException;
import org.gcube.data.spd.plugin.fwk.exceptions.MethodNotSupportedException;
import org.gcube.data.spd.plugin.fwk.model.TaxonomyItem;
import org.gcube.data.spd.plugin.fwk.model.TaxonomyStatus;
import org.gcube.data.spd.plugin.fwk.model.TaxonomyStatus.Status;
import org.gcube.data.spd.plugin.fwk.writers.ObjectWriter;

public class ClassificationCapabilityImpl extends ClassificationCapability {

	GCUBELog logger = new GCUBELog(ClassificationCapabilityImpl.class);

	public Set<Properties> getSupportedProperties() {
		return Collections.emptySet();
	}

	@Override
	public void getSynonymnsById(ObjectWriter<TaxonomyItem> writer, String id)
			throws IdNotValidException, MethodNotSupportedException,
			ExternalRepositoryException {
		logger.trace("getSynonimnsById " + id);

		ResultSet results =  null;
		try{
			results = getSynRSItem(id);
			if (results!=null){	
				String idSyn = null;
				while (results.next()){
					idSyn = results.getString(1);
					TaxonomyItem tax = retrieveTaxonById(idSyn);
					if (tax!=null)
						writer.put(tax);
				}
			}
		} catch (SQLException ex) {
			logger.error("sql Error", ex);
		}finally{
			writer.close();
			try {
				if (results != null) {
					results.close();
				}
			} catch (SQLException ex) {
				logger.error("sql Error", ex);
			}
		}
	}

	/**
	 * get a RSItem by id
	 */
	private ResultSet getSynRSItem(String id) {

		ConnectionPool pool = null;
		Connection con = null;
		ResultSet results = null;
		try {
			pool = ConnectionPool.getConnectionPool();
			con = pool.getConnection();

			String query = "select distinct(taxonid) from taxon where taxonomicstatus = 'synonym' and acceptednameusageid = ?";
			results =  pool.selectPrestatement(query, id);

		}		
		catch (Throwable e) {
			logger.error("general Error", e);
		}finally{
			if ((pool!=null) && (con!=null)){
				pool.releaseConnection(con);
			}			
		}
		return results;
	}


	@Override
	public List<TaxonomyItem> retrieveTaxonChildsByTaxonId(String id_parent) {

		logger.trace("retrieveTaxonChildsByTaxonId " + id_parent);
		List<TaxonomyItem> list = null;
		ResultSet results = null;
		try{
			results = getRSItemChildren(id_parent);
			if (results!=null){
				list = new ArrayList<TaxonomyItem>(); 
				while(results.next()) {	

					RSItem item = null;
					try {
						item = new RSItem();
						item.setId(results.getString(1));
						item.setRank(results.getString(2));
						item.setAuthor(results.getString(3));
						item.setStatus(results.getString(4));
						item.setModified(results.getString(5));
						item.setAcceptednameusageid(results.getString(6));						
						item.setNameaccordingto(results.getString(7));
						
						item.setParentId(id_parent);
						
						TaxonomyItem tax = createTaxonomyItem(item, false);
						list.add(tax);
					}finally{
						if (item!=null){
							try {
								item.finalize();
							} catch (Throwable e) {
								logger.error("Error deleting RSItem", e);
							}
						}
					}
				}
			}
		} catch (SQLException ex) {
			logger.error("sql Error", ex);
		}finally{
			try {
				if (results != null) {
					results.close();
				}
			} catch (SQLException ex) {
				logger.error("sql Error", ex);
			}
		}
		return list;	

	}

	/**
	 * get a list of RSItem by id
	 */
	private ResultSet getRSItemChildren(String id) {

		ConnectionPool pool = null;
		Connection con = null;
		ResultSet results = null;
		try {
			pool = ConnectionPool.getConnectionPool();
			con = pool.getConnection();

			String query = "select taxonid, taxonrank, scientificnameauthorship, taxonomicstatus, modified, acceptednameusageid, nameaccordingto from taxon where parentnameusageid = ?";
			results =  pool.selectPrestatement(query, id);
//			logger.trace(query);
		}
		catch (Throwable e) {
			logger.error("general Error", e);
		}finally{
			if ((pool!=null) && (con!=null)){
				pool.releaseConnection(con);
			}
		}
		return results;

	}


	/**
	 * Set status in Result Item
	 */
	private TaxonomyStatus setTaxStatus(String status, String syn_parent) {

		TaxonomyStatus tax = null;
		if ( status!= null){
			if (status.equals("accepted"))
				tax = new TaxonomyStatus("accepted", Status.ACCEPTED);
			else if (status.equals("valid"))
				tax = new TaxonomyStatus("valid", Status.VALID);
			else if (status.equals("synonym"))
				tax = new TaxonomyStatus(Status.SYNONYM, syn_parent, "synonym");
			else
				tax = new TaxonomyStatus(status, Status.UNKNOWN);
		} 
		else
			tax = new TaxonomyStatus(status, Status.UNKNOWN);
		//		else
		//			tax = new TaxonomyStatus(Status.UNKNOWN);
		return tax;
	}


	@Override
	public void searchByCommonName(String name, ObjectWriter<TaxonomyItem> writer, Property... arg2) {
	}

	@Override
	public void searchByScientificName(String scientificName, ObjectWriter<TaxonomyItem> writer, Property... arg2) {

		logger.trace("searchByScientificName " + scientificName);
		ResultSet results = null;
		try{
			results = getRSItemByName(scientificName);
			if (results!=null){		
				while(results.next()) {
					RSItem item = null;
					try{
						item = new RSItem();

						item.setId(results.getString(1));
						item.setRank(results.getString(2));
						item.setAuthor(results.getString(3));
						item.setStatus(results.getString(4));
						item.setModified(results.getString(5));
						item.setAcceptednameusageid(results.getString(6));
						item.setParentId(results.getString(7));
						item.setNameaccordingto(results.getString(8));

						TaxonomyItem tax = createTaxonomyItem(item, true);
						if (tax!=null)
							writer.put(tax);	
					}finally{
						if (item!=null)
							try {
								item.finalize();
							} catch (Throwable e) {
								logger.error("Error deleting RSItem", e);
							}
					}
				}
			}
		}catch (SQLException sqlExcept) {        	
			logger.error("sql Error", sqlExcept);
		}finally{
			writer.close();
			try {
				if (results != null) {
					results.close();
				}
			} catch (SQLException ex) {
				logger.error("sql Error", ex);
			}
		}
	}


	/**
	 * Create Taxonomy Item by result
	 */
	private ResultSet getRSItemByName(String scientificName) {

		ConnectionPool pool = null;
		Connection con = null;
		ResultSet results = null;
		try {
			pool = ConnectionPool.getConnectionPool();
			con = pool.getConnection();

			String name = "%" + scientificName + "%";
			String query = "select taxonid, taxonrank, scientificnameauthorship, taxonomicstatus, modified, acceptednameusageid, parentnameusageid, nameaccordingto from taxon where UPPER(scientificname) like UPPER(?)";	
			results =  pool.selectPrestatement(query, name);

		}catch (ConnectionPoolException e) {
			logger.error("ConnectionPoolException",e);
		} catch (SQLException e) {
			logger.error("SQLException",e);
		}finally{
			if ((pool!=null) && (con!=null)){
				pool.releaseConnection(con);
			}
		}
		return results;
	}


	@Override
	public void retrieveTaxonByIds(Iterator<String> ids, ObjectWriter<TaxonomyItem> writer) {
		try{
			while(ids.hasNext()) {
				String id = ids.next(); 
				//logger.trace("retrieveTaxonById " + id);
				TaxonomyItem item = retrieveTaxonById(id);
				writer.put(item);
			}
		} catch (Exception e) {
			logger.error("ExceptionPrintStackTrace",e);
		} finally{
			writer.close();	
		}
	}

	/**
	 * Create Taxonomy Item by result (if flag = false, parent=null)
	 */
	private TaxonomyItem createTaxonomyItem(RSItem rs, boolean flag) throws SQLException {

		Calendar dateModified = Utils.getCalendar(rs.getModified());		

		TaxonomyItem item = null;

		try{
			item = new TaxonomyItem(rs.getId());

			item.setScientificName(Utils.setScName(rs.getId(), rs.getRank()));
			item.setAuthor(rs.getAuthor());		
			item.setRank(rs.getRank());

			item.setCredits(Utils.createCredits());

			StringBuilder cit = new StringBuilder();
			String  citation = rs.getNameaccordingto();
			if (citation != null){
				cit.append(citation);
				cit.append(". ");
			}
			cit.append(Utils.createCitation());
			item.setCitation(cit.toString());

			item.setStatus(setTaxStatus(rs.getStatus(), rs.getAcceptednameusageid()));
			item.setModified(dateModified);

			if (flag){
				String id_parent = rs.getParentId();
				//parent null
				if  (id_parent != null){   
					try {
						item.setParent(retrieveTaxonById(id_parent));
					} catch (IdNotValidException e) {
						logger.error("Id Not Valid",e);
					}	
				}
			}
			else
				item.setParent(null);

		}finally{
			if (rs!=null)
				try {
					rs.finalize();
				} catch (Throwable e) {
					logger.error("Error deleting RSItem", e);
				}
		}
		return item;
	}



	@Override
	public TaxonomyItem retrieveTaxonById(String id)
			throws IdNotValidException {
		TaxonomyItem item = null;
		RSItem rs = null;

		try{
			rs = getRSItemById(id);
			if (rs!=null){
				try {
					item = createTaxonomyItem(rs, true);
				} catch (SQLException e) {
					logger.error("sql Error", e);
				}				
			}	
		}finally{
			if (rs!=null)
				try {
					rs.finalize();
				} catch (Throwable e) {
					logger.error("Error deleting RSItem", e);
				}
		}
		return item;
	}

	/**
	 * get a RSItem by id
	 */
	private RSItem getRSItemById(String id) {

		RSItem item = null;
		ConnectionPool pool = null;
		Connection con = null;
		ResultSet result = null;
		try {
			pool = ConnectionPool.getConnectionPool();
			con = pool.getConnection();

			String query = "select taxonrank, scientificnameauthorship, taxonomicstatus, modified, acceptednameusageid, parentnameusageid, nameaccordingto from taxon where taxonid = ?";	
			result =  pool.selectPrestatement(query, id);

			if (result!=null){
				if(result.next()) {	     	
					item = new RSItem();
					item.setId(id);
					//logger.trace("*********"+result.getString(1));
					item.setRank(result.getString(1));
					item.setAuthor(result.getString(2));
					item.setStatus(result.getString(3));
					item.setModified(result.getString(4));
					item.setAcceptednameusageid(result.getString(5));
					item.setParentId(result.getString(6));
					item.setNameaccordingto(result.getString(7));
				}
			}
		}
		catch (SQLException sqlExcept) {
			logger.error("sql Error",sqlExcept);
		} catch (ConnectionPoolException e) {
			logger.error("ConnectionPoolException",e);
		} finally{

			if ((pool!=null) && (con!=null)){
				pool.releaseConnection(con);
			}
			try {
				if (result != null) {
					result.close();
				}
			} catch (SQLException ex) {
				logger.error("sql Error", ex);
			}
		}
		return item;		
	}


}

