package org.gcube.data.spd.specieslink;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.gcube.common.core.resources.GCUBERuntimeResource;
import org.gcube.common.core.resources.runtime.AccessPoint;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.spd.parser.DarwinRecord;
import org.gcube.data.spd.parser.RecordsIterator;
import org.gcube.data.spd.plugin.fwk.AbstractPlugin;
import org.gcube.data.spd.plugin.fwk.Capabilities;
import org.gcube.data.spd.plugin.fwk.Property;
import org.gcube.data.spd.plugin.fwk.SearchTypes;
import org.gcube.data.spd.plugin.fwk.capabilities.OccurrencesCapability;
import org.gcube.data.spd.plugin.fwk.exceptions.StreamBlockingException;
import org.gcube.data.spd.plugin.fwk.model.DataProvider;
import org.gcube.data.spd.plugin.fwk.model.DataSet;
import org.gcube.data.spd.plugin.fwk.model.Product;
import org.gcube.data.spd.plugin.fwk.model.ResultItem;
import org.gcube.data.spd.plugin.fwk.model.Taxon;
import org.gcube.data.spd.plugin.fwk.util.RepositoryInfo;
import org.gcube.data.spd.plugin.fwk.writers.ObjectWriter;

public class SpeciesLinkPlugin extends  AbstractPlugin{

	GCUBELog logger = new GCUBELog(SpeciesLinkPlugin.class);

	public static String baseurl;
	public static String model;

	public final static String credits ="This information object has been generated via the Species Product Discovery service on XDATEX by interfacing with speciesLink (http://splink.cria.org.br/)";
	public final static String citation = "Accessed through: speciesLink at http://splink.cria.org.br/ on XDATEX";


//			public static String baseurl = "http://tapir.cria.org.br/tapirlink/tapir.php/specieslink";
//			public static String model = "http://rs.tdwg.org/tapir/cs/dwc/1.4/model/dw_core_geo_cur.xml";

	@Override
	public void initialize(GCUBERuntimeResource res) throws Exception {
		//		logger.trace("SpeciesPlugin starting initialization");
		setUseCache(true);
		for (AccessPoint ap:res.getAccessPoints()){		
			if (ap.getEntryname().equals("tapir")){
				baseurl = ap.getEndpoint();
				model = ap.getProperty("model");
			}
		}
		logger.trace("SpeciesPlugin initialized");
		super.initialize(res);
	}


	@Override
	public void update(GCUBERuntimeResource res) throws Exception {

		for (AccessPoint ap:res.getAccessPoints()){		
			if (ap.getEntryname().equals("tapir")){
				baseurl = ap.getEndpoint();
				model = ap.getProperty("model");
			}
		}
		logger.trace("SpeciesPlugin updated");
		super.update(res);
	}

	@Override
	public String getDescription() {
		return ("Species Link Plugin");
	}

	@Override
	public String getRepositoryName() {
		return ("SpeciesLink");
	}

	@Override
	public Set<SearchTypes> getSupportedSearch() {
		return Collections.singleton(SearchTypes.ScientificName);
	}


	@SuppressWarnings("serial")
	@Override
	public Set<Capabilities> getSupportedCapabilities() {
		return new HashSet<Capabilities>(){{add(Capabilities.Occurences);}};
	}


	@Override
	public OccurrencesCapability getOccurrencesInterface() {
		return new OccurrencesCapabilityImpl(); 
	}


	@Override
	public void searchByCommonName(String arg0, ObjectWriter<ResultItem> arg1,
			Property... arg2) {

	}

	@Override
	public void searchByScientificName(String scientificName, ObjectWriter<ResultItem> writer, Property... p) {

		String f = "";
		try {
			f = Utils.elaborateProps(p);
		} catch (Exception e) {
			logger.error("error elaborating properties",e);
			writer.close();
			return;
		}

		String filter = "http://rs.tdwg.org/dwc/dwcore/ScientificName%20like%20%22" + scientificName.replace(" ", "%20") + "%22" + f + "&orderBy=http://rs.tdwg.org/dwc/dwcore/ScientificName&orderBy=http://rs.tdwg.org/dwc/dwcore/InstitutionCode";

		RecordsIterator set = new RecordsIterator(baseurl, filter, model);
		Iterator<DarwinRecord> it = set.iterator();
		DarwinRecord element = null;
		DarwinRecord element1 = null;
		int count = 0;

		boolean flag = false;
		try{
			while (it.hasNext()){
				//							logger.trace("result found");
				flag = true;
				element = it.next();

				if (element1 == null){
					//				logger.trace("element1 null");
					element1 = element;
				}
				//				logger.trace("cod " + element1.institutionCode + " scientificname " + element1.scientificName);
				//group results by istitutionCode
				if (((element1.institutionCode).toLowerCase()).equals((element.institutionCode).toLowerCase()) & ((element1.scientificName).toLowerCase()).equals((element.scientificName).toLowerCase()))			
					count++;
				else{		
					//				logger.trace("set result");
					if (setResult(element1, count, writer)){
						element1 = element;
						count = 1;
					}else{
						flag = false;
						break;
					}
				}			
			}

			if (flag){
				//			logger.trace("flag true");
				setResult(element1, count, writer);		
			}

		}catch (Exception e) {
			logger.error("General Error", e);
			writer.write(new StreamBlockingException());
		}finally{
			writer.close();
		}

	}

	//Creates a ResultItem and put it in writer
	private boolean setResult(DarwinRecord element, int count, ObjectWriter<ResultItem> writer){

		try{
			ResultItem rs = new ResultItem(Integer.toString(element.globalUniqueIdentifier), element.scientificName);

			rs.setAuthor(element.authorYearOfScientificName);
			rs.setCitation(element.remarks);
			rs.setCitation(element.collector);

			Calendar now = Calendar.getInstance();
			SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
			rs.setCredits(credits.replace("XDATEX", format.format(now.getTime())));
			rs.setCitation(citation.replace("XDATEX", format.format(now.getTime())));

			DataSet dataSet = new DataSet(element.collectionCode);			
			dataSet.setName(element.collectionCode);		
			rs.setDataSet(dataSet);

			DataProvider dp = new DataProvider(element.institutionCode);
			dp.setName(element.institutionCode);
			dataSet.setDataProvider(dp);

			//Capabilities
			Product prod = new Product(Capabilities.Occurences, "http://rs.tdwg.org/dwc/dwcore/ScientificName%20equals%20%22" + (element.scientificName).replace(" ", "%20") + "%22%20and%20http://rs.tdwg.org/dwc/dwcore/CollectionCode%20equals%20%22" + element.collectionCode + "%22");
			//		System.out.println("***********");
			//		System.out.println("http://rs.tdwg.org/dwc/dwcore/ScientificName%20like%20%22" + (element.scientificName).replace(" ", "%20") + "%22%20and%20http://rs.tdwg.org/dwc/dwcore/CollectionCode%20like%20%22" + element.collectionCode + "%22");
			//		System.out.println("***********");
			prod.setCount(count);
			rs.setProducts(Collections.singletonList(prod));

			//		rs.setProvider("Species Link");

			//find taxonomy
			Taxon last = null;

			if (element.kingdom != ""){	
				if ((element.kingdom).equals(element.scientificName))
					rs.setRank("kingdom");
				else{
					Taxon k = new Taxon(element.kingdom);  
					k.setRank("kingdom");
					k.setScientificName(element.kingdom);   				
					last = k;
				}
			}

			if (element.phylum != ""){	
				if ((element.phylum).equals(element.scientificName))
					rs.setRank("phylum");
				else{
					Taxon p = new Taxon(element.phylum);
					p.setRank("phylum");
					p.setScientificName(element.phylum);    				   
					if (last != null)
						p.setParent(last);
					last = p;	
				}
			}

			if (element.clazz != ""){	
				if ((element.clazz).equals(element.scientificName))
					rs.setRank("class");
				else{
					Taxon c = new Taxon(element.clazz);
					c.setRank("class");
					c.setScientificName(element.clazz);
					if (last != null)
						c.setParent(last);
					last = c;  
				}
			}

			if (element.order != ""){	
				if ((element.order).equals(element.scientificName))
					rs.setRank("order");
				else{
					Taxon o = new Taxon(element.order);
					o.setRank("order");
					o.setScientificName(element.order);
					if (last != null)
						o.setParent(last);
					last = o;   
				}
			}

			if (element.family != ""){	
				if ((element.family).equals(element.scientificName))
					rs.setRank("family");
				else
				{
					Taxon f = new Taxon(element.family);   
					f.setRank("family");
					f.setScientificName(element.family);  
					if (last != null)
						f.setParent(last);
					last = f;
				}
			}

			if (element.genus != ""){	
				if ((element.genus).equals(element.scientificName))
					rs.setRank("genus");
				else{
					Taxon g = new Taxon(element.genus);  
					g.setRank("genus");
					g.setScientificName(element.genus);    				
					if (last != null)   				
						g.setParent(last);
					last = g;   
				}
			}

			//		logger.trace(last);
			if (rs.getRank() == null)
				rs.setRank("species");

			if (last != null)
				rs.setParent(last);  

			rs.setCredits(Utils.credits());
			rs.setCitation(Utils.citation());

			if (writer.isAlive())
				writer.write(rs);
			
//			logger.trace(rs);

		}catch (Exception e) {
			return false;
		}		

		return true;

	}


	@Override
	public RepositoryInfo getRepositoryInfo() {
		RepositoryInfo info = new RepositoryInfo(
				"http://splink.cria.org.br/images/logo_peq.gif", 
				"http://splink.cria.org.br/",
				"The goal of the speciesLink network is to integrate specie and specimen data available in natural history museums, herbaria and culture collections, making it openly and freely available on the Internet.");
		return info;
	}



}
