package org.gcube.data.transfer.plugins.sis;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.gcube.data.transfer.model.PluginInvocation;
import org.gcube.data.transfer.plugin.AbstractPluginFactory;
import org.gcube.data.transfer.plugin.fails.ParameterException;
import org.gcube.data.transfer.plugin.fails.PluginInitializationException;
import org.gcube.data.transfer.plugin.fails.PluginShutDownException;

import lombok.extern.slf4j.Slf4j;

@Slf4j
public class SISPluginFactory extends AbstractPluginFactory<SisPlugin> {

	static final String PLUGIN_ID="SIS/GEOTK";

	public static final String SOURCE_PARAMETER="SOURCE_FILE";
	public static final String GEONETWORK_CATEGORY="GEONETWORK_CATEGORY";
	public static final String GEONETWORK_STYLESHEET="GEONETWORK_STYLESHEET";

	static final Map<String,String> PARAMETERS_DESCRIPTION= new HashMap<String,String>();

	static{
		PARAMETERS_DESCRIPTION.put(SOURCE_PARAMETER, "[String value] Absolute path of source archive file.");
		PARAMETERS_DESCRIPTION.put(GEONETWORK_CATEGORY, "[String value] GeoNetwork category for publiehd metadata. Default is 'Dataset'.");
		PARAMETERS_DESCRIPTION.put(GEONETWORK_STYLESHEET, "[String value] GeoNetwork stylesheet for publiehd metadata. Default is '_none_'.");
	}


	public SISPluginFactory() {

	}

	@Override
	public void checkInvocation(PluginInvocation arg0) throws ParameterException {
		log.debug("Checking parameters for {} ",arg0);
		Map<String,String> params=arg0.getParameters();
		if(params==null||params.isEmpty()||(!params.containsKey(SOURCE_PARAMETER)))
			throw new ParameterException(SOURCE_PARAMETER+" is mandatory");
		String source=params.get(SOURCE_PARAMETER);
		if(source==null||source.length()==0) throw new ParameterException(SOURCE_PARAMETER+" cannot be null.");
		try{
			File f=new File(source);
			if(!f.exists()) throw new ParameterException(SOURCE_PARAMETER+" ["+source+"] not found.");
			if(!f.canRead()) throw new ParameterException("Cannot read "+SOURCE_PARAMETER+" ["+source+"].");
		}catch(ParameterException e){
			throw e;
		}catch(Exception e){
			throw new ParameterException("Unable to access source file ",e);
		}
	}

	@Override
	public SisPlugin createWorker(PluginInvocation arg0) {
		return new SisPlugin(arg0);
	}

	@Override
	public String getDescription() {
		return String.format("Extracts ISO metadata file from <%s> and publishes to GeoNetwork.", SOURCE_PARAMETER);
	}

	@Override
	public String getID() {
		return PLUGIN_ID;
	}

	@Override
	public Map<String,String> getParameters() {
		return PARAMETERS_DESCRIPTION;
	}

	@Override
	public boolean init() throws PluginInitializationException {
		return true;
	}

	@Override
	public boolean shutDown() throws PluginShutDownException {
		return true;
	}

}
