/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.importer.seplugin;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Calendar;

import org.gcube.dataanalysis.copernicus.cmems.importer.api.ImportOptions;
import org.gcube.dataanalysis.copernicus.cmems.importer.client.TaskManagerClient;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.Execution;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.ExecutionReport;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.ExecutionStatus;
import org.gcube.dataanalysis.datasetimporter.util.ISClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This is an helper class to update (publish) the information about the
 * execution. This class does not update the task; just the execution progress
 * and status, and its reports.
 * 
 * @author Paolo Fabriani
 *
 */
public class ExecutionTracker {

    private static Logger logger = LoggerFactory.getLogger(ExecutionTracker.class);
    
    /**
     * Tha id of the task this execution refers.
     */
    private String taskId;
    
    /**
     * The execution being updated.
     */
    private Execution execution;
    
    private TaskManagerClient client;
    
    public ExecutionTracker(String taskId, String executionId) {
        this.taskId = taskId;
        this.execution = new Execution();
        this.execution.setId(executionId);
    }
    
    public void checkTaskExists() throws Exception {
        try {
            this.getClient().getTask(this.taskId);
        } catch(Exception e) {
            throw new Exception("No such task: " + this.taskId);
        }
    }

//    public void taskInit(ImportOptions options) {
//        try {
//            this.getClient().addTask(options);
//        } catch(Exception e) {
//            logger.error(e.toString());
//            e.printStackTrace();
//        }
//    }
    
    public void executionInit() {
        this.execution.setBegin(Calendar.getInstance());
        this.execution.setProgress(0d);
        this.execution.setStatus(ExecutionStatus.CREATED);
        try {
            this.getClient().addExecution(this.taskId, this.execution);
        } catch(Exception e) {
            logger.error(e.toString());
            e.printStackTrace();
        }
    }
    
    public void executionProgress(int stepsCompleted, int stepsTotal) {
        // round progress to 3 decimal digits (e.g. 0.345 = 34.5%)
        Double progress = (double)stepsCompleted/(double)stepsTotal;
        BigDecimal bd = new BigDecimal(Double.toString(progress));
        bd = bd.setScale(3, RoundingMode.HALF_UP);
        progress = bd.doubleValue();        
        
        // update the execution
        this.execution.setProgress(progress);
        this.execution.setStatus(ExecutionStatus.RUNNING);
        try {
            this.getClient().updateExecution(taskId, this.execution);
        } catch(Exception e) {
            logger.error(e.toString());
            e.printStackTrace();
        }
    }
    
    public void executionComplete() {
        this.execution.setProgress(1d);
        this.execution.setStatus(ExecutionStatus.DONE);
        this.execution.setEnd(Calendar.getInstance());
        try {
            this.getClient().updateExecution(taskId, this.execution);
        } catch(Exception e) {
            logger.error(e.toString());
            e.printStackTrace();
        }
    }
    
    public void executionError() {
        this.execution.setStatus(ExecutionStatus.FAILED);
        this.execution.setEnd(Calendar.getInstance());
        try {
            this.getClient().updateExecution(taskId, this.execution);
        } catch(Exception e) {
            logger.error(e.toString());
            e.printStackTrace();
        }
    }
    
    public void executionReport(ExecutionReport report) {
        try {
            this.getClient().addReport(taskId, this.execution.getId(), report);
        } catch(Exception e) {
            logger.error(e.toString());
            e.printStackTrace();
        }
    }

    private TaskManagerClient getClient() throws MalformedURLException {
        if(this.client==null) {
            this.client = this.buildTaskManagerClient();
        }
        return this.client;
    }
    
    private TaskManagerClient buildTaskManagerClient() throws MalformedURLException {
        URL endpoint = new ISClient().getCmemsImporterEndpoint();
        endpoint = new URL(endpoint, "cmems-importer-service/api");
        logger.info("using task tracker at " + endpoint);
        TaskManagerClient client = new TaskManagerClient(endpoint);
        return client;
    }
    

}
