package org.gcube.couchbase.helpers;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URL;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.couchbase.client.ClusterManager;
import com.google.gson.Gson;

/**
 * 
 * @author Alex Antoniadis
 * 
 */
public class CouchBaseRestHelper {
	
	private static final Logger logger = LoggerFactory.getLogger(CouchBaseRestHelper.class);
	private static Gson gson = new Gson();

	private static final Integer DEFAULT_RAM_QUOTA = 256;
	private static final Integer DEFAULT_PROXY_PORT = 11215;

	private static final String BUCKETS_URI = "/pools/default/buckets";
	private static final String FAILOVER_URI = "/controller/failOver";
	private static final String EJECTNODE_URI = "/controller/ejectNode";
	private static final String ADDNODE_URI = "/controller/addNode";
	private static final String REBALANCE_URI = "/controller/rebalance";
	private static final String REBALANCE_PROGRESS_URI = "/pools/default/rebalanceProgress";
	private static final String CLUSTER_STATS_URI = "/pools/default";
	private static final String NODE_LIST_DELIM = ","; // needs to be comma without any spaces

	enum Status {
		NONE, RUNNING, UNKNOWN,
	}

	public static void createBucket(String host, String port, String username, String password, String bucketName,
			Integer replicaNumber) throws Exception {
		createBucket(host, port, username, password, bucketName, DEFAULT_RAM_QUOTA, replicaNumber, DEFAULT_PROXY_PORT);
	}

	public static void createBucket(String host, String port, String username, String password, String bucketName,
			Integer ramQuota, Integer replicaNumber, Integer proxyPort) throws Exception {

		logger.info("Creating a bucket with name : " + bucketName + " at " + host + ":" + port);

		String authType = "none";

		HttpURLConnection conn = createHttpConnection("http://" + host + ":" + port + BUCKETS_URI, "POST", username,
				password);

		Writer writer = new OutputStreamWriter(conn.getOutputStream(), "UTF-8");

		Map<String, String> params = new HashMap<String, String>();
		params.put("name", bucketName);
		params.put("ramQuotaMB", String.valueOf(ramQuota));
		params.put("authType", authType);
		params.put("replicaNumber", String.valueOf(replicaNumber));
		params.put("proxyPort", String.valueOf(proxyPort));

		addParamsToWriter(writer, params);

		writer.flush();
		writer.close();

		int responseCode = conn.getResponseCode();
		String responseMsg = conn.getResponseMessage();

		conn.disconnect();

		if (responseCode != 200 && responseCode != 202)
			throw new Exception("Error creating the bucket. Error code : " + responseCode + ". Error msg : "
					+ responseMsg);
	}

	public static void deleteBucket(String host, String port, String username, String password, String bucketName)
			throws Exception {

		logger.info("Delete a bucket with name : " + bucketName + " at " + host + ":" + port);

		HttpURLConnection conn = createHttpConnection("http://" + host + ":" + port + BUCKETS_URI + "/" + bucketName,
				"DELETE", username, password);

		int responseCode = conn.getResponseCode();
		String responseMsg = conn.getResponseMessage();

		conn.disconnect();

		
		//System.out.println(responseCode);
		//System.out.println(responseMsg);
		if (responseCode != 200)
			throw new Exception("Error deleting the bucket. Error code : " + responseCode + ". Error msg : "
					+ responseMsg);
	}
	
	public static void deleteBucketSDK(String host, String port, String username, String password, String bucketName)
			throws Exception {

		logger.info("Delete a bucket with name : " + bucketName + " at " + host + ":" + port);

		ClusterManager manager = null;
		try {
			List<URI> uris = new LinkedList<URI>();
			uris.add(URI.create("http://" + host + ":" + port +"/pools"));
			manager = new ClusterManager(uris, username, password);
			manager.deleteBucket(bucketName);
		 
		} catch (Exception e) {
			logger.error("Error deleting the bucket.",e);
			//TODO: just until moving to version 2.1 we will ignore the error and check if the bucket exists after the deletion
			//throw new Exception("Error deleting the bucket.",e);
		} finally {
			if (manager != null)
				manager.shutdown();
		}
	}
	
	

	public static void failOverNode(String host, String port, String username, String password,
			String nodeNamespacedAddress) throws Exception {
		HttpURLConnection conn = createHttpConnection("http://" + host + ":" + port + FAILOVER_URI, "POST", username,
				password);

		Writer writer = new OutputStreamWriter(conn.getOutputStream(), "UTF-8");

		Map<String, String> params = new HashMap<String, String>();
		params.put("otpNode", nodeNamespacedAddress);

		addParamsToWriter(writer, params);

		writer.flush();
		writer.close();

		int responseCode = conn.getResponseCode();
		String responseMsg = conn.getResponseMessage();

		conn.disconnect();

		if (responseCode != 200)
			throw new Exception("Error on failover the node : " + nodeNamespacedAddress + ". Error code : "
					+ responseCode + ". Error msg : " + responseMsg);
	}

	public static void ejectNode(String host, String port, String username, String password,
			String nodeNamespacedAddress) throws Exception {
		HttpURLConnection conn = createHttpConnection("http://" + host + ":" + port + EJECTNODE_URI, "POST", username,
				password);

		Writer writer = new OutputStreamWriter(conn.getOutputStream(), "UTF-8");

		Map<String, String> params = new HashMap<String, String>();
		params.put("otpNode", nodeNamespacedAddress);

		addParamsToWriter(writer, params);

		writer.flush();
		writer.close();

		int responseCode = conn.getResponseCode();
		String responseMsg = conn.getResponseMessage();

		conn.disconnect();

		if (responseCode != 200)
			throw new Exception("Error on failover the node : " + nodeNamespacedAddress + ". Error code : "
					+ responseCode + ". Error msg : " + responseMsg);
	}

	public static String addNode(String host, String port, String username, String password, String nodeIP,
			String nodeUsername, String nodePassword) throws Exception {
		HttpURLConnection conn = createHttpConnection("http://" + host + ":" + port + ADDNODE_URI, "POST", username,
				password);

		Writer writer = new OutputStreamWriter(conn.getOutputStream(), "UTF-8");

		Map<String, String> params = new HashMap<String, String>();
		params.put("hostname", nodeIP);
		params.put("user", nodeUsername);
		params.put("password", nodePassword);

		addParamsToWriter(writer, params);

		writer.flush();
		writer.close();

		int responseCode = conn.getResponseCode();
		String responseMsg = conn.getResponseMessage();

		InputStream in = conn.getInputStream();
		String encoding = conn.getContentEncoding();
		encoding = encoding == null ? "UTF-8" : encoding;
		String body = IOUtils.toString(in, encoding);
		conn.disconnect();
		in.close();

		logger.info("Response : " + responseCode + " . " + responseMsg);

		if (responseCode != 200)
			throw new Exception("Error on adding the node : " + nodeIP + ". Error code : " + responseCode
					+ ". Error msg : " + responseMsg);

		@SuppressWarnings("unchecked")
		Map<String, String> response = gson.fromJson(body, Map.class);
		String nodeAddress = response.get("otpNode");

		logger.info("NodeAddress : " + nodeAddress);

		return nodeAddress;
	}

	public static void removeNode(String host, String port, String username, String password,
			String nodeNamespacedAddress) throws Exception {
		List<String> nodes = getClusterNodes(host, port, username, password);
		
		rebalanceCluster(host, port, username, password, nodes, Arrays.asList(nodeNamespacedAddress));
	}

	public static void removeNode(String host, String port, String username, String password, List<String> nodes,
			String nodeNamespacedAddress) throws Exception {
		rebalanceCluster(host, port, username, password, nodes, Arrays.asList(nodeNamespacedAddress));
	}

	public static void rebalanceCluster(String host, String port, String username, String password) throws Exception {
		List<String> nodes = getClusterNodes(host, port, username, password);
		rebalanceCluster(host, port, username, password, nodes, null);
	}

	public static void rebalanceCluster(String host, String port, String username, String password, List<String> nodes)
			throws Exception {
		rebalanceCluster(host, port, username, password, nodes, null);
	}

	public static void rebalanceCluster(String host, String port, String username, String password, List<String> nodes,
			List<String> ejectedNodes) throws Exception {
		logger.info("rebalancing the nodes : " + nodes + " . ejected are : " + ejectedNodes);
		
//		nodes.removeAll(ejectedNodes);
//		
//		if (nodes.size() == 0) {
//			logger.warn("No nodes left to rebalace");
//		}

		HttpURLConnection conn = createHttpConnection("http://" + host + ":" + port + REBALANCE_URI, "POST", username,
				password);

		Writer writer = new OutputStreamWriter(conn.getOutputStream(), "UTF-8");

		Map<String, String> params = new HashMap<String, String>();
		String nodesParam = StringUtils.join(nodes, NODE_LIST_DELIM);
		String ejectedNodesParam = ejectedNodes != null ? StringUtils.join(ejectedNodes, NODE_LIST_DELIM) : "";

		params.put("ejectedNodes", ejectedNodesParam);
		params.put("knownNodes", nodesParam);

		addParamsToWriter(writer, params);

		writer.flush();
		writer.close();

		int responseCode = conn.getResponseCode();
		String responseMsg = conn.getResponseMessage();

		conn.disconnect();

		logger.info("Response : " + responseCode + " . " + responseMsg);

		if (responseCode != 200)
			throw new Exception("Error on rebalancing the known nodes : " + nodesParam + " , ejected : "
					+ ejectedNodesParam + ". Error code : " + responseCode + ". Error msg : " + responseMsg);

	}

	public static Status getRebalanceProgress(String host, String port, String username, String password)
			throws Exception {
		logger.info("rebalance progress");

		HttpURLConnection conn = createHttpConnection("http://" + host + ":" + port + REBALANCE_PROGRESS_URI, "GET",
				username, password);

		int responseCode = conn.getResponseCode();
		String responseMsg = conn.getResponseMessage();

		InputStream in = conn.getInputStream();
		String encoding = conn.getContentEncoding();
		encoding = encoding == null ? "UTF-8" : encoding;
		String body = IOUtils.toString(in, encoding);
		conn.disconnect();
		in.close();

		logger.info("Response : " + responseCode + " . " + responseMsg);

		if (responseCode != 200)
			throw new Exception("Error getting the rebalance progress" + ". Error code : " + responseCode
					+ ". Error msg : " + responseMsg);

		@SuppressWarnings("unchecked")
		Map<String, String> response = gson.fromJson(body, Map.class);
		String status = response.get("status");

		if (status.equalsIgnoreCase("none")) {
			return Status.NONE;
		} else if (status.equalsIgnoreCase("running")) {
			return Status.RUNNING;
		} else {
			return Status.UNKNOWN;
		}
	}

	public static List<String> getClusterNodes(String host, String port, String username, String password)
			throws Exception {
		logger.info("rebalance progress");

		HttpURLConnection conn = createHttpConnection("http://" + host + ":" + port + CLUSTER_STATS_URI, "GET",
				username, password);

		int responseCode = conn.getResponseCode();
		String responseMsg = conn.getResponseMessage();

		InputStream in = conn.getInputStream();
		String encoding = conn.getContentEncoding();
		encoding = encoding == null ? "UTF-8" : encoding;
		String body = IOUtils.toString(in, encoding);
		conn.disconnect();
		in.close();

		logger.info("Response : " + responseCode + " . " + responseMsg);

		if (responseCode != 200)
			throw new Exception("Error getting the rebalance progress" + ". Error code : " + responseCode
					+ ". Error msg : " + responseMsg);

		List<String> nodeAddresses = new ArrayList<String>();

		@SuppressWarnings("unchecked")
		Map<String, List<Map<String, String>>> response = gson.fromJson(body, HashMap.class);
		List<Map<String, String>> nodes = response.get("nodes");
		for (Map<String, String> node : nodes) {
			String nodeAddr = node.get("otpNode");
			logger.info("found node with addr : " + nodeAddr);
			nodeAddresses.add(nodeAddr);
		}
		return nodeAddresses;
	}

	public static Map<String, String> getClusterNodesAddressesAndPorts(String host, String port, String username,
			String password) throws Exception {
		logger.info("get cluster nodes  addresses + ports progress");

		HttpURLConnection conn = createHttpConnection("http://" + host + ":" + port + CLUSTER_STATS_URI, "GET",
				username, password);

		int responseCode = conn.getResponseCode();
		String responseMsg = conn.getResponseMessage();

		InputStream in = conn.getInputStream();
		String encoding = conn.getContentEncoding();
		encoding = encoding == null ? "UTF-8" : encoding;
		String body = IOUtils.toString(in, encoding);
		conn.disconnect();
		in.close();

		logger.info("Response : " + responseCode + " . " + responseMsg);

		if (responseCode != 200)
			throw new Exception("Error getting the rebalance progress" + ". Error code : " + responseCode
					+ ". Error msg : " + responseMsg);

		Map<String, String> nodeAddresses = new HashMap<String, String>();

		@SuppressWarnings("unchecked")
		Map<String, List<Map<String, String>>> response = gson.fromJson(body, HashMap.class);
		List<Map<String, String>> nodes = response.get("nodes");
		for (Map<String, String> node : nodes) {
			String nodeAddr = node.get("hostname");
			logger.info("found node with addr : " + nodeAddr);
			String[] terms = nodeAddr.split(":");

			nodeAddresses.put(terms[0], terms[1]);
		}
		return nodeAddresses;
	}

	public static boolean checkIfBucketExists(String host, String port, String username, String password, String bucketName) {
		try {
			statusBucketsRestfully(host, port, username, password, bucketName);
			return true;
		} catch (Exception e) {
			logger.error("bucket not found", e);
			return false;
		}
	}
	  
	public static void statusBucketsRestfully(String host, String port, String username, String password, String bucketName) throws Exception {
		logger.info("Status a bucket with name : " + bucketName + " at " + host + ":" + port);

		HttpURLConnection conn = createHttpConnection("http://" + host + ":" + port + BUCKETS_URI + "/" + bucketName,
				"GET", username, password);

		int responseCode = conn.getResponseCode();
		String responseMsg = conn.getResponseMessage();

		conn.disconnect();

		if (responseCode != 200)
			throw new Exception("Error deleting the bucket. Error code : " + responseCode + ". Error msg : "
					+ responseMsg);
		
		System.out.println(responseMsg);
		
	}
	 
	// / Helpers

	private static void addBasicHttpAuth(HttpURLConnection conn, String username, String password) {
		String userpassword = username + ":" + password;
		String encodedAuthorization = Base64.encodeBase64String(userpassword.getBytes());
		conn.setRequestProperty("Authorization", "Basic " + encodedAuthorization);
	}

	private static void addParamsToWriter(Writer writer, Map<String, String> params)
			throws UnsupportedEncodingException, IOException {
		logger.info("request will have the following params : " + params);

		for (Entry<String, String> param : params.entrySet()) {
			writer.write(param.getKey());
			writer.write("=");
			writer.write(URLEncoder.encode(param.getValue(), "UTF-8"));
			writer.write("&");
		}
	}

	private static HttpURLConnection createHttpConnection(String urlString, String method) throws IOException {
		logger.info("creating connection at  : " + urlString);
		URL url = new URL(urlString);

		HttpURLConnection conn = (HttpURLConnection) url.openConnection();
		conn.setRequestMethod(method);
		conn.setDoOutput(true);
		conn.setDoInput(true);
		conn.setUseCaches(false);
		conn.setAllowUserInteraction(false);
		conn.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");

		return conn;
	}

	private static HttpURLConnection createHttpConnection(String urlString, String method, String username,
			String password) throws IOException {
		HttpURLConnection conn = createHttpConnection(urlString, method);
		addBasicHttpAuth(conn, username, password);

		return conn;
	}

	/*public static void main(String[] args) throws Exception {
		String host = "195.134.66.179";
		String port = "8091";
		// String host = "localhost";
		// String port = "9999";

		String username = "Administrator";
		String password = "mycouchbase";
		String bucketName = "mybucket";
		Integer replicaNumber = 1;

		
		deleteBucket(host, port, username, password, bucketName);
		//createBucket(host, port, username, password, bucketName, replicaNumber);
		//statusBucketsRestfully(host, port, username, password, bucketName);

		// String nodeIP = "192.168.56.103";

		// addNode(host, port, username, password, nodeIP, username, password);

		// rebalanceCluster(host, port, username, password,
		// Arrays.asList("ns_1@192.168.56.102", "ns_1@192.168.56.103"));

		// Status s = getRebalanceProgress(host, port, username, password);
		// System.out.println(s);
		//
		// getClusterNodes(host, port, username, password);
//		removeNode(host, port, username, password, Arrays.asList("ns_1@192.168.56.103"), "ns_1@192.168.56.103");

	}*/

}
