package org.gcube.textextractor.extractors;

import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.DefaultHttpClient;
import org.gcube.semantic.annotator.AnnotationBase;
import org.gcube.semantic.annotator.utils.ANNOTATIONS;
import org.gcube.textextractor.entities.RestExtractorEntities;
import org.gcube.textextractor.entities.RestExtractorEntities.CriterionObj;
import org.gcube.textextractor.entities.RestExtractorEntities.IntermediateCriterionObj;
import org.gcube.textextractor.entities.RestExtractorEntities.TableDataObj;
import org.gcube.textextractor.entities.RestExtractorEntities.ValueObj;
import org.gcube.textextractor.helpers.ExtractorHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

public class RESTServiceExtractor extends InformationExtractor {
	private static final Logger logger = LoggerFactory.getLogger(RESTServiceExtractor.class);

	private static Gson gson = new Gson();
	
	@Override
	public String convertInfoToRowset(Map<String, String> info) {
		String documentID = info.get("documentID");
		info.remove("documentID");
		
		return ExtractorHelper.createRowseFromFields(documentID, collectionID, idxType, info.get("language"), info);
	}

	@Override
	public Map<String, String> extractFieldsFromFile(String filename)
			throws Exception {
		
		
		String docID = filename;
		Map<String, String> document = Maps.newHashMap();
		
		try {
			List<IntermediateCriterionObj> intermediateCriteria  = getCriteria(filename);
			for (IntermediateCriterionObj intermediateCriterion : intermediateCriteria){
				
				String fieldNamePrefix = null;
				
				System.out.println("-----> " + intermediateCriterion.criterionName);
				
				if (	intermediateCriterion.criterionName.equalsIgnoreCase("SPECIES") ||
						intermediateCriterion.criterionName.equalsIgnoreCase("especes") ||
						intermediateCriterion.criterionName.equalsIgnoreCase("espece"))
					fieldNamePrefix = ANNOTATIONS.getLocalName(ANNOTATIONS.SPECIES);
				else if (intermediateCriterion.criterionName.equalsIgnoreCase("GEARS") ||
						intermediateCriterion.criterionName.equalsIgnoreCase("GEAR") ||
						intermediateCriterion.criterionName.equalsIgnoreCase("GEAR 1") ||
						intermediateCriterion.criterionName.equalsIgnoreCase("GEARS 1") ||
						intermediateCriterion.criterionName.equalsIgnoreCase("FISHING GEARS"))
					fieldNamePrefix = ANNOTATIONS.getLocalName(ANNOTATIONS.GEAR);
				else if (intermediateCriterion.criterionName.equalsIgnoreCase("COUNTRY") ||
						intermediateCriterion.criterionName.equalsIgnoreCase("PAY") || 
						intermediateCriterion.criterionName.equalsIgnoreCase("PAYS")
						)
					fieldNamePrefix = ANNOTATIONS.getLocalName(ANNOTATIONS.COUNTRY);
				else if (intermediateCriterion.criterionName.equalsIgnoreCase("BOAT") || 
						intermediateCriterion.criterionName.equalsIgnoreCase("BOATS") ||
						intermediateCriterion.criterionName.equalsIgnoreCase("VESSEL") ||
						intermediateCriterion.criterionName.equalsIgnoreCase("VESSELS") 
						)
					fieldNamePrefix = ANNOTATIONS.getLocalName(ANNOTATIONS.VESSEL);
				else if (intermediateCriterion.criterionName.equalsIgnoreCase("MAR-ZONE") || 
						intermediateCriterion.criterionName.equalsIgnoreCase("MAR-ZONE 1") ||
						intermediateCriterion.criterionName.equalsIgnoreCase("MAR-ZONE 2"))
					fieldNamePrefix = ANNOTATIONS.getLocalName(ANNOTATIONS.WATER_AREA);
				else if (intermediateCriterion.criterionName.equalsIgnoreCase("TER-ZONE") || 
						intermediateCriterion.criterionName.equalsIgnoreCase("TER-ZONE 1"))
					fieldNamePrefix = ANNOTATIONS.getLocalName(ANNOTATIONS.LAND_AREA);
				
				
				if (fieldNamePrefix == null) {
					logger.warn("####### no field matched for : " + intermediateCriterion.criterionName);
					continue;
				}
				//String fieldNamePrefix = kebabCaseToSnakeCase(intermediateCriterion.criterionName);
				
				
				
				
				
				String criterionUrl = docID + "/criteria/" + String.valueOf(intermediateCriterion.criterionId);
				document.put("documentID", docID);
				
				try {
					CriterionObj criterion = getCriterionValues(criterionUrl);
					
					
					for (ValueObj value : criterion.values){
						
						String frLabelName = fieldNamePrefix + "_" + "frLabel";
						String frLabelValue = value.frLabel  == null ? "" : value.frLabel;
						
						String enLabelName = fieldNamePrefix + "_" + "enLabel";
						String enLabelValue = value.enLabel == null ? "" : value.enLabel;
						
						String enDescriptionName = fieldNamePrefix + "_" + "enDescription";
						String enDescriptionValue = value.enDescription == null ? "" : value.enDescription;
						
						String frDescriptionName = fieldNamePrefix + "_" + "frDescription";
						String frDescriptionValue = value.frDescription  == null ? "" : value.frDescription;
						
						
						
						document.put(enLabelName, enLabelValue);
						document.put(frLabelName, frLabelValue);
						
						document.put(enDescriptionName, enDescriptionValue);
						document.put(frDescriptionName, frDescriptionValue);
						
					}
					
					logger.info("criterion values : " + criterion.values);
					logger.info("doc : " + document);
				} catch (Exception e1) {
					logger.error("error while getting criteriaValues for : " + criterionUrl, e1);
				} 
		} 
		} catch (Exception e) {
			logger.error("error while getting the criteria for : " + filename, e);
			return document;
		}
		
		
		return document;
	}

	@Override
	public Map<String, String> enrichRecord(Map<String, String> record,
			String filename) {
		
		Map<String, String> enrichedRecord = Maps.newHashMap();
		Map<String, List<String>> uris = Maps.newHashMap();
		
		String docURI = filename;
		enrichedRecord.put("documentID", docURI);
		enrichedRecord.putAll(record);
		
		ExtractorHelper.enrichListField(record, enrichedRecord, uris, ANNOTATIONS.getLocalName(ANNOTATIONS.SPECIES), new ExtractorHelper.QueryWrapperList() {
			@Override
			public String doCall(List<String> arg) throws Exception {
				return ExtractorHelper.querySpecies(arg);
			}
		});
		
		ExtractorHelper.enrichSimpleField(record, enrichedRecord, uris, ANNOTATIONS.getLocalName(ANNOTATIONS.GEAR), new ExtractorHelper.QueryWrapperSimple() {
			@Override
			public String doCall(String arg) throws Exception {
				return ExtractorHelper.queryGear(arg);
			}
		});
		
		ExtractorHelper.enrichListField(record, enrichedRecord, uris, ANNOTATIONS.getLocalName(ANNOTATIONS.COUNTRY), new ExtractorHelper.QueryWrapperList() {
			@Override
			public String doCall(List<String> arg) throws Exception {
				return ExtractorHelper.queryCountry(arg);
			}
		});
		
		ExtractorHelper.enrichSimpleField(record, enrichedRecord, uris, ANNOTATIONS.getLocalName(ANNOTATIONS.VESSEL), new ExtractorHelper.QueryWrapperSimple() {
			@Override
			public String doCall(String arg) throws Exception {
				return ExtractorHelper.queryVessel(arg);
			}
		});
		
		ExtractorHelper.enrichListField(record, enrichedRecord, uris, ANNOTATIONS.getLocalName(ANNOTATIONS.WATER_AREA), new ExtractorHelper.QueryWrapperList() {
			@Override
			public String doCall(List<String> arg) throws Exception {
				return ExtractorHelper.queryWaterArea(arg);
			}
		});
		
		
		ExtractorHelper.enrichListField(record, enrichedRecord, uris, ANNOTATIONS.getLocalName(ANNOTATIONS.LAND_AREA), new ExtractorHelper.QueryWrapperList() {
			@Override
			public String doCall(List<String> arg) throws Exception {
				return ExtractorHelper.queryLandArea(arg);
			}
		});
		
        try {
            annotate(docURI, uris);
        } catch (Exception ex) {
        	logger.error("file : " + filename + " not found", ex);
        }
		
		return enrichedRecord;
	}
	
	
	static List<String> getDocumentUrls(String path) throws Exception {
		List<String> documentUrls = Lists.newArrayList();
		
		List<String> countriesIds = getCountriesIds(path);
		
		
		for (String countryId : countriesIds){
			try {
				String tableUrl = createUrlForTablesQueries(path, countryId);
				logger.debug("getting tables for url : " + tableUrl);
				List<String> tablesIds = getTablesIds(tableUrl);
				for (String tableId : tablesIds){
					logger.debug("tables : " + tablesIds);
					String tableDataUrl = createUrlForTableDataQueries(path, countryId, tableId);
					logger.debug("tableID : " + tableDataUrl);
					
					documentUrls.add(tableDataUrl);
				}
			} catch (Exception e) {
				
			}
		}
		
		return documentUrls;
	}

	@Override
	public List<Map<String, String>> extractInfo(String path)
			throws FileNotFoundException {
		
		int cnt = 0;

		List<String> documentUrls;
		try {
			documentUrls = getDocumentUrls(path);
		} catch (Exception e) {
			return Lists.newArrayList();
		}
		
		List<Map<String, String>> extractedInfo = Lists.newArrayList();;
		
		for (String documentUrl : documentUrls){
			 logger.info("Processing documentUrl : " + (++cnt) + " " + documentUrl);
			 try {
				 Map<String, String> info = this.extractFieldsFromFile(documentUrl);
				 
				 long part_start_time = System.currentTimeMillis();
                Map<String, String> enriched = enrichRecord(info, documentUrl);
                long part_end_time = System.currentTimeMillis();
				 
                logger.info("~> field enrichment time  : " + (part_end_time - part_start_time)/1000.0 + " secs");
                extractedInfo.add(enriched);
			 } catch (Exception e) {
				logger.error("error while extracting info from : " + documentUrl + " . will skip this documentUrl", e);
			}
		}
		
		return extractedInfo;
	}
	
	
	static List<RestExtractorEntities.CountryObj> getCountries(String url) throws Exception {
		HttpClient client = new DefaultHttpClient();
		
		HttpGet request = new HttpGet(url);
		
		HttpResponse httpResponse;
		httpResponse = client.execute(request);
		try (InputStream responseStream = httpResponse.getEntity().getContent()) {
		
			Map<String, List<RestExtractorEntities.CountryObj>> resp = gson.fromJson(new InputStreamReader(responseStream),
					new TypeToken<Map<String, List<RestExtractorEntities.CountryObj>>>() {
					}.getType());
			
			
			List<RestExtractorEntities.CountryObj> countries = resp.get("countries");
			
			return countries;
		} catch (Exception e) {
			throw e;
		}
	}
	
	
	static List<IntermediateCriterionObj> getCriteria(String url) throws Exception {
		HttpClient client = new DefaultHttpClient();
		
		HttpGet request = new HttpGet(url);
		
		HttpResponse httpResponse;
		httpResponse = client.execute(request);
		try (InputStream responseStream = httpResponse.getEntity().getContent()) {
		
			TableDataObj resp = gson.fromJson(new InputStreamReader(responseStream),
					new TypeToken<TableDataObj>() {
					}.getType());
			
			
			List<IntermediateCriterionObj> criteria = resp.criteria;
			
			return criteria;
		} catch (Exception e) {
			throw e;
		}
	}
	
	static CriterionObj getCriterionValues(String url) throws Exception {
		HttpClient client = new DefaultHttpClient();
		
		HttpGet request = new HttpGet(url);
		
		HttpResponse httpResponse;
		httpResponse = client.execute(request);
		try (InputStream responseStream = httpResponse.getEntity().getContent()) {
		
			CriterionObj resp = gson.fromJson(new InputStreamReader(responseStream),
					new TypeToken<CriterionObj>() {
					}.getType());
			
			
			
			return resp;
		} catch (Exception e) {
			throw e;
		}
	}
	
	static List<String> getCountriesIds(String url) throws Exception {
		List<RestExtractorEntities.CountryObj> countries = getCountries(url);
		
		List<String> ids = Lists.newArrayList();
		for (RestExtractorEntities.CountryObj country : countries){
			ids.add(String.valueOf(country.id));
		}
		
		return ids;
	}
	
	static String createUrlForTablesQueries(String baseUrl, String id){
		return baseUrl + "/" + id;
	}
	
	static List<String> createUrlsForTablesQueries(String baseUrl, List<String> ids){
		List<String> urls = Lists.newArrayList();
		
		for (String id : ids){
			urls.add(baseUrl + "/" + id);
		}
		
		return urls;
	}
	
	static String createUrlForCriteriaQueries(String baseUrl, String countryID, String tableID, String criterionId){
		return baseUrl + "/" + countryID + "/tables/" + tableID + "/criteria/" + criterionId;
	}
	
	static String createUrlForTableDataQueries(String baseUrl, String countryID, String tableID){
		return baseUrl + "/" + countryID + "/tables/" + tableID;
	}
	
	static List<String> createUrlsForTableDataQueries(String baseUrl, String countryID,List<String> ids){
		List<String> urls = Lists.newArrayList();
		
		for (String id : ids){
			urls.add(baseUrl + "/" + countryID + "/tables/" + id);
		}
		
		return urls;
	}
	
	static List<String> getTablesIds(String url) throws Exception {
		
		
		List<String> listings = new ArrayList<String>();
		
		HttpClient client = new DefaultHttpClient();
		HttpGet request = new HttpGet(url);
		
		HttpResponse httpResponse;
		httpResponse = client.execute(request);
		try (InputStream responseStream = httpResponse.getEntity().getContent()) {
		
			Map<String, Object> resp = gson.fromJson(new InputStreamReader(responseStream),
					new TypeToken<Map<String, Object>>() {
					}.getType());
			
			@SuppressWarnings("unchecked")
			List<Map<String, Object>> listingObjs = (List<Map<String, Object>>) resp.get("listing");
			
			for (Map<String, Object> listing : listingObjs) {
				Double idStr = (double)listing.get("id");
				Integer id = idStr.intValue();
				
				listings.add(String.valueOf(id));
			}
			
			
			return listings;
		} catch (Exception e) {
			throw e;
		}
		
	}
	
	public static void main(String[] args) throws Exception {
		RESTServiceExtractor ex = new RESTServiceExtractor();
		ex.extractInfo("http://statbase.pirogprod.info:8080/statbase/rest/country");
		
		
		
		
//		String path = "http://statbase.pirogprod.info:8080/statbase/rest/country";
//		
//		List<String> countriesIds = getCountriesIds(path);
//		
//		//List<String> urls = createUrlsForTablesQueries(path, countriesIds);
//		
//		for (String countryId : countriesIds){
//			try {
//				String tableUrl = createUrlForTablesQueries(path, countryId);
//				System.out.println("getting tables for url : " + tableUrl);
//				List<String> tablesIds = getTablesIds(tableUrl);
//				for (String tableId : tablesIds){
//					System.out.println("tables : " + tablesIds);
//					String tableDataUrl = createUrlForTableDataQueries(path, countryId, tableId);
//					System.out.println("tableID : " + tableDataUrl);
//					
//					
//					String docID = tableDataUrl;
//					
//					List<IntermediateCriterionObj> intermediateCriteria  = getCriteria(tableDataUrl);
//					for (IntermediateCriterionObj intermediateCriterion : intermediateCriteria){
//						
//						Map<String, String> document = Maps.newHashMap();
//						
//						String fieldNamePrefix = kebabCaseToSnakeCase(intermediateCriterion.criterionName);
//						
//						String criterionUrl = createUrlForCriteriaQueries(path, countryId, tableId, String.valueOf(intermediateCriterion.criterionId));
//						
//						CriterionObj criterion = getCriterionValues(criterionUrl);
//						
//						
//						for (ValueObj value : criterion.values){
//							
//							String frLabelName = fieldNamePrefix + "_" + "frLabel";
//							String frLabelValue = value.frLabel;
//							
//							String enLabelName = fieldNamePrefix + "_" + "enLabel";
//							String enLabelValue = value.enLabel;
//							
//							String enDescriptionName = fieldNamePrefix + "_" + "enDescription";
//							String enDescriptionValue = value.enDescription;
//							
//							String frDescriptionName = fieldNamePrefix + "_" + "frDescription";
//							String frDescriptionValue = value.frDescription;
//							
//							
//							document.put("documentID", docID);
//							
//							document.put(enLabelName, enLabelValue);
//							document.put(frLabelName, frLabelValue);
//							
//							document.put(enDescriptionName, enDescriptionValue);
//							document.put(frDescriptionName, frDescriptionValue);
//							
//						}
//						
//						
//						
//						System.out.println(criterion.values);
//						
//						System.out.println("doc : " + document);
//					}
//					
//					
//				}
//				
//				
//				
//			} catch (Exception e) {
//				e.printStackTrace();
//			}
//		}
	}
	
	
	private void annotate(String filename, Map<String, List<String>> uris) throws FileNotFoundException {
        AnnotationBase annotator = AnnotationBase.getInstance();
        Set<Map.Entry<String, List<String>>> entrySet = uris.entrySet();
        for (Map.Entry<String, List<String>> entry : entrySet) {
            if (entry.getKey().equals(ANNOTATIONS.getLocalName(ANNOTATIONS.COUNTRY)+"_uris")) {
                List<String> uris_ = entry.getValue();
                for (String uri_ : uris_) {
                    annotator.STATBASE_country(filename, uri_);
                }
            }
            if (entry.getKey().equals(ANNOTATIONS.getLocalName(ANNOTATIONS.SPECIES)+"_uris")) {
                List<String> uris_ = entry.getValue();
                for (String uri_ : uris_) {
                    annotator.STATBASE_species(filename, uri_);
                }
            }
            if (entry.getKey().equals(ANNOTATIONS.getLocalName(ANNOTATIONS.GEAR)+"_uris")) {
                List<String> uris_ = entry.getValue();
                for (String uri_ : uris_) {
                    annotator.STATBASE_gear(filename, uri_);
                }
            }
            if (entry.getKey().equals(ANNOTATIONS.getLocalName(ANNOTATIONS.VESSEL)+"_uris")) {
                List<String> uris_ = entry.getValue();
                for (String uri_ : uris_) {
                    annotator.STATBASE_vessel(filename, uri_);
                }
            }
        }
    }
	
	static String kebabCaseToCamelCase(String str){
		str = str.replace(" ", "-");
		
		List<String> w = Splitter.on("-").splitToList(str.toLowerCase());
		StringBuilder camelCase = new StringBuilder();
		
		camelCase.append(w.get(0));
		for (int i = 1 ; i < w.size() ; ++i){
			camelCase.append(capitalizeFirst(w.get(i)));
		}
		
		return camelCase.toString();
	}
	
	static String kebabCaseToSnakeCase(String str){
		str = str.toLowerCase().replace(" ", "-");
		
		return str;
	}
	
	static String capitalizeFirst(String input){
		String output = input.substring(0, 1).toUpperCase() + input.substring(1);
		return output;
	}
	
	
}
