package org.gcube.indexmanagement.storagehandling;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.LinkedList;
import java.util.zip.Adler32;
import java.util.zip.CheckedOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import org.gcube.common.core.contexts.GCUBEServiceContext;
import org.gcube.common.core.informationsystem.notifier.ISNotifier.NotificationEvent;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.contentmanagement.blobstorage.service.IClient;
import org.gcube.contentmanager.storageclient.wrapper.AccessType;
import org.gcube.contentmanager.storageclient.wrapper.StorageClient;
//import org.gcube.contentmanagement.gcubedocumentlibrary.io.DocumentWriter;
//import org.gcube.contentmanagement.gcubemodellibrary.elements.GCubeDocument;
import org.gcube.indexmanagement.common.IndexException;
import org.gcube.indexmanagement.common.IndexNotificationConsumer;
import org.gcube.indexmanagement.common.IndexUpdaterWSResource;
import org.gcube.indexmanagement.storagehandling.stubs.ConnectUpdaterResponse;
import org.gcube.indexmanagement.storagehandling.stubs.DeltaActionType;
import org.gcube.indexmanagement.storagehandling.stubs.DeltaFileInfoType;
import org.globus.wsrf.core.notification.SubscriptionManager;
import org.oasis.wsrf.lifetime.Destroy;

/**
 * This class implements an uploader of delta files to the Content Management layer. Each time
 * a new delta file is created as a result of new data added to (or removed from) the index,
 * the file is added to the uploader queue in order to be stored in the CMS. After the file
 * finally gets uploaded, a notification is sent to the index management resource, so that the
 * delta file list is updated.
 */
public class DeltaFileUploader {

	 /**
      * Logger
      */
    private static GCUBELog logger = new GCUBELog(DeltaFileUploader.class);
/*    private static SimpleCredentialsListener credentialsListener;
    static{
        try{
            credentialsListener = new SimpleCredentialsListener();
            logger = new GCUBELog(DeltaFileUploader.class);
            DelegationLocalInterface.registerCredentialsListener(credentialsListener);
            logger.debug("DHN secure. Credential listener created and registered.");
            
            ExtendedGSSCredential cred = credentialsListener.getCredentials();
            if(cred == null){
                logger.debug("No credentials currently available");
            }
            else{
                logger.debug("Credentials available. Distinguished name: " + cred.getName());
            }
        }
        catch(Exception e){
            logger.error("Unable to create CredentialsListener:", e);
        }
    }*/
    
    /**
     * The wrapper for the delta list management interface for this index
     */
    private DeltaListManagementWrapper manager;
    
    /**
     * The current connection ID
     */
    private int connectionID;
    
    /**
     * Die automatically after the pending uploads are done?
     */
    private volatile boolean doDieAfterCurrentUploads = false;
    
    /**
     * The ID of the collection holding the delta files for this index
     */
    private String deltaFileCollectionID = null; 
    
    /**
     * The delta file upload queue
     */
    private LinkedList<DeltaFileInfoType> uploadQueue;
    
    /**
     * The notification consumer used by this DeltaFileUploader
     */
    private IndexNotificationConsumer notificationConsumer;
    
    /**
     * The SubscriptionManager used for index removal notifications
     */
    private SubscriptionManager removalSubscription;
    
    /**
     * The ID of the index which this uploader feeds with data
     */
    private String indexID;
    
    /**
     * The index updater resource that uses this uploader
     */
    private IndexUpdaterWSResource resource;
    
    /**
     * Constructs a new DeltaFileUploader
     * @param indexID the ID of the index that this DeltaFileUploader feeds
     * @param resource the index updater resource that uses this DeltaFileUploader
     * @param ctx the service context used for security and scoping
     * @throws Exception an error occured
     */
    public DeltaFileUploader(String indexID, IndexUpdaterWSResource resource, GCUBEServiceContext ctx) throws Exception {
        this(resource, new RemoteDeltaListManager(indexID, ctx, resource.getManagementResourceNamespace()), indexID, ctx);
    }
    
    /**
     * Constructs a new DeltaFileUploader
     * @param resource the index updater resource that uses this DeltaFileUploader
     * @param manager the delta list manager
     * @param indexID the ID of the index that this DeltaFileUploader feeds
     * @param ctx the service context used for security and scoping
     * @throws Exception
     */
    private DeltaFileUploader(IndexUpdaterWSResource resource, DeltaListManagementWrapper manager, String indexID, GCUBEServiceContext ctx) throws Exception {
    	logger.debug(">>>>> Constructing DeltaFileUploader");
    	
        this.manager = manager;
        this.indexID = indexID;
        this.resource = resource;
        this.uploadQueue = new LinkedList<DeltaFileInfoType>();
        
        ConnectUpdaterResponse connectResponse = manager.connectUpdater();
        this.deltaFileCollectionID = connectResponse.getDeltaFileCollectionID();
        
        subscribeForIndexRemoval(ctx.getScope());

        this.connectionID = connectResponse.getConnectionID();
        
        ctx.setScope(cmsUploader, ctx.getScope());
        ctx.useCredentials(cmsUploader, ctx.getCredentials());
        this.cmsUploader.start();
        
        logger.debug("<<<<< Constructing DeltaFileUploader");
    }
    
    /**
     * Thread used to zip, upload and call the GeneratorService.merge() method (uploadList consumer)
     */
    Thread cmsUploader = new Thread(
            new Runnable(){
                public void run(){
                    String deltaPath = null;
                    DeltaFileInfoType deltaInfo = null;
                    File deltaFile = null;
                    File zipFile = null;
                    String documentID = null;
               
                    while (true){
                        synchronized(uploadQueue){
                            while(uploadQueue.size() == 0){
                                try{
                                    if(!doDieAfterCurrentUploads){
                                        //--Check if interrupt's been called to stop the thread--
                                        Thread.yield();
                                        if (Thread.currentThread().isInterrupted()) {
                                            logger.debug("Uploader thread cancelled after cms upload.");
                                            cleanUp(documentID, deltaFile, zipFile);
                                            shutdown();
                                            return;
                                        }
                                        uploadQueue.wait();
                                    }
                                    else {
                                        logger.debug("Uploader thread done");
                                        shutdown();
                                        return;
                                    }
                                }
                                catch(InterruptedException ie){
                                    logger.debug("Uploader thread cancelled while waiting");
                                    cleanUp(documentID, deltaFile, zipFile);
                                    shutdown();
                                    return;
                                }
                            }
                            deltaInfo = uploadQueue.removeLast();
                            deltaPath = deltaInfo.getDeltaFileID();
                            uploadQueue.notifyAll();
                        }
                        try{                            
                            deltaFile = new File(deltaPath);
                            
                            logger.debug("zipping: " + deltaPath + " of size: " + deltaFile.length());
                            zipFile = zip(deltaFile);

                            //--Check if interrupt's been called to stop the thread--
                            Thread.yield();
                            if (Thread.currentThread().isInterrupted()) {
                                logger.debug("Uploader thread cancelled after zipping.");
                                shutdown();
                                return;
                            }
                            
                            logger.debug("uploading: " + zipFile.getAbsolutePath() + " of size: " + zipFile.length());
                            documentID = cmsUpload(zipFile);
                            if (!removeFileOrDir(deltaFile) ) {
                                logger.error("Unable to completely delete deltaFile: " + deltaPath);
                            }
                            if (!zipFile.delete()) {
                                logger.error("Unable to completely delete zipFile: " + zipFile.getAbsolutePath());
                            }
                            
                            //change the id to the one used received from cms
                            deltaInfo = new DeltaFileInfoType(deltaInfo.getDeltaAction(), documentID, deltaInfo.getDocumentCount(), deltaInfo.getIndexTypeID());
                            
                            //--Check if interrupt's been called to stop the thread--
                            Thread.yield();
                            if (Thread.currentThread().isInterrupted()) {
                                logger.debug("Uploader thread cancelled after cms upload.");
                                cleanUp(documentID, deltaFile, zipFile);
                                shutdown();
                                return;
                            }
                            manager.mergeDeltaFile(deltaInfo);
                        }
                        catch(Exception e) {
                            logger.error("Uploading and merging the delta file \"" + deltaPath + "\" failed.", e);
                        }
                    }
                }
            }
    );
    
    /**
     * Uploads a delta file to the CMS
     * @param deltaFileName the filename of the delta file to upload
     * @param indexTypeID the index type ID
     * @param documentCount the number of documents in the delta file to be uploaded
     */
    public void upload(String deltaFileName, String indexTypeID, int documentCount) {
        logger.debug("adding to upload queue: " + deltaFileName + " of size: " + new File(deltaFileName).length());
        upload(deltaFileName, DeltaActionType.Addition, indexTypeID, documentCount);
    }
    
    /**
     * Uploads a delta file to the CMS
     * @param deltaFileName the filename of the delta file to upload
     * @param action the delta file action: addition or deletion
     * @param indexTypeID the index type ID
     * @param documentCount the number of documents in the delta file to be uploaded
     */
    public void upload(String deltaFileName, DeltaActionType action, String indexTypeID, int documentCount) {
        synchronized(uploadQueue){
            //add old filename to uploading queue
            uploadQueue.addFirst(new DeltaFileInfoType(action, deltaFileName, documentCount, indexTypeID));
            //notify uploading thread 
            uploadQueue.notifyAll();
        }
    }
    
    /**
     * Returns the current connection ID
     * @return the connection ID
     */
    public int getConnectionID(){
        return connectionID;
    }
    
    /**
     * Closes this DeltaFileUploader
     */
    public void close() {
    	synchronized(uploadQueue){
    		//signal cmsUploader to handle self destruction after upload
    		doDieAfterCurrentUploads = true;
    		//in case cmsUploader is sleeping
    		uploadQueue.notifyAll();
    	}
    }
    
    /**
     * Shuts down this DeltaFileUploader
     */
    private void shutdown() {
    	//Unregister updater
		try{
			manager.disconnectUpdater(connectionID);
		}
		catch(Exception e){
			logger.error("Unable to disconnect updater: ", e);
		}

		try{
			if(removalSubscription != null){
				removalSubscription.destroy(new Destroy());
			}
		}
		catch(Exception e){
			logger.error("Remove IndexRemoval notification subscription: ", e);
		}

//		try{
//			notificationConsumer.stopListening();
//		}
//		catch(Exception e){
//			logger.error("Unable to stop the notification consumer: ", e);
//		}
    }
    
    /**
     * Zips a file
     * @param delta the delta file to zip
     * @return the zipped file
     * @throws Exception an error occured
     */
    private File zip(File delta) throws Exception {
        File zipFile = new File(delta.getParentFile(), delta.getName() + ".zip");
        FileOutputStream out = new FileOutputStream(zipFile);

        CheckedOutputStream checksum = new CheckedOutputStream(out, new Adler32());
        ZipOutputStream zipOut = new ZipOutputStream(new BufferedOutputStream(checksum));

        File[] inputFileList;
        String entryPath;
        
        if(delta.isDirectory()){
            inputFileList = delta.listFiles(new java.io.FileFilter() {
                public boolean accept(File f) {
                    return  !f.isDirectory();
                }
            });
            
            entryPath = delta.getName() + "/";
        }
        else{
            inputFileList = new File[]{delta};
            entryPath = "";
        }
        
        for(int i = 0; i < inputFileList.length; i++){
        
            FileInputStream in = new FileInputStream(inputFileList[i]);
            
            byte[] buf = new byte[1024];
            int len;
            ZipEntry entry = new ZipEntry(entryPath + inputFileList[i].getName());
            zipOut.putNextEntry( entry);
            logger.debug(entry.getName());

            while ((len = in.read(buf)) >= 0) {
                zipOut.write(buf, 0, len);
            }
            in.close();
            zipOut.closeEntry();
        }
        zipOut.close();
        
        return zipFile;
    }

    /**
     * Uploads a delta file to the Content Management
     * @param uploadFile the file to upload
     * @return the ID that the CMS assigned to the uploaded file
     * @throws Exception an error occured
     */
    private String cmsUpload(final File uploadFile) throws Exception {
    	
    	final StringBuilder cmsID = new StringBuilder();
    	/*final BufferedInputStream localInput = new BufferedInputStream(new FileInputStream(uploadFile), 2048);
        final ByteArrayOutputStream memorizedFile = new ByteArrayOutputStream(new Long(uploadFile.length()).intValue());
        final BufferedOutputStream localOutput = new BufferedOutputStream(memorizedFile, 2048);
        byte[] readBuffer = new byte[2048];
        int readLength;
        while ( (readLength = localInput.read(readBuffer)) >= 0){
            localOutput.write(readBuffer, 0, readLength);
        }
        localOutput.close();
        
    	CMSServiceHandler cmsHandler = new CMSServiceHandler(resource.getServiceContext()) {
			@Override
			protected void interact(EndpointReferenceType arg0) throws Exception {
				CMSPortType1PortType cms = getCmsPortType(arg0);
				try{
					StoreDocumentParameters params = new StoreDocumentParameters();
	                SimpleDateFormat dateFormat = new SimpleDateFormat("EEEE MMM dd HH:mm:ss:SSS yyyy");
	                params.setDocumentName("IndexDeltaFile: " +  dateFormat.format(Calendar.getInstance().getTime()));
	                params.setSourceFileLocation(BasicInfoObjectDescription.RAW_CONTENT_IN_MESSAGE);
	                params.setCollectionID(deltaFileCollectionID);
	                params.setRawContent(memorizedFile.toByteArray());
	                cmsID.append(cms.storeDocument(params).getDocumentID());
				} catch (Exception e) {
					logger.warn("Failed to upload delta file to CMS: " + uploadFile.getAbsolutePath(), e);
					try{Thread.sleep(1000);} catch(InterruptedException ie){};
					throw e;
				}
			}
    	};
    	cmsHandler.setHandled(resource);
		cmsHandler.setAttempts(10);
		*/
		
		
		try {
			if(this.deltaFileCollectionID == null)
			{
				logger.error("There is no delta collection ID during delta file upload to CMS ");
				throw new Exception("No delta collection ID during delta file upload to CMS ");
			}
			//instantiate the CMWriter
			
			logger.trace("trying to create a writer");
			
//			DocumentWriter cmWriter = new DocumentWriter(this.deltaFileCollectionID, resource.getServiceContext().getScope());
//			 
//			//create an instance of a gcube document
//			SimpleDateFormat dateFormat = new SimpleDateFormat("EEEE MMM dd HH:mm:ss:SSS yyyy");
//			GCubeDocument document = new GCubeDocument();
//			document.setName("IndexDeltaFile: " +  dateFormat.format(Calendar.getInstance().getTime()));
//			document.setMimeType("application/octet-stream");
//			document.setBytestream(new FileInputStream(uploadFile));
//			logger.trace("adding to cmWriter");
			
			
			ScopeProvider.instance.set(resource.getServiceContext().getScope().toString());
			
			IClient client=new StorageClient("Index", "StorageHandler", "delta", AccessType.SHARED).getClient();
			
			SimpleDateFormat dateFormat = new SimpleDateFormat("EEEE MMM dd HH:mm:ss:SSS yyyy");
			
			String localFile = uploadFile.getAbsolutePath();
			String remoteFile = this.deltaFileCollectionID + "/" + Calendar.getInstance().getTimeInMillis();
			
			logger.info("PUT local : " + localFile + " , remote : " + remoteFile);
			
			String id=client.put(true).LFile(localFile).RFile(remoteFile);
			
			logger.info("upload id : " + id);
			//ask for adding the new document
			cmsID.append(id);
			
		} catch (Exception e) {
			throw new Exception("Failed to upload delta file to CMS: " + uploadFile.getAbsolutePath(), e);
		}
		String OID = cmsID.toString();
    	logger.info("Uploaded delta file: " + uploadFile.getAbsolutePath() + " to CMS, OID = " + OID);
		return OID;
    }

    /**
     * Delete a file belonging to the index (both locally and remotely)
     * @param cmsID the CMS ID of the file to delete
     * @param delta the local file to delete
     * @param zip the zipped local file to delete
     */
    private void cleanUp(final String cmsID, final File delta, final File zip) {
        // delete remote zip
    	try {
	    	/*CMSServiceHandler cmsHandler = new CMSServiceHandler(resource.getServiceContext()) {
				@Override
				protected void interact(EndpointReferenceType arg0) throws Exception {
					CMSPortType1PortType cms = getCmsPortType(arg0);
					cms.deleteDocument(cmsID);
				}
	    	};
	    	cmsHandler.setHandled(resource);
	    	cmsHandler.setAttempts(5);
	    	cmsHandler.run();
	    	*/
    		if(this.deltaFileCollectionID == null)
			{
				logger.error("There is no delta collection ID during delta file deletion from CMS ");
				throw new Exception("No delta collection ID during delta file deletion from CMS ");
			}
			//instantiate the CMWriter
    		ScopeProvider.instance.set(resource.getServiceContext().getScope().toString());
    		IClient client=new StorageClient("Index", "StorageHandler", "delta", AccessType.SHARED).getClient();
    		
    		logger.info("REMOVE " + cmsID);
    		
    		//client.remove().RFileById(cmsID);
    		client.remove().RFile(cmsID);
    	} 
    	catch(Exception e){
    		logger.error("Failed to delete object with OID = " + cmsID + " from CMS.", e);
    	}

        // delete local delta
    	if (delta!=null) { 
    		if (!removeFileOrDir(delta))
    			logger.error("Failed to delete local delta file: " + delta.getAbsolutePath());
    	}
        
        // delete local zip
    	if (zip!=null) { 
    		if (!removeFileOrDir(zip))
    			logger.error("Failed to delete local zipped delta file: " + zip.getAbsolutePath());
    	}
    }
    
    /**
     * Subscribes this DeltaFileConsumer for notifications involving the destruction
     * of this index.
     */
    private void subscribeForIndexRemoval(GCUBEScope scope){
        try{
            notificationConsumer = new ConsumerNotification(scope);
            
            // Subscribe to index destruction notifications
            removalSubscription = manager.subscribeForIndexRemoval(notificationConsumer);
            logger.debug("Uploader subscribed for removal notification.");
        }
        catch(Exception e){
            logger.error("Failed to subscribe for index removal", e);
        }
    }
    
    /**
     * Deletes a local file or directory.
     * 
     * @param file the file or directory to delete
     * @return true if the file or directory was successfully deleted, else false
     */
    private boolean removeFileOrDir(File file) {
        if (file.canRead()) {
            if (file.isDirectory()) {
                String[] files = file.list();
                if (files != null) {
                    for (int i = 0; i < files.length; i++) {
                        removeFileOrDir(new File(file, files[i]));
                    }
                }
            }
            return  file.delete();
        } else return false;
    }

    
    /**
     * Destroys the DeltaFileUploader
     * @throws IndexException an error occurred
     */
    private void destroyResource() throws IndexException{
        try{
            cmsUploader.interrupt();
            resource.getPorttypeContext().getWSHome().remove(resource.getID());
        }
        catch(Exception e){
            throw new IndexException(e);
        }
    }
    
    /**
     * Class that handles the consuming of received notifications
     * @author Spyros Boutsis, NKUA
     */
    public class ConsumerNotification extends IndexNotificationConsumer {
    	public ConsumerNotification(GCUBEScope scope) { super(resource, scope); }
    	
        protected void onNewNotification(NotificationEvent event){ 
           try {
	            logger.debug("DeltaFileUploader received index removal notification (IndexID: " + indexID + ")");
	            destroyResource();
            }  catch (Exception e) {
            	logger.error("Failed to process received notification", e);
            }
         }
    }
}
