package org.gcube.indexmanagement.storagehandling;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.StringReader;
import java.util.Calendar;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import javax.xml.namespace.QName;

import org.apache.axis.components.uuid.UUIDGen;
import org.apache.axis.components.uuid.UUIDGenFactory;
import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.core.contexts.GCUBEServiceContext;
import org.gcube.common.core.informationsystem.notifier.ISNotifier.NotificationEvent;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.contentmanagement.blobstorage.service.IClient;
import org.gcube.contentmanager.storageclient.wrapper.AccessType;
import org.gcube.contentmanager.storageclient.wrapper.StorageClient;
import org.gcube.indexmanagement.common.IndexException;
import org.gcube.indexmanagement.common.IndexLookupWSResource;
import org.gcube.indexmanagement.common.IndexNotificationConsumer;
import org.gcube.indexmanagement.storagehandling.stubs.DeltaActionType;
import org.gcube.indexmanagement.storagehandling.stubs.DeltaFileInfoType;
import org.gcube.indexmanagement.storagehandling.stubs.DeltaListManagerCreatedNotificationMessageType;
import org.gcube.indexmanagement.storagehandling.stubs.UpdateNotificationMessageType;
import org.globus.wsrf.core.notification.SubscriptionManager;
import org.globus.wsrf.encoding.ObjectDeserializer;
import org.globus.wsrf.encoding.ObjectSerializer;
import org.oasis.wsn.PauseSubscription;
import org.oasis.wsn.ResumeSubscription;
import org.oasis.wsrf.lifetime.Destroy;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

/**
 * This class implements a consumer for delta files that make up an index. Each time a new delta file
 * is added to the index, the DeltaFileConsumer fetches it from the Content Management layer and stores it
 * locally.
 */
public class DeltaFileConsumer {

	/**
	 *  Logger
	 */
	private static GCUBELog logger = new GCUBELog(DeltaFileConsumer.class);
	/*private static SimpleCredentialsListener credentialsListener;
    static{
        try{
            credentialsListener = new SimpleCredentialsListener();
            logger = new GCUBELog(DeltaFileConsumer.class);
            DelegationLocalInterface.registerCredentialsListener(credentialsListener);
            logger.debug("DHN secure. Credential listener created and registered.");

            ExtendedGSSCredential cred = credentialsListener.getCredentials();
            if(cred == null){
                logger.debug("No credentials currently available");
            }
            else{
                logger.debug("Credentials available. Distinguished name: " + cred.getName());
            }
        }
        catch(Exception e){
            logger.error("Unable to create CredentialsListener:", e);
        }
    }*/

	/** The index replicator */
	private IndexReplicator replicator;

	/** The ID of the collection that stores the documents related to delta files */
	private String deltaCollectionID = null;

	/** The starting delta file for this index */
	private int deltaListStartIdx;

	/** The interface of the corresponding management resource for this index */
	private DeltaListManagementWrapper indexManager;

	/** The notification consumer used by this DeltaFileConsumer for delta file additions */
	private IndexNotificationConsumer additionNotificationConsumer;

	/** The notification consumer used by this DeltaFileConsumer for delta file deletions */
	private IndexNotificationConsumer deletionNotificationConsumer;

	/** The notification consumer used by this DeltaFileConsumer for index removal */
	private IndexNotificationConsumer removalNotificationConsumer;

	/** The notification consumer used by this DeltaFileConsumer for index manager creation */
	private IndexNotificationConsumer managerCreationNotificationConsumer;

	/** The SubscriptionManager used for delta file addition notifications */
	private SubscriptionManager additionSubscription;

	/** The SubscriptionManager used for delta file deletion notifications */
	private SubscriptionManager deletionSubscription;

	/** The SubscriptionManager used for index removal notifications */
	private SubscriptionManager removalSubscription;

	/** The SubscriptionManager used for index management creation notifications */
	private SubscriptionManager managerCreationSubscription;

	/** Does this consumer handle delta file additions? */
	private boolean doesHandleAdditions = true;

	/** Does this consumer handle delta file deletions? */
	private boolean doesHandleDeletions = true;

	/** Does this consumer enforce strict ordering? */
	private boolean enforceStrictOrder = true;

	/** The index lookup resource that uses this consumer */
	private IndexLookupWSResource resource;

	/** The consumer's download queue */
	private LinkedList<DeltaData> downloadQueue;

	/** The thread that replicates the index */
	private IndexReplicatorThread replicatorThread;
	
	/** The list of delta files that have been merged into the local index */
	private List<DeltaFileInfoType> deltaFileList;
	
	/** UUIDGen for generating random, unique filenames */
	private static final UUIDGen uuidGen = UUIDGenFactory.getUUIDGen();
	
	/**
	 * Constructs a new, empty (uninitialized) DeltaFileConsumer
	 */
	public DeltaFileConsumer() {}
	
	/**
	 * Constructs a new DeltaFileConsumer
	 * 
	 * @param resource the index lookup resource that uses this DeltaFileConsumer
	 * @param replicator the index replicator to use
	 * @param startIdx the index of the first delta file
	 * @throws Exception an error occured
	 */
	public DeltaFileConsumer(IndexLookupWSResource resource, IndexReplicator replicator, int startIdx) throws Exception{
		this(
				resource, 
				replicator, 
				new RemoteDeltaListManager(resource.getIndexID(), resource.getServiceContext(), resource.getManagementResourceNamespace()), 
				startIdx, 
				true
			);
	}

	/**
	 * Constructs a new DeltaFileConsumer
	 * 
	 * @param resource the index lookup resource that uses this DeltaFileConsumer
	 * @param replicator the index replicator to use
	 * @param indexManager the index manager
	 * @param startIdx the index of the first delta file
	 * @param subscribeForUpdates subscribe for update notifications or not?
	 * @throws Exception an error occured
	 */
	private DeltaFileConsumer(IndexLookupWSResource resource, IndexReplicator replicator, DeltaListManagementWrapper indexManager,
			int startIdx, boolean subscribeForUpdates) throws Exception {
		this.deltaFileList = new LinkedList<DeltaFileInfoType>();
		initialize(resource, replicator, indexManager, startIdx, subscribeForUpdates);
		synchronizeWithIndexManager();
		this.replicatorThread.start();
	}

	/**
	 * Initializes this DeltaFileConsumer
	 * 
	 * @param resource the index lookup resource that uses this DeltaFileConsumer
	 * @param replicator the index replicator to use
	 * @param indexManager the index manager
	 * @param startIdx the index of the first delta file
	 * @param subscribeForUpdates subscribe for update notifications or not?
	 * @throws Exception
	 */
	private void initialize(IndexLookupWSResource resource, IndexReplicator replicator, DeltaListManagementWrapper indexManager,
			int startIdx, boolean subscribeForUpdates) throws Exception {
		
		GCUBEServiceContext ctx = resource.getServiceContext();
		
		this.replicator = replicator;
		this.indexManager = indexManager;
		this.downloadQueue = new LinkedList<DeltaData>();
		this.resource = resource;
		this.deltaListStartIdx = startIdx;
		this.additionNotificationConsumer = new AdditionConsumerNotification(ctx.getScope());
		this.deletionNotificationConsumer = new DeletionConsumerNotification(ctx.getScope());
		this.removalNotificationConsumer = new RemovalConsumerNotification(ctx.getScope());
		this.managerCreationNotificationConsumer = new ManagerCreationConsumerNotification(ctx.getScope());
		this.deltaCollectionID = indexManager.getDeltaCollectionID();

		if(subscribeForUpdates){
			subscribeForDeltaListChanges();
		}
		subscribeForManagerCreation();
		subscribeForIndexRemoval();

		this.replicatorThread = new IndexReplicatorThread();
		ctx.setScope(this.replicatorThread, ctx.getScope());
		ctx.useCredentials(this.replicatorThread, ctx.getCredentials());
	}
	
    /**
     * Stores the current state of this DeltaFileConsumer to an object stream.
     * @param oos the output stream
     */
    public void storeState(ObjectOutputStream oos) {
    	try {
    		/* Serialize the remote index manager EPR */
    		if (indexManager.isEmpty())
    			oos.writeBoolean(false);
    		else {
    			oos.writeBoolean(true);
    			oos.writeObject(
    				ObjectSerializer.toString(indexManager.getIndexManagerEPR(), new QName(resource.getNamespace(), "indexManagerEPR")));
    		}
    		
    		oos.writeBoolean(this.doesHandleAdditions);
    		oos.writeBoolean(this.doesHandleDeletions);
    		oos.writeBoolean(this.enforceStrictOrder);
    		oos.writeInt(this.deltaFileList.size());
    		for (int i=0; i<this.deltaFileList.size(); i++)
    			oos.writeObject(this.deltaFileList.get(i));
    		
    	} catch (Exception e) {
    		logger.error("Failed to store DeltaFileConsumer state.", e);
    	}
    }
    
    /**
     * Loads the current state of this DeltaFileConsumer from an object stream.
     * @param ois the input stream
     * @param firstLoad specifies whether this resource is being loaded for the first time or not
	 * @param resource the index lookup resource that uses this DeltaFileConsumer
	 * @param replicator the index replicator to use
	 * @param startIdx the index of the first delta file
     */
    public void loadState(ObjectInputStream ois, boolean firstLoad, IndexLookupWSResource resource, IndexReplicator replicator, int startIdx) throws Exception {
    	
    	logger.debug(">>>>>>>>>>>>>>>>>>>>>>>>>>>>>> Starting DeltaFileConsumer state loading");
    	RemoteDeltaListManager remoteDeltaManager = null;
    	
    	/* Check if an index manager EPR has been stored in the delta file consumer state. If found, this EPR will
    	 * be used if this is not the first time that the resource is being loaded. If an EPR is not found or this is
    	 * the first time the resource is being loaded, a RemoteDeltaListManager is constructed without specifying an
    	 * EPR, so that it will search for the correct EPR to use.
    	 */
    	boolean bCreateWithEPR = false;
    	String sIndexManagerEPR = null;
    	boolean bReadIndexManagerEPR = ois.readBoolean();
    	if (bReadIndexManagerEPR) {
    		sIndexManagerEPR = (String) ois.readObject();
    		if (!firstLoad)
    			bCreateWithEPR = true;
    	}
    	
    	if (bCreateWithEPR) {
    		logger.info("A manager EPR was stored in the DeltaFileConsumer's state. Going to use it: " + sIndexManagerEPR);
    		EndpointReferenceType indexManagerEPR = (EndpointReferenceType) ObjectDeserializer.deserialize(
    				new InputSource(new StringReader(sIndexManagerEPR)), EndpointReferenceType.class);
    		remoteDeltaManager = new RemoteDeltaListManager(resource.getIndexID(), indexManagerEPR, resource.getServiceContext());
    	}
    	else {
    		logger.info("Not using a stored manager EPR, will search for one.");
    		remoteDeltaManager = new RemoteDeltaListManager(resource.getIndexID(), resource.getServiceContext(), resource.getManagementResourceNamespace());
    	}
    	
    	this.doesHandleAdditions = ois.readBoolean();
    	this.doesHandleDeletions = ois.readBoolean();
    	this.enforceStrictOrder = ois.readBoolean();
    	this.deltaFileList = new LinkedList<DeltaFileInfoType>();
    	int size = ois.readInt();
    	for (int i=0; i<size; i++)
    		this.deltaFileList.add((DeltaFileInfoType) ois.readObject());
    	
    	/* initialize */
    	initialize(resource,
    			replicator, 
    			remoteDeltaManager, 
    			startIdx, 
    			true);

    	this.replicatorThread.start();
    	
    	logger.debug("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< DeltaFileConsumer state loading done.");
    }
    
	/**  
	 * Thread used to build a replica of the main index, 
	 * and to keep the replica up to date
	 */
	class IndexReplicatorThread extends Thread {

		/**
		 * Class constructor
		 */
		public IndexReplicatorThread() { }
	    
	    /*
	     * (non-Javadoc)
	     * @see java.lang.Thread#run()
	     */
		public void run(){
			//DeltaFileInfoType[] deltaList = null;

			/* Check if the indexManager is empty. If it is,
			 * this consumer is not currently connected to a
			 * manager (because a manager does not exist in
			 * the infrastructure). In this case, just wait
			 * until a manager is created.
			 */
			synchronized (indexManager) {
				if (indexManager.isEmpty()) {
					try {
						//--Check if interrupt's been called to stop the thread--
						Thread.yield();
						if (Thread.currentThread().isInterrupted()) {
							logger.debug("Consumer thread cancelled before wait.");
							return;
						}//------------------------------------------------------

						logger.info("Consumer is not currently connected to a manager. Going to wait until manager creation...");
						indexManager.wait();
					} catch(InterruptedException e){
						logger.debug("Consumer thread cancelled while waiting.");
						return;
					}
				}
			}

			/*
			try {
				synchronizeWithIndexManager();
			}
			catch(Exception e) {
				logger.error("Unable to build index from initial delta file list.", e);
			}
			*/
			
			int nextIdx = deltaFileList.size();
			boolean bDownloadedNewData = false;
			while (true){
				try{

					/* Get the next delta file descriptor from the queue.
					 * If the queue is empty, wait until it is populated.
					 */
					DeltaData delta;
					/*
					 * Hold the monitor of downloadQueue until the info of the deltafiles removed from the queue, 
					 * is reflected to the deltaFileList 
					 */
					synchronized (downloadQueue) {
						if(downloadQueue.size() == 0){
							
							/* If new data has been downloaded, store the state of the resource
							 * now that there are no more files to download.
							 */
							if (bDownloadedNewData == true)
								resource.store();
							
							try{
								//--Check if interrupt's been called to stop the thread--
								Thread.yield();
								if (Thread.currentThread().isInterrupted()) {
									logger.debug("Consumer thread cancelled before wait.");
									return;
								}//------------------------------------------------------

								logger.info("DeltaFileConsumer waiting... " );
								downloadQueue.wait();
							}
							catch(InterruptedException e){
								logger.debug("Consumer thread cancelled while waiting.");
								return;
							}
						}
						delta = downloadQueue.removeLast();
						
						bDownloadedNewData = false;
						logger.info("DeltaFileConsumer fetching DeltaInfo... Received " + delta.idx + ", expected " + (enforceStrictOrder ? nextIdx : "any"));
						DeltaFileInfoType deltaInfo;

						if(enforceStrictOrder){
							//when doing both additions and deletion, the order is important.
							//this is a loop to make sure all lower id delta files have been merged.
							for(;nextIdx <= delta.idx; nextIdx++){
								//--Check if interrupt's been called to stop the thread--
								Thread.yield();
								if (Thread.currentThread().isInterrupted()) {
									logger.debug("Consumer thread cancelled in merge loop.");
									return;
								}//------------------------------------------------------

								if (delta.idx != nextIdx){
									if ((deltaInfo = getDeltaInfoFromQueue(nextIdx)) == null){
										// get the ID of a missing delta file
										deltaInfo = indexManager.getDeltaFileInfo(nextIdx);
									}
								}
								else {
									deltaInfo = delta.info;
								}
								logger.info("Merging deltafile: " + nextIdx);
								mergeDeltaFile(deltaInfo);
								bDownloadedNewData = true;
							}
						}
						else{                
							deltaInfo = delta.info;
							logger.info("Merging deltafile: " + delta.idx);
							mergeDeltaFile(deltaInfo);
							bDownloadedNewData = true;
						}
					}
					
				}catch(Exception e){
					logger.error(" Error downloading or merging delta file.", e);
				}
			}
		}
	}

	/**
     * This function must be called when the consumer has been down for some time, so it's possible
     * for the producer to have been updated with new data which the consumer does not know about.
     * This function compares the locally kept delta file list with the delta file list of the
     * remote producer and returns the index of the first delta file that corresponds to an update
     * that has not been received by the consumer.
     * If any delta files are found to be missing, they must be downloaded and merged with the local
     * index. If some delta files of the locally kept list do not exist anymore in the remote producer's
     * list (e.g. because of a rollback of the producer to a previous state while the consumer was down),
     * the local list is deleted and the index is re-built from scratch.
     * 
     * Of course, all of this could have been avoided if we chose to rebuild the whole index from
     * scratch by fetching all the delta files every time the consumer was loaded, but this would
     * be obsolete and a waste of network resources most of the time.
     * 
     * @param producerDeltaFileList the delta file list of the remote producer
     * @return the start index for the portion of delta files that need to be merged with the local
     * index, as counted inside the given 'producerDeltaFileList'. If 0, it means that all the deltas
     * in the producer list will be merged (i.e. the index is fully rebuild from scratch). If -1,
     * no deltas should be merged at all (i.e. the two lists are exactly the same).
     */
    private int findFirstMissedUpdate(DeltaFileInfoType[] producerDeltaFileList) {	    	
    	/* Find the length of the smaller delta file list among the local and the remote one. If
    	 * the smaller one is the remote (producer), then there has been a rollback. In this case
    	 * we need to rebuild the whole index. */
    	int minCommonListLen = Math.min(producerDeltaFileList.length, deltaFileList.size());
    	if ( (minCommonListLen == producerDeltaFileList.length) && (deltaFileList.size() != producerDeltaFileList.length) ) {
    		logger.info("Remote producer delta file list is smaller than the local list. The whole index will be rebuilt.");
    		return 0;
    	}
    	else {
    		
    		/* Compare the two delta file lists up to the length of the smaller one. If any of the
    		 * delta file descriptors is different between the two lists, the index must be rebuilt
    		 * from scratch. */
	    	for (int i=0; i<minCommonListLen; i++) {
	    		if ( !deltaFileList.get(i).equals(producerDeltaFileList[i]) ) {
	    			logger.info("Differences found between the remote producer delta file list and the local list, " +
	    					"and the size of the producer list is less than or equal to the size of the local" +
	    					" list. The whole index will be rebuilt.");
	    			return 0;
	    		}
	    	}
    	}
    	
    	if (producerDeltaFileList.length > deltaFileList.size()) {
    		logger.info("Remote producer delta file list is larger than the local list, and the two lists are identical" +
    				" up to the length of the local list. The extra delta files found in the producer list will be merged" +
    				"with the local index.");
    		return deltaFileList.size();
    	}
    	
    	logger.info("The remote producer's list and the local consumer's list are identical, no merging need to be done.");
    	return -1;
    }

    /**
     * Synchronizes the local index (built by the consumer) with the manager's delta file list.
     */
    synchronized private void synchronizeWithIndexManager() throws Exception {
    	logger.info("Starting DeltaFileConsumer synchronization with DeltaListManager's file list...");
    	if (!indexManager.isEmpty()) {
	    	DeltaFileInfoType[] producerDeltaFileList = indexManager.getDeltaFileList();
			logger.info("Remote producer's delta file list contains " + producerDeltaFileList.length + " deltas.");
			logger.info("Local consumer's delta file list contains " + deltaFileList.size() + " deltas.");
			
			/* Do the synchronization while having the lock to the queue. We want to be sure that when we synchronize,
			 * we won't add twice the same Delta File in the queue and that when a deltafile is removed from the 
			 * queue in order to be merged, this method doesn't add it to the queue, before it is reported in the 
			 * deltaFileList object.
			 */
			synchronized (downloadQueue) {
				
				/* Check for consistency between the local delta file list and the remote (producer) one.
				 * If the local delta file list is empty, then download the whole list of delta files
				 * from the CMS (build the whole index locally). If not, then we just need to find which
				 * delta files we must merge with the local index.
				 * The 'checkForMissedUpdates' function does this check, and returns the index of the 
				 * starting delta file of the remote producer delta file list to merge with the local index.
				 */
				int firstDeltaToMerge = 0;
				if (deltaFileList.size() > 0) {
					logger.info("-----------------------------> 1");
					
					firstDeltaToMerge = findFirstMissedUpdate(producerDeltaFileList);
					
					logger.info("-----------------------------> 2");
					
					/* If the index of the first delta file to merge is 0, it means that the whole index
					 * must be recreated. In this case, the document count RP must be "reset" to 0.
					 */
					if (firstDeltaToMerge == 0)
						replicator.clearIndex();
					
					logger.info("-----------------------------> 3");
				}
				
				/* If there are any delta files to be merged, merge them with the local index now. */
				logger.info("-----------------------------> 4");
				
				if (firstDeltaToMerge != -1) {
					int listSize = producerDeltaFileList.length - firstDeltaToMerge;
					if (listSize > 0) {
						
						logger.info("-----------------------------> 5");
						
						//DeltaFileInfoType[] deltasToMerge = new DeltaFileInfoType[listSize];
						for (int i=firstDeltaToMerge; i<producerDeltaFileList.length; i++)
							addDeltaToDownloadQueue(i, producerDeltaFileList[i]);
						
						downloadQueue.notifyAll();
							//deltasToMerge[i-firstDeltaToMerge] = producerDeltaFileList[i];
						
						
						logger.info("-----------------------------> 6");
						
						//createFromCMS(deltasToMerge);
						
						logger.info("-----------------------------> 7");
					}
				}
				logger.info("DeltaFileConsumer synchronization with DeltaListManager's file list done!");
				
			}
    	}
    	else
    		logger.warn("Remote DeltaListManager is empty, synchronization aborted.");
    }
    
	/**
	 * Enables or disables delta file addition notification handling
	 * @param doesHandleAdditions specifies whether delta file addition notifications should be handled or not
	 * @throws Exception an error occured
	 */
	public void setDoesHandleAdditions(boolean doesHandleAdditions) throws Exception {
		if(additionSubscription != null){
			this.doesHandleAdditions = doesHandleAdditions;
			if(doesHandleAdditions){
				additionSubscription.pauseSubscription(new PauseSubscription());
			}
			else{
				additionSubscription.resumeSubscription(new ResumeSubscription());
			}
		}
	}

	/**
	 * Enables or disables delta file deletion notification handling
	 * @param doesHandleAdditions specifies whether delta file deletion notifications should be handled or not
	 * @throws Exception an error occured
	 */
	public void setDoesHandleDeletions(boolean doesHandleDeletions) throws Exception {
		if(deletionSubscription != null){
			this.doesHandleDeletions = doesHandleDeletions;
			if(doesHandleDeletions){
				deletionSubscription.pauseSubscription(new PauseSubscription());
			}
			else{
				deletionSubscription.resumeSubscription(new ResumeSubscription());
			}
		}
	}

	/**
	 * Enables or disables enforcement of strict ordering
	 * @param enforceStrictOrder specifies whether strict ordering should be enforced or not
	 * @throws Exception an error occured
	 */
	public void setEnforceStrictOrder(boolean enforceStrictOrder) throws Exception {
		this.enforceStrictOrder = enforceStrictOrder;
	}

	/**
	 * Retrieves an entry from the delta file queue
	 * @param idx the index of the entry to retrieve
	 * @return the entry
	 */
	private DeltaFileInfoType getDeltaInfoFromQueue(int idx){
		for (int i = 0; i < downloadQueue.size() ; i++){
			if(downloadQueue.get(i).idx == idx){
				return downloadQueue.remove(idx).info;
			}
		}		
		return null;
	}

	/**
	 * Creates an index from a list of delta files residing in the Content Management
	 * @param updateList the list of delta files
	 * @throws IndexException an error occured
	 */
/*	private void createFromCMS(DeltaFileInfoType[] updateList) throws IndexException{
		StringBuffer message = new StringBuffer("DeltaFileConsumer building its index based on the following update list:");
		if(updateList.length == 0){
			message.append(" List is empty");
			logger.debug(message);
			return;
		}
		for(DeltaFileInfoType info : updateList){
			message.append("\nid = " + info.getDeltaFileID() + " action = " + info.getDeltaAction().getValue());
		}
		logger.debug(message);

		for (int i = deltaListStartIdx; i < updateList.length; i++){
			try{
				mergeDeltaFile(updateList[i]);

			}catch (Exception e){logger.error("Unable to merge DeltaFile: " + updateList[i], e);}
		}
		logger.info("Initial DeltaFileConsumer build done!");
	}   
*/

	/**
	 * Downloads a file from the CMS
	 * @param indexID the ID of the index
	 * @return the downloaded file
	 * @throws Exception an error occured
	 */
	private File cmsDownload(final String indexID) throws Exception {
		final File downloadedFile = new File(resource.getIndexDataDirectory() + uuidGen.nextUUID() + ".zip");

		/*CMSServiceHandler cmsHandler = new CMSServiceHandler(resource.getServiceContext()) {
			@Override
			protected void interact(EndpointReferenceType arg0) throws Exception {
				CMSPortType1PortType cms = getCmsPortType(arg0);
				try{
					GetDocumentParameters getDocumentParams = new GetDocumentParameters();
					getDocumentParams.setDocumentID(indexID);
					getDocumentParams.setTargetFileLocation(BasicInfoObjectDescription.RAW_CONTENT_IN_MESSAGE);
					getDocumentParams.setUnrollLevels(5); //??????
					DocumentDescription description = cms.getDogetcument(getDocumentParams);

					downloadedFile.getParentFile().mkdirs();
					BufferedInputStream input = new BufferedInputStream( new ByteArrayInputStream(description.getRawContent()), 2048);
					BufferedOutputStream output = new BufferedOutputStream( new FileOutputStream(downloadedFile), 2048);
					byte[] buffer = new byte[2048];
					int length;
					while ( (length = input.read(buffer)) >= 0){
						output.write(buffer, 0, length);
					}
					input.close();
					output.close();

					logger.info("Downloaded file of size: " + downloadedFile.length());
				}
				catch (Exception e) {get
					logger.warn("Failed to download delta file from CMS.", e);
					try{Thread.sleep(1000);} catch(InterruptedException ie){};
					throw e;
				}
			}
		};
		
		cmsHandler.setHandled(resource);
		cmsHandler.setAttempts(10);*/
		
		try {
			//get the document using this DocID and the Delta Collection ID
			if(this.deltaCollectionID == null)
			{
				logger.error("No Delta collection ID is retrieved.. Trying again");
				this.deltaCollectionID = indexManager.getDeltaCollectionID();
			}
			if(this.deltaCollectionID == null)
			{
				logger.error("There is no delta collection ID during delta file download from CMS ");
				throw new Exception("No delta collection ID during delta file download from CMS ");
			}
			
			//instantiate the CMReader
//			DocumentReader docReader = new DocumentReader(this.deltaCollectionID, resource.getServiceContext().getScope());
//			 
//			DocumentProjection projection = document();
//            //setting the include content into true is not needed in the current version of CM - 22/10/10
//			//projection.includeContent(true);
//            //retrieve the document, we are not specifying projections, therefore the entire document is returned.
//			GCubeDocument document = docReader.get(indexID, projection);
//			InputStream is = new ByteArrayInputStream(document.bytestream());
			
			ScopeProvider.instance.set(resource.getServiceContext().getScope().toString());
			IClient client=new StorageClient("Index", "StorageHandler", "delta", AccessType.SHARED).getClient();
			
			downloadedFile.getParentFile().mkdirs();
			
			//String remoteFile = this.deltaCollectionID + "/" + indexID;
			String remoteFile = indexID;
			String localFile = downloadedFile.getAbsolutePath();
			
			logger.info("GET local : " + localFile + " , remote : " + remoteFile);
			
			client.get().LFile(localFile).RFileById(remoteFile);
			
			
			logger.info("Downloaded file of size: " + downloadedFile.length());
		} catch (Exception e) {
			throw new Exception("Failed to download delta file from CMS.", e);
		}
		return downloadedFile;
	}

	/**
	 * Unzips a file
	 * @param index the zipped file to unzip
	 * @param targetDir the target directory to unzip the file to
	 * @return the unzipped file
	 * @throws Exception an error occured
	 */
	private File unzipIndex(File index, String targetDir) throws Exception {
		ZipFile zip = new ZipFile(index, ZipFile.OPEN_READ);

		Enumeration<? extends ZipEntry> zipEntries = zip.entries();
		File extractedFile = null, returnFile = null;
		ZipEntry entry = null;
		String fullEntryName = null;
		String entryName = null;
		String uid = uuidGen.nextUUID();
		
		while (zipEntries.hasMoreElements())
		{
			entry = zipEntries.nextElement();
			logger.debug("Unzipping entry: " + entry.getName());

			/* Generate a random ID as the file name			
			 * If the entry contains a path, replace the initial directory name
			 * with the random ID generated above.
			 */
			entryName = uid;
			int start;
			if ((start = entry.getName().indexOf('/')) != -1) {
				entryName = entryName + entry.getName().substring(start);
			}

			/* Construct the full entry name (path + filename) */
			fullEntryName = targetDir + entryName;

			extractedFile = new File (fullEntryName);
			File parent = extractedFile.getParentFile();
			if(!parent.exists())
				parent.mkdir();

			if(! entry.isDirectory()){
				BufferedInputStream entryIn = new BufferedInputStream(zip.getInputStream(entry), 2048);
				BufferedOutputStream entryOut = new BufferedOutputStream(new FileOutputStream(extractedFile), 2048);
				byte[] buffer = new byte[2048];
				int len;

				while((len = entryIn.read(buffer)) >= 0)
					entryOut.write(buffer, 0, len);

				entryIn.close();
				entryOut.close();
			}

		}

		//either a directory containing everything or a single file
		if(entry != null){
			int directoryNameEnd = entryName.indexOf('/');
			if(directoryNameEnd != -1){
				returnFile = new File (targetDir + entryName.substring(0, directoryNameEnd));
			}
			else{
				returnFile = extractedFile;
			}
		}

		return returnFile;
	}

	/**
	 * Merges a delta file in the local index
	 * @param deltaInfo info about the delta file to be merged
	 * @throws Exception an error occured
	 */
	private void mergeDeltaFile(DeltaFileInfoType deltaInfo) throws Exception {
		logger.info("In mergeDeltaFile deltaFileID : " + deltaInfo.getDeltaFileID());
		
		
		if(deltaInfo.getDeltaAction().equals(DeltaActionType.Addition) && doesHandleAdditions){
			File zip = cmsDownload(deltaInfo.getDeltaFileID());
			logger.info("downloaded update file: " + zip.getAbsolutePath());
			File deltaFile = null;
			deltaFile = unzipIndex(zip, resource.getIndexDataDirectory());
			logger.info("unzipped addition update file to: " + deltaFile.getAbsolutePath());
			zip.delete();

			long before = Calendar.getInstance().getTimeInMillis();
			replicator.mergeAddition(deltaFile, deltaInfo.getDeltaFileID(), deltaInfo.getDocumentCount());
			long after = Calendar.getInstance().getTimeInMillis();

			if (!deleteDirectory(deltaFile)) {
				logger.info("Addition complete, but was unable to completely delete temporary directory: " + deltaFile.getAbsolutePath());
			} else {
				logger.info("Addition of \"" + deltaInfo.getDeltaFileID() + "\"completed in " + (after - before) + "ms and the temporary directory \"" + deltaFile.getAbsolutePath() + "\" was deleted.");
			}
			addDeltaFileToList(deltaInfo);
		}
		else if(deltaInfo.getDeltaAction().equals(DeltaActionType.Deletion) && doesHandleDeletions){
			File zip = cmsDownload(deltaInfo.getDeltaFileID());
			logger.info("downloaded update file: " + zip.getAbsolutePath());
			File deltaFile = null;
			deltaFile = unzipIndex(zip, resource.getIndexDataDirectory());
			logger.info("unzipped deletion update file to: " + deltaFile.getAbsolutePath());
			zip.delete();

			long before = Calendar.getInstance().getTimeInMillis();
			replicator.mergeDeletion(deltaFile, deltaInfo.getDeltaFileID(), deltaInfo.getDocumentCount());
			long after = Calendar.getInstance().getTimeInMillis();

			if (!deleteDirectory(deltaFile)) {
				logger.info("Deletion complete, but was unable to completely delete temporary directory: " + deltaFile.getAbsolutePath());
			} else {
				logger.info("Deletion of \"" + deltaInfo.getDeltaFileID() + "\"completed in " + (after - before) + "ms and the temporary directory \"" + deltaFile.getAbsolutePath() + "\" was deleted.");
			}
			addDeltaFileToList(deltaInfo);
		}   
	}

	/**
	 * Adds a delta file descriptor to the local delta file list
	 * @param deltaFileInfo the delta file descriptor to add
	 */
	private void addDeltaFileToList(DeltaFileInfoType deltaFileInfo) {
		this.deltaFileList.add(deltaFileInfo);
    }
	
	/**
	 * Subscribes this DeltaFileConsumer for notifications involving changes in the
	 * delta file list for this index.
	 */
	private void subscribeForDeltaListChanges(){
		try {
			// Subscribe to addition notifications
			additionSubscription = indexManager.subscribeForAdditions(additionNotificationConsumer);
			logger.debug("Consumer subscribed for addition notification.");

			// Subscribe to deletion notifications
			deletionSubscription = indexManager.subscribeForDeletions(deletionNotificationConsumer);
			logger.debug("Consumer subscribed for deletion notification.");
		}
		catch(Exception e){
			logger.error("Failed to subscribe for delta list changes.", e);
		}
	}

	/**
	 * Subscribes this DeltaFileConsumer for notifications involving the destruction
	 * of this index.
	 */
	private void subscribeForIndexRemoval(){
		try{
			// Subscribe to index destruction notifications
			removalSubscription = indexManager.subscribeForIndexRemoval(removalNotificationConsumer);
			logger.debug("Consumer subscribed for removal notification.");
		}
		catch(Exception e){
			logger.error("Failed to subscribe for index removal", e);
		}
	}

	/**
	 * Subscribes this DeltaFileConsumer for notifications involving the creation
	 * of a new management resource for this index.
	 */
	private void subscribeForManagerCreation(){
		try{
			// Subscribe to index manager creation notifications
			managerCreationSubscription = indexManager.subscribeForManagerCreation(managerCreationNotificationConsumer);
			logger.debug("Consumer subscribed for manager creation notification.");
		}
		catch(Exception e){
			logger.error("Failed to subscribe for manager creation", e);
		}
	}

	/**
	 * Deletes a directory on the local file system.
	 * @param dir the directory to delete
	 * @return true if the directory was successfully deleted, else false
	 */
	private boolean deleteDirectory(File dir) {
		if (dir.canRead()) {
			if (dir.isDirectory()) {
				String[] files = dir.list();
				if (files != null) {
					for (int i = 0; i < files.length; i++) {
						deleteDirectory(new File(dir, files[i]));
					}
				}
			}
			return  dir.delete();
		} else return false;
	}



	/**
	 * Holds information about a delta file
	 */
	private class DeltaData{

		/**
		 * The delta file info
		 */
		public DeltaFileInfoType info;

		/**
		 * The delta file index
		 */
		public int idx;

		/**
		 * Class constructor
		 * @param info the delta file info
		 * @param idx the delta file index
		 */
		public DeltaData(DeltaFileInfoType info, int idx){
			this.info = info;
			this.idx = idx;
		}

		@Override
		public boolean equals(Object obj){
			try{
				DeltaData otherData = (DeltaData) obj;
				return this.idx == otherData.idx && this.info.equals(otherData.info) ;
			}catch (Exception e){return false;}
		}
	}

	/**
	 * Adds a delta file to the download queue
	 * @param deltaFileIdx
	 * @param deltaFileInfo
	 */
	private void addDeltaToDownloadQueue(int deltaFileIdx, DeltaFileInfoType deltaFileInfo) {
		//check if this deltaFile is already stored in the queue 
		Iterator<DeltaData> iter = downloadQueue.iterator();
		while(iter.hasNext())
		{
			DeltaData data = iter.next();
			if(data.info.getDeltaFileID().equalsIgnoreCase(deltaFileInfo.getDeltaFileID()))
			{
				logger.info("The DeltaFile with id: " + deltaFileInfo.getDeltaFileID() + ", is already in the queue.");
				return;
			}
		}			
		downloadQueue.addFirst(new DeltaData(deltaFileInfo, deltaFileIdx));
		logger.info("DeltaFileConsumer added " + deltaFileInfo.getDeltaAction().getValue() + " "
				+ deltaFileInfo.getDeltaFileID() + " (#" + deltaFileIdx + ") to download queue.");
	}
	
	/**
	 * Handles a notification regarding an index update
	 * @param notification the notification data
	 */
	private void handleUpdateNotification(UpdateNotificationMessageType notification){
		logger.info("Update notification received (idxID: " + resource.getIndexID());
		if(notification != null){
			synchronized (downloadQueue){
				addDeltaToDownloadQueue(notification.getDeltaFileIdx(), notification.getDeltaFileInfo());
				downloadQueue.notifyAll();
			}
		}
	}

	/**
	 * Handles a notification regarding the creation of a delta list manager for this index
	 * @param notification the notification data
	 */
	private void handleManagerCreationNotification(DeltaListManagerCreatedNotificationMessageType notification) {
		try {
			synchronized (downloadQueue) {
				downloadQueue.clear();
			}

			synchronized (indexManager) {
				logger.info("Connecting consumer to newly-created manager");
				EndpointReferenceType managerEPR = notification.getEndpointReference();
				indexManager.setIndexManagerEPR(managerEPR);
				resource.setConnectionID(indexManager.connectLookup());
				indexManager.notifyAll();
			}
			
			synchronizeWithIndexManager();
		} catch (Exception e) {
			logger.error("Failed to set the new delta list manager EPR, received by notification.");
		}
	}
	
	/**
	 * Closes the DeltaFileConsumer
	 * @throws Exception an error occured
	 */
	public void close() throws Exception {
		if(additionSubscription != null){
			additionSubscription.destroy(new Destroy());
		}
		if(deletionSubscription != null){
			deletionSubscription.destroy(new Destroy());
		}
		if(removalSubscription != null){
			removalSubscription.destroy(new Destroy());
		}
		if(managerCreationSubscription != null){
			managerCreationSubscription.destroy(new Destroy());
		}
		
		//signal to the indexReplicator its work is done...
		this.replicatorThread.interrupt();
	}

	/**
	 * Destroys the DeltaFileConsumer
	 * @throws IndexException an error occurred
	 */
	private void destroyResource() throws Exception{
		try{
			/*
			GCUBETProviderServiceAddressingLocator sal = new GCUBETProviderServiceAddressingLocator();
			GCUBEProvider selfReference = sal.getGCUBEProviderPort(resource.getEPR());
			selfReference = GCUBERemotePortTypeContext.getProxy(selfReference, resource.getServiceContext());
			selfReference.destroy(new Destroy());
			*/
			resource.getPorttypeContext().getWSHome().remove(resource.getID());
		}
		catch(Exception e){
			logger.error("Error while removing the index lookup resource.", e);
			throw new Exception("Error while removing the index lookup resource.", e);
		}
	}


	/**
	 * Class that handles the consuming of received addition notifications
	 * @author Spyros Boutsis, NKUA
	 */
	public class AdditionConsumerNotification extends IndexNotificationConsumer {
		public AdditionConsumerNotification(GCUBEScope scope) { super(resource, scope); }
		
		protected void onNewNotification(NotificationEvent event){
			logger.debug("DeltaFileConsumer received addition notification (idxID: " + resource.getIndexID() + ").");

			Element message = null;
			try {
				message = (Element) event.getPayload().getMessageObject();
				UpdateNotificationMessageType notification = (UpdateNotificationMessageType) ObjectDeserializer.toObject(message, UpdateNotificationMessageType.class);
				handleUpdateNotification(notification);
			}  catch (Exception e) {
				logger.error("Error while receiving delta file addition notification.", e);
			}
		}
	}
	
	/**
	 * Class that handles the consuming of received deletion notifications
	 * @author Spyros Boutsis, NKUA
	 */
	public class DeletionConsumerNotification extends IndexNotificationConsumer {
		public DeletionConsumerNotification(GCUBEScope scope) { super(resource, scope); }
		
		protected void onNewNotification(NotificationEvent event){
			logger.debug("DeltaFileConsumer received deletion notification (idxID: " + resource.getIndexID() + ").");

			Element message = null;
			try {
				message = (Element) event.getPayload().getMessageObject();
				UpdateNotificationMessageType notification = (UpdateNotificationMessageType) ObjectDeserializer.toObject(message, UpdateNotificationMessageType.class);
				handleUpdateNotification(notification);
			}  catch (Exception e) {
				logger.error("Error while receiving delta file deletion notification.", e);
			}
		}
	}
	
	/**
	 * Class that handles the consuming of received deletion notifications
	 * @author Spyros Boutsis, NKUA
	 */
	public class RemovalConsumerNotification extends IndexNotificationConsumer {
		public RemovalConsumerNotification(GCUBEScope scope) { super(resource, scope); }
		
		protected void onNewNotification(NotificationEvent event){
			logger.debug("DeltaFileConsumer received index removal notification (idxID: " + resource.getIndexID() + ").");

			try {
				destroyResource();
			}  catch (Exception e) {
				logger.error("Error while handling index removal notification.", e);
			}
		}
	}
	
	/**
	 * Class that handles the consuming of received manager creation notifications
	 * @author Spyros Boutsis, NKUA
	 */
	public class ManagerCreationConsumerNotification extends IndexNotificationConsumer {
		public ManagerCreationConsumerNotification(GCUBEScope scope) { super(resource, scope); }
		
		protected void onNewNotification(NotificationEvent event){
			logger.debug("DeltaFileConsumer received manager creation notification (idxID: " + resource.getIndexID() + ").");
			
			Element message = null;
			try {
				message = (Element) event.getPayload().getMessageObject();
				DeltaListManagerCreatedNotificationMessageType notification = (DeltaListManagerCreatedNotificationMessageType) ObjectDeserializer.toObject(message, DeltaListManagerCreatedNotificationMessageType.class);
				handleManagerCreationNotification(notification);
			}  catch (Exception e) {
				logger.error("Error while deserializing manager creation notification.", e);
			}
		}
	}
}
