/**
 * 
 */
package org.gcube.informationsystem.resourceregistry.client.proxy;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.UUID;

import javax.xml.ws.EndpointReference;

import org.gcube.common.authorization.client.Constants;
import org.gcube.common.authorization.library.provider.SecurityTokenProvider;
import org.gcube.common.clients.Call;
import org.gcube.common.clients.delegates.AsyncProxyDelegate;
import org.gcube.common.clients.delegates.ProxyDelegate;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.informationsystem.impl.utils.ISMapper;
import org.gcube.informationsystem.model.ER;
import org.gcube.informationsystem.model.ISManageable;
import org.gcube.informationsystem.model.entity.Entity;
import org.gcube.informationsystem.model.entity.Resource;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.er.ERException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.er.ERNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.query.InvalidQueryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.schema.SchemaException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.schema.SchemaNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.rest.AccessPath;
import org.gcube.informationsystem.types.TypeBinder;
import org.gcube.informationsystem.types.TypeBinder.TypeDefinition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 * 
 */
public class ResourceRegistryClientImpl implements ResourceRegistryClient {

	private static final Logger logger = LoggerFactory
			.getLogger(ResourceRegistryClientImpl.class);

	private final AsyncProxyDelegate<EndpointReference> delegate;

	public static final String PATH_SEPARATOR = "/";

	public final class RREntry<K, V> implements Entry<K, V> {

		private final K key;
		private V value;

		public RREntry(K key, V value) {
			this.key = key;
			this.value = value;
		}

		@Override
		public K getKey() {
			return key;
		}

		@Override
		public V getValue() {
			return value;
		}

		@Override
		public V setValue(V value) {
			V old = this.value;
			this.value = value;
			return old;
		}
	}

	public ResourceRegistryClientImpl(ProxyDelegate<EndpointReference> config) {
		this.delegate = new AsyncProxyDelegate<EndpointReference>(config);
	}

	protected enum HTTPMETHOD {
		GET, POST, PUT, DELETE;

		@Override
		public String toString() {
			return this.name();
		}
	}

	class HTTPInputs {

		public static final String PARAM_STARTER = "?";
		public static final String PARAM_EQUALS = "=";
		public static final String PARAM_SEPARATOR = "&";
		public static final String UTF8 = "UTF-8";

		protected final String path;
		protected final HTTPMETHOD method;
		protected final String urlParameters;

		protected String getParametersDataString(
				List<Map.Entry<String, String>> parameters)
				throws UnsupportedEncodingException {
			if (parameters == null) {
				return null;
			}

			StringBuilder result = new StringBuilder();
			boolean first = true;
			for (Entry<String, String> entry : parameters) {
				if (first) {
					first = false;
				} else {
					result.append(PARAM_SEPARATOR);
				}

				result.append(URLEncoder.encode(entry.getKey(), UTF8));
				result.append(PARAM_EQUALS);
				result.append(URLEncoder.encode(entry.getValue(), UTF8));

			}

			return result.toString();
		}

		/**
		 * @param path
		 * @param method
		 * @param requestProperties
		 * @throws UnsupportedEncodingException
		 */
		public HTTPInputs(String path, HTTPMETHOD method,
				List<Entry<String, String>> parameters)
				throws UnsupportedEncodingException {
			super();
			this.path = path;
			this.method = method;
			this.urlParameters = getParametersDataString(parameters);
		}

		/**
		 * @return the path
		 */
		public String getPath() {
			return path;
		}

		/**
		 * @return the method
		 */
		public HTTPMETHOD getMethod() {
			return method;
		}

		/**
		 * @return the urlParameters
		 */
		public String getUrlParameters() {
			return urlParameters;
		}

	}

	class ResourceRegistryClientCall<C> implements Call<EndpointReference, C> {

		protected final Class<C> clazz;
		protected final HTTPInputs httpInputs;

		public ResourceRegistryClientCall(Class<C> clazz, HTTPInputs httpInputs) {
			this.clazz = clazz;
			this.httpInputs = httpInputs;
		}

		protected String getURLStringFromEndpointReference(
				EndpointReference endpoint) throws IOException {
			JaxRSEndpointReference jaxRSEndpointReference = new JaxRSEndpointReference(
					endpoint);
			return jaxRSEndpointReference.toString();
		}

		protected HttpURLConnection getConnection(URL url, HTTPMETHOD method)
				throws Exception {
			/*
			 * if(method!=HTTPMETHOD.POST &&
			 * httpInputs.getUrlParameters()!=null){
			 */
			url = new URL(url + "?" + httpInputs.getUrlParameters());
			// }

			HttpURLConnection connection = (HttpURLConnection) url
					.openConnection();
			if (SecurityTokenProvider.instance.get() == null) {
				if (ScopeProvider.instance.get() == null) {
					throw new RuntimeException(
							"Null Token and Scope. Please set your token first.");
				}
				connection.setRequestProperty("gcube-scope",
						ScopeProvider.instance.get());
			} else {
				connection.setRequestProperty(Constants.TOKEN_HEADER_ENTRY,
						SecurityTokenProvider.instance.get());
			}
			connection.setDoOutput(true);

			connection.setRequestProperty("Content-type", "application/json");
			connection.setRequestProperty("User-Agent",
					ResourceRegistryClient.class.getSimpleName());

			connection.setRequestMethod(method.toString());

			/*
			 * if(method==HTTPMETHOD.POST){ connection.setDoOutput(true);
			 * DataOutputStream wr = new
			 * DataOutputStream(connection.getOutputStream());
			 * wr.writeBytes(httpInputs.getUrlParameters()); wr.flush();
			 * wr.close(); }
			 */

			return connection;
		}

		@SuppressWarnings("unchecked")
		@Override
		public C call(EndpointReference endpoint) throws Exception {
			String urlFromEndpointReference = getURLStringFromEndpointReference(endpoint);
			StringBuilder callUrl = new StringBuilder(urlFromEndpointReference);
			callUrl.append(httpInputs.getPath());

			URL url = new URL(callUrl.toString());
			HttpURLConnection connection = getConnection(url, httpInputs.method);

			logger.debug("Response code for {} is {} : {}",
					connection.getURL(), connection.getResponseCode(),
					connection.getResponseMessage());

			if (connection.getResponseCode() != HttpURLConnection.HTTP_OK) {
				throw new Exception(
						"Error Contacting Resource Registry Service");
			}

			StringBuilder result = new StringBuilder();
			try (BufferedReader reader = new BufferedReader(
					new InputStreamReader((InputStream) connection.getContent()))) {
				String line;
				while ((line = reader.readLine()) != null) {
					result.append(line);
				}
			}

			String res = result.toString();
			logger.trace("Server returned content : {}", res);

			if(Boolean.class.isAssignableFrom(clazz)){
				return (C) ((Boolean) Boolean.valueOf(res)) ;
			}else if(ISManageable.class.isAssignableFrom(clazz)){
				return (C) ISMapper.unmarshal((Class<ISManageable>) clazz, res);
			}
			
			return (C) res;
		}

	}

	@Override
	public <ERType extends ER> ERType getInstance(Class<ERType> clazz, UUID uuid)
			throws ERNotFoundException, ResourceRegistryException {
		String type = clazz.getSimpleName();
		try {
			logger.info("Going to get {} with UUID {}", type, uuid);
			StringWriter stringWriter = new StringWriter();
			stringWriter.append(PATH_SEPARATOR);
			stringWriter.append(AccessPath.ACCESS_PATH_PART);
			stringWriter.append(PATH_SEPARATOR);
			stringWriter.append(AccessPath.INSTANCE_PATH_PART);
			stringWriter.append(PATH_SEPARATOR);
			stringWriter.append(type);
			stringWriter.append(PATH_SEPARATOR);
			stringWriter.append(uuid.toString());

			HTTPInputs httpInputs = new HTTPInputs(stringWriter.toString(),
					HTTPMETHOD.GET, null);

			ResourceRegistryClientCall<ERType> call = new ResourceRegistryClientCall<>(
					clazz, httpInputs);

			ERType erType = delegate.make(call);
			logger.info("Got {} with UUID {} is {}", type, uuid, erType);
			return erType;

		} catch (Exception e) {
			logger.error("Error while getting {} with UUID {}", type, uuid, e);
			throw new ResourceRegistryException(e);
		}
	}

	@Override
	public List<? extends Entity> getInstances(String type,
			Boolean polymorphic) throws ERNotFoundException,
			ResourceRegistryException {
		try {
			logger.info("Going to get all instances of {} ", type);
			StringWriter stringWriter = new StringWriter();
			stringWriter.append(PATH_SEPARATOR);
			stringWriter.append(AccessPath.ACCESS_PATH_PART);
			stringWriter.append(PATH_SEPARATOR);
			stringWriter.append(AccessPath.INSTANCES_PATH_PART);
			stringWriter.append(PATH_SEPARATOR);
			stringWriter.append(type);

			List<Entry<String, String>> parameters = new ArrayList<>();
			parameters.add(new RREntry<String, String>(
					AccessPath.POLYMORPHIC_PARAM, polymorphic.toString()));

			HTTPInputs httpInputs = new HTTPInputs(stringWriter.toString(),
					HTTPMETHOD.GET, parameters);

			ResourceRegistryClientCall<String> call = new ResourceRegistryClientCall<>(
					String.class, httpInputs);

			String ret = delegate.make(call);
			logger.info("Got instances of {} are {}", type, ret);

			return ISMapper.unmarshalList(Entity.class, ret);

		} catch (Exception e) {
			logger.error("Error while getting {} instances", type, e);
			throw new ResourceRegistryException(e);
		}
	}

	@Override
	public List<Resource> getInstancesFromEntity(
			String relationType, Boolean polymorphic, UUID reference,
			Direction direction) throws ERException, SchemaException,
			ResourceRegistryException {
		try {
			logger.info("Going to get all instances of {} from/to {}", relationType,
					reference.toString());
			StringWriter stringWriter = new StringWriter();
			stringWriter.append(PATH_SEPARATOR);
			stringWriter.append(AccessPath.ACCESS_PATH_PART);
			stringWriter.append(PATH_SEPARATOR);
			stringWriter.append(AccessPath.INSTANCES_PATH_PART);
			stringWriter.append(PATH_SEPARATOR);
			stringWriter.append(relationType);

			List<Entry<String, String>> parameters = new ArrayList<>();
			parameters.add(new RREntry<String, String>(
					AccessPath.POLYMORPHIC_PARAM, polymorphic.toString()));
			parameters.add(new RREntry<String, String>(AccessPath.REFERENCE,
					reference.toString()));
			parameters.add(new RREntry<String, String>(AccessPath.DIRECTION,
					direction.toString()));

			HTTPInputs httpInputs = new HTTPInputs(stringWriter.toString(),
					HTTPMETHOD.GET, parameters);

			ResourceRegistryClientCall<String> call = new ResourceRegistryClientCall<>(
					String.class, httpInputs);

			String ret = delegate.make(call);
			logger.info("Got instances of {} from/to {} are {}", relationType,
					reference.toString(), ret);

			return ISMapper.unmarshalList(Resource.class, ret);

		} catch (Exception e) {
			logger.error("Error while getting instances of {} from/to {}", relationType, e);
			throw new ResourceRegistryException(e);
		}
	}

	@Override
	public <ISM extends ISManageable> List<TypeDefinition> getSchema(
			Class<ISM> clazz, Boolean polymorphic)
			throws SchemaNotFoundException {

		String type = clazz.getSimpleName();
		try {
			logger.info("Going to get {} schema", type);
			StringWriter stringWriter = new StringWriter();
			stringWriter.append(PATH_SEPARATOR);
			stringWriter.append(AccessPath.ACCESS_PATH_PART);
			stringWriter.append(PATH_SEPARATOR);
			stringWriter.append(AccessPath.SCHEMA_PATH_PART);
			stringWriter.append(PATH_SEPARATOR);
			stringWriter.append(type);

			List<Entry<String, String>> parameters = new ArrayList<>();
			parameters.add(new RREntry<String, String>(
					AccessPath.POLYMORPHIC_PARAM, polymorphic.toString()));

			HTTPInputs httpInputs = new HTTPInputs(stringWriter.toString(),
					HTTPMETHOD.GET, parameters);

			ResourceRegistryClientCall<String> call = new ResourceRegistryClientCall<>(
					String.class, httpInputs);
			String schema = delegate.make(call);
			logger.info("Got schema for {} is {}", type, schema);

			return TypeBinder.deserializeTypeDefinitions(schema);

		} catch (Exception e) {
			logger.error("Error while getting {}schema for {}",
					polymorphic ? AccessPath.POLYMORPHIC_PARAM + " " : "",
					type, e);
			throw new SchemaNotFoundException(e);
		}
	}

	@Override
	public String query(String query, int limit, String fetchPlan)
			throws InvalidQueryException {
		ResourceRegistryQuery rrq = new ResourceRegistryQuery(delegate);
		return rrq.query(query, limit, fetchPlan);
	}

}
