package org.gcube.informationsystem.resourceregistry.client;

import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.databind.JavaType;
import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.com.fasterxml.jackson.databind.ObjectMapper;
import org.gcube.common.gxhttp.reference.GXConnection;
import org.gcube.common.gxhttp.request.GXHTTPStringRequest;
import org.gcube.common.security.providers.SecretManagerProvider;
import org.gcube.common.security.secrets.Secret;
import org.gcube.informationsystem.base.reference.Direction;
import org.gcube.informationsystem.contexts.reference.entities.Context;
import org.gcube.informationsystem.model.knowledge.ModelKnowledge;
import org.gcube.informationsystem.model.reference.ERElement;
import org.gcube.informationsystem.model.reference.ModelElement;
import org.gcube.informationsystem.model.reference.entities.Entity;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.model.reference.relations.IsRelatedTo;
import org.gcube.informationsystem.model.reference.relations.Relation;
import org.gcube.informationsystem.queries.templates.reference.entities.QueryTemplate;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextCache;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextCacheRenewal;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextUtility;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.contexts.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.queries.InvalidQueryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.queries.templates.QueryTemplateNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.types.SchemaNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.request.BaseRequestInfo;
import org.gcube.informationsystem.resourceregistry.api.rest.AccessPath;
import org.gcube.informationsystem.resourceregistry.api.rest.ContextPath;
import org.gcube.informationsystem.resourceregistry.api.rest.InstancePath;
import org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath;
import org.gcube.informationsystem.resourceregistry.api.rest.TypePath;
import org.gcube.informationsystem.resourceregistry.api.rest.httputils.HTTPUtility;
import org.gcube.informationsystem.serialization.ElementMapper;
import org.gcube.informationsystem.tree.Node;
import org.gcube.informationsystem.types.TypeMapper;
import org.gcube.informationsystem.types.knowledge.TypeInformation;
import org.gcube.informationsystem.types.knowledge.TypesKnowledge;
import org.gcube.informationsystem.types.reference.Type;
import org.gcube.informationsystem.utils.TypeUtility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class ResourceRegistryClientImpl extends BaseRequestInfo implements ResourceRegistryClient {
	
	private static final Logger logger = LoggerFactory.getLogger(ResourceRegistryClientImpl.class);
	
	private static final String ACCEPT_HTTP_HEADER_KEY = "Accept";
	private static final String CONTENT_TYPE_HTTP_HEADER_KEY = "Content-Type";
	
	/** The base address of the Resource Registry service */
	protected final String address;
	
	/** Additional HTTP headers to be included in requests */
	protected Map<String, String> headers;
	
	/** Cache for context information */
	protected ContextCache contextCache;
	
	/** Knowledge base for type information */
	protected TypesKnowledge typesKnowledge;
	
	private void addOptionalQueryParameters(Map<String,String> queryParams) throws UnsupportedEncodingException {
		addHierarchicalMode(queryParams);
		addIncludeContexts(queryParams);
		addIncludeMeta(queryParams);
		addIncludeAllMeta(queryParams);
	}
	
	private GXHTTPStringRequest includeAdditionalQueryParameters(GXHTTPStringRequest gxHTTPStringRequest) throws UnsupportedEncodingException{
		Map<String,String> queryParams = new HashMap<>();
		return includeAdditionalQueryParameters(gxHTTPStringRequest, queryParams);
	}
	
	private GXHTTPStringRequest includeAdditionalQueryParameters(GXHTTPStringRequest gxHTTPStringRequest, Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(queryParams==null) {
			queryParams = new HashMap<>();
		}
		addOptionalQueryParameters(queryParams);
		return gxHTTPStringRequest.queryParams(queryParams);
	}
	
	private void addHierarchicalMode(Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(hierarchicalMode) {
			queryParams.put(AccessPath.HIERARCHICAL_MODE_QUERY_PARAMETER, Boolean.toString(hierarchicalMode));
		}
	}
	
	private void addIncludeContexts(Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(includeContexts) {
			queryParams.put(AccessPath.INCLUDE_CONTEXTS_QUERY_PARAMETER, Boolean.toString(includeContexts));
		}
	}
	
	private void addIncludeMeta(Map<String,String> queryParams) throws UnsupportedEncodingException{
		addIncludeMeta(queryParams, includeMeta);
	}
	
	private void addIncludeMeta(Map<String,String> queryParams, boolean includeMeta) throws UnsupportedEncodingException{
		if(includeMeta) {
			queryParams.put(AccessPath.INCLUDE_META_QUERY_PARAMETER, Boolean.toString(includeMeta));
		}
	}
	
	private void addIncludeAllMeta(Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(allMeta) {
			queryParams.put(AccessPath.INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER, Boolean.toString(allMeta));
		}
	}
	
	private void addOffset(Map<String,String> queryParams) throws UnsupportedEncodingException{
		addOffset(queryParams, offset);
	}
	
	private void addOffset(Map<String,String> queryParams, Integer offset) throws UnsupportedEncodingException{
		if(offset!=null) {
			queryParams.put(AccessPath.OFFSET_QUERY_PARAMETER, offset.toString());
		}
	}
	
	private void addLimit(Map<String,String> queryParams) throws UnsupportedEncodingException{
		addLimit(queryParams, limit);
	}
	
	private void addLimit(Map<String,String> queryParams, Integer limit) throws UnsupportedEncodingException{
		if(limit!=null) {
			queryParams.put(AccessPath.LIMIT_QUERY_PARAMETER, limit.toString());
		}
	}
	
	/** Context cache renewal strategy that reads all contexts from the server */
	protected ContextCacheRenewal contextCacheRenewal = new ContextCacheRenewal() {
		
		@Override
		public List<Context> renew() throws ResourceRegistryException {
			return getAllContextFromServer(true, 0, BaseRequestInfo.UNBOUNDED_LIMIT);
		}
		
	};

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addHeader(String name, String value) {
		headers.put(name, value);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addHeader(String name, boolean value) {
		addHeader(name, Boolean.toString(value));
	}
	
	/**
	 * Creates and configures a new HTTP request with authorization headers and custom headers.
	 * 
	 * @return A configured GXHTTPStringRequest instance ready for use
	 */
	protected GXHTTPStringRequest getGXHTTPStringRequest() {
		GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
		
		/* It is a gcube request */
		Secret secret = SecretManagerProvider.get();
		Map<String, String> authorizationHeaders = secret.getHTTPAuthorizationHeaders();
		for(String key : authorizationHeaders.keySet()) {
			gxHTTPStringRequest.header(key, authorizationHeaders.get(key));
		}
		
		gxHTTPStringRequest.from(this.getClass().getSimpleName());
		for(String name : headers.keySet()) {
			gxHTTPStringRequest.header(name, headers.get(name));
		}

		return gxHTTPStringRequest;
	}
	
	/**
	 * Creates a new ResourceRegistryClientImpl with default configuration.
	 * Uses shared context cache and shared model knowledge.
	 * 
	 * @param address the base address of the Resource Registry service
	 */
	public ResourceRegistryClientImpl(String address) {
		this(address, true, true);
	}
	
	/**
	 * Creates a new ResourceRegistryClientImpl with configurable cache and knowledge sharing.
	 * 
	 * @param address the base address of the Resource Registry service
	 * @param sharedContextCache true to use shared context cache, false for dedicated cache
	 * @param sharedModelKnowledge true to use shared model knowledge, false for dedicated knowledge
	 */
	public ResourceRegistryClientImpl(String address, boolean sharedContextCache, boolean sharedModelKnowledge) {
		super();
		this.address = address;
		this.headers = new HashMap<>();
		if(sharedContextCache) {
			this.contextCache = ContextCache.getInstance();
		}else {
			this.contextCache = new ContextCache();
		}
		this.contextCache.setContextCacheRenewal(contextCacheRenewal);
		
		if(sharedModelKnowledge) {
			this.typesKnowledge = TypesKnowledge.getInstance();
		}else {
			this.typesKnowledge = new TypesKnowledge();
		}
		typesKnowledge.setTypesDiscoverer(new RRCTypesDiscoverer(this));
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ContextCache getContextCache() {
		return contextCache;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Context> getAllContexts() throws ResourceRegistryException {
		return contextCache.getContexts();
	}
	
	/**
	 * Retrieves the list of all contexts in the system.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/contexts[?limit={limit}&offset={offset}&includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Returns all contexts that are accessible to the current user</li>
	 * <li>Results are filtered based on user authorization and context access permissions</li>
	 * <li>Supports pagination and metadata inclusion based on client configuration</li>
	 * <li>Returns strongly-typed Context objects with role-appropriate detail level.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters</strong>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of instances to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 instances per request</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of instances to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 instances (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>No input parameters required for this method</li>
	 * <li>All contexts accessible to the current user are returned based on authorization level</li>
	 * <li>Context filtering is automatically applied based on user permissions.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Results include only contexts that the current user has permission to access</li>
	 * <li>IS-Manager and Infrastructure-Manager see all contexts in the system</li>
	 * <li>Context-Manager sees contexts within their management scope</li>
	 * <li>Other users see contexts they have explicit access to.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and basic read permissions to list contexts they have access to.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Basic usage - get all contexts with default settings
	 * List&lt;Context&gt; contexts = client.getContexts();
	 * 
	 * // With metadata for non-admin users
	 * client.setIncludeMeta(true);
	 * List&lt;Context&gt; contextsWithMeta = client.getContexts();
	 * 
	 * // With pagination
	 * client.setLimit(20);
	 * client.setOffset(10);
	 * List&lt;Context&gt; paginatedContexts = client.getContexts();
	 * </pre>
	 * 
	 * @return List of Context objects with role-appropriate detail level accessible to the current user
	 * @throws ResourceRegistryException If an error occurs during context retrieval or if there are authorization issues
	 */
	public List<Context> getAllContextFromServer() throws ResourceRegistryException {
		return getAllContextFromServer(includeMeta, offset, limit);
	}
	
	/**
	 * It reads all the contexts from server with specific parameters. 
	 * The cache used for contexts is bypassed and not updated.
	 * 
	 * @param includeMeta whether to include metadata in the response
	 * @param offset the starting position for pagination
	 * @param limit the maximum number of contexts to return
	 * @return All Contexts read from server
	 * @throws ResourceRegistryException if an error occurs during the retrieval
	 */
	protected List<Context> getAllContextFromServer(boolean includeMeta, Integer offset, Integer limit) throws ResourceRegistryException {
		try {
			logger.info("Going to read all {}s", Context.NAME);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.CONTEXTS_PATH_PART);
			
			Map<String,String> parameters = new HashMap<>();
			addIncludeMeta(parameters, includeMeta);
			addOffset(parameters, offset);
			addLimit(parameters, limit);
			gxHTTPStringRequest.queryParams(parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got Contexts are {}", ret);
			return ElementMapper.unmarshalList(Context.class, ret);
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {}schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public ModelKnowledge<Type, TypeInformation> getModelKnowledge() {
		return typesKnowledge.getModelKnowledge();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void renewModelKnowledge() {
		typesKnowledge.renew();
	}

	/**
	 * It reads the context from server by UUID.
	 * 
	 * The cache used for contexts is bypassed and not updated.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/contexts/{context-uuid}[?includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves the context with the specified UUID from the system</li>
	 * <li>Returns the complete context object with all accessible properties</li>
	 * <li>Direct UUID-based context retrieval without additional processing.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Accepts a properly formatted UUID object</li>
	 * <li>No additional processing or conversion required</li>
	 * <li>Direct UUID-based context retrieval.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The context must be accessible to the current user based on authorization level</li>
	 * <li>IS-Manager and Infrastructure-Manager can retrieve any context without restrictions</li>
	 * <li>Context-Manager can retrieve contexts within their management scope</li>
	 * <li>Other users can retrieve contexts they have explicit access to.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context to retrieve it.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * UUID contextUuid = UUID.fromString("c0f314e7-2807-4241-a792-2a6c79ed4fd0");
	 * 
	 * // Basic usage - read context
	 * Context context = client.getContext(contextUuid);
	 * System.out.println("Context name: " + context.getName());
	 * 
	 * // With metadata for non-admin users
	 * client.setIncludeMeta(true);
	 * Context contextWithMeta = client.getContext(contextUuid);
	 * </pre>
	 * 
	 * @param uuid the UUID of the context to retrieve
	 * @return the Contexts read from server
	 * @throws ContextNotFoundException if the context is not found
	 * @throws ResourceRegistryException if an error occurs during the retrieval
	 */
	public Context getContextFromServer(UUID uuid) throws ContextNotFoundException, ResourceRegistryException {
		return getContextFromServer(uuid.toString());
	}
	
	/**
	 * It reads the context from server by UUID string. 
	 * The cache used for contexts is bypassed and not updated.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/contexts/{context-uuid}[?includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves the context with the specified UUID from the system</li>
	 * <li>Returns the complete context object with all accessible properties</li>
	 * <li>Direct UUID-based context retrieval without additional processing.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Accepts a properly formatted UUID object</li>
	 * <li>No additional processing or conversion required</li>
	 * <li>Direct UUID-based context retrieval.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The context must be accessible to the current user based on authorization level</li>
	 * <li>IS-Manager and Infrastructure-Manager can retrieve any context without restrictions</li>
	 * <li>Context-Manager can retrieve contexts within their management scope</li>
	 * <li>Other users can retrieve contexts they have explicit access to.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context to retrieve it.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * UUID contextUuid = UUID.fromString("c0f314e7-2807-4241-a792-2a6c79ed4fd0");
	 * 
	 * // Basic usage - read context
	 * Context context = client.getContext(contextUuid);
	 * System.out.println("Context name: " + context.getName());
	 * 
	 * // With metadata for non-admin users
	 * client.setIncludeMeta(true);
	 * Context contextWithMeta = client.getContext(contextUuid);
	 * </pre>
	 * 
	 * @param uuid the UUID string of the context to retrieve
	 * @return the Context read from server
	 * @throws ContextNotFoundException if the context is not found
	 * @throws ResourceRegistryException if an error occurs during the retrieval
	 */
	protected Context getContextFromServer(String uuid) throws ContextNotFoundException, ResourceRegistryException {
		try {
			logger.info("Going to get current {} ", Context.NAME);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(uuid);
			
			Map<String,String> parameters = new HashMap<>();
			addIncludeMeta(parameters);
			gxHTTPStringRequest.queryParams(parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			Context context = HTTPUtility.getResponse(Context.class, httpURLConnection);
			
			logger.debug("Got Context is {}", ElementMapper.marshal(context));
			return context;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {}schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean existContext(String uuid) throws ResourceRegistryException {
		return existContext(UUID.fromString(uuid));
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean existContext(UUID uuid) throws ResourceRegistryException {
		try {
			getContext(uuid);
			return true;
		}catch (ContextNotFoundException e) {
			return false;
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Context getContext(String uuid) throws ContextNotFoundException, ResourceRegistryException {
		return getContext(UUID.fromString(uuid));
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Context getContext(UUID uuid) throws ContextNotFoundException, ResourceRegistryException {
		Context context = contextCache.getContextByUUID(uuid);;
		if(context == null) {
			context = getContextFromServer(ContextPath.CURRENT_CONTEXT_PATH_PART);
			contextCache.cleanCache();
			contextCache.refreshContextsIfNeeded();
			Context c = contextCache.getContextByUUID(context.getID());
			if(c!=null){
				context = c;
			}else {
				logger.error("Context with UUID {} is {}. It is possibile to get it from the server but not from the cache. This is very strange and should not occur.", uuid, context);
			}
		}
		return context;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Context getCurrentContext() throws ContextNotFoundException, ResourceRegistryException {
		String contextFullName = SecretManagerProvider.get().getContext();
		UUID uuid = contextCache.getUUIDByFullName(contextFullName);
		Context context = null;
		if(uuid == null) {
			context = getContextFromServer(ContextPath.CURRENT_CONTEXT_PATH_PART);
			contextCache.cleanCache();
			contextCache.refreshContextsIfNeeded();
			Context c = contextCache.getContextByUUID(context.getID());
			if(c!=null){
				context = c;
			}else {
				logger.error("Current Context is {}. It is possibile to get it from the server but not from the cache. This is very strange and should not occur.", contextFullName);
			}
		}else {
			context = contextCache.getContextByUUID(uuid);
		}
		return context;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public <ME extends ModelElement> boolean existType(Class<ME> clazz) throws ResourceRegistryException {
		return existType(TypeUtility.getTypeName(clazz));
	}	
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean existType(String typeName) throws ResourceRegistryException {
		try {
			return typesKnowledge.getModelKnowledge().getTypeByName(typeName) != null;
		}catch (RuntimeException e) {
			return false;
		}
	}
	
	/**
	 * Checks if a type exists by querying the server directly, bypassing the local knowledge cache.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code HEAD /access/types/{type-name}[?polymorphic={true|false}&includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Verifies the existence of the specified type definition using its name</li>
	 * <li>Does not return the type data, only confirms existence</li>
	 * <li>Checks accessibility within the current user authorization level</li>
	 * <li>Most direct method when type name is already known.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Type exists and is accessible</li>
	 * <li><strong>404 Not Found</strong>: Type does not exist in the system</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to access type information</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Type name must be a valid Information System model type</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes)</li>
	 * <li>Type name is case-sensitive and must match exactly.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can check existence of any type definition</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can check existence of any type definition</li>
	 * <li><strong>All Other Users</strong>: Can check existence of any type definition with filtered access to sensitive metadata.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Check if ContactFacet type exists
	 * boolean contactFacetExists = client.existType("ContactFacet");
	 * 
	 * // Check if a custom resource type exists
	 * boolean customResourceExists = client.existType("MyCustomResource");
	 * </pre>
	 * 
	 * @param typeName the name of the type to check
	 * @return true if the type exists on the server, false otherwise
	 * @throws ResourceRegistryException if an error occurs during the check
	 */
	public boolean existTypeFromServer(String typeName) throws ResourceRegistryException {
		try {
			logger.info("Going to get {} schema", typeName);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.TYPES_PATH_PART);
			gxHTTPStringRequest.path(typeName);
			
			Map<String,String> parameters = new HashMap<>();
			parameters.put(InstancePath.POLYMORPHIC_QUERY_PARAMETER, Boolean.FALSE.toString());
			gxHTTPStringRequest.queryParams(parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.head();
			HTTPUtility.getResponse(String.class, httpURLConnection);

			return true;
		} catch (NotFoundException e) {
			return false;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} schema for {}", polymorphic ? AccessPath.POLYMORPHIC_PARAM + " " : "",
			//		type, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {}schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw new ResourceRegistryException(e);
		}
	}
	
	/**
	 * Gets type information from the local types knowledge cache.
	 * 
	 * @param <ERElem> the type parameter extending ERElement
	 * @param typeName the name of the type to retrieve
	 * @param polymorphic whether to include polymorphic types
	 * @return a list of Type objects
	 * @throws SchemaNotFoundException if the type schema is not found
	 * @throws ResourceRegistryException if an error occurs during retrieval
	 */
	public <ERElem extends ERElement> List<Type> getTypeFromTypesKnowledge(String typeName, Boolean polymorphic)
			throws SchemaNotFoundException, ResourceRegistryException {
		return getTypeFromTypesKnowledge(typeName, polymorphic, -1);
	}
	
	/**
	 * Gets type information from the local types knowledge cache with level limitation.
	 * 
	 * @param <ERElem> the type parameter extending ERElement
	 * @param typeName the name of the type to retrieve
	 * @param level the depth level for type hierarchy traversal
	 * @return a list of Type objects
	 * @throws SchemaNotFoundException if the type schema is not found
	 * @throws ResourceRegistryException if an error occurs during retrieval
	 */
	public <ERElem extends ERElement> List<Type> getTypeFromTypesKnowledge(String typeName, int level)
			throws SchemaNotFoundException, ResourceRegistryException {
		return getTypeFromTypesKnowledge(typeName, true, level);
	}
	
	/**
	 * Recursively adds child types to a list based on type hierarchy.
	 * 
	 * @param node the current node in the type hierarchy
	 * @param types the list to add types to
	 * @param currentLevel the current depth level
	 * @param maxLevel the maximum depth level to traverse (-1 for unlimited)
	 * @return the updated list of types
	 */
	protected List<Type> addChildren(Node<Type> node, List<Type> types, int currentLevel, int maxLevel) {
		if(maxLevel>=0 && maxLevel <= currentLevel) {
			return types;
		}
		
		Set<Node<Type>> children = node.getChildren();
		if(children!=null && children.size()>0) {
			for(Node<Type> child : children) {
				types.add(child.getNodeElement());
				types = addChildren(child, types, ++currentLevel, maxLevel);
			}
		}
		
		return types;
	}
	
	/**
	 * Gets type information from the local types knowledge cache with full configuration.
	 * 
	 * @param <ERElem> the type parameter extending ERElement
	 * @param typeName the name of the type to retrieve
	 * @param polymorphic whether to include polymorphic types
	 * @param level the depth level for type hierarchy traversal (-1 for unlimited)
	 * @return a list of Type objects
	 * @throws SchemaNotFoundException if the type schema is not found
	 * @throws ResourceRegistryException if an error occurs during retrieval
	 */
	public <ERElem extends ERElement> List<Type> getTypeFromTypesKnowledge(String typeName, Boolean polymorphic, int level)
			throws SchemaNotFoundException, ResourceRegistryException {
		
		Node<Type> node = getTypeTreeNode(typeName);
		
		List<Type> types = new ArrayList<>();
		types.add(node.getNodeElement());
		
		if (polymorphic) {
			addChildren(node, types, 0, level);
		} 
		
		return types;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getType(String typeName, Boolean polymorphic) throws SchemaNotFoundException, ResourceRegistryException {
		try { 
			List<Type> types = getTypeFromTypesKnowledge(typeName, polymorphic);
			return TypeMapper.serializeTypeDefinitions(types);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getType(String typeName, int level) throws SchemaNotFoundException, ResourceRegistryException {
		try { 
			List<Type> types = getTypeFromTypesKnowledge(typeName, level);
			return TypeMapper.serializeTypeDefinitions(types);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public <ME extends ModelElement> List<Type> getType(Class<ME> clazz, Boolean polymorphic)
			throws SchemaNotFoundException, ResourceRegistryException {
		try {
			String typeName = TypeUtility.getTypeName(clazz);
			return getTypeFromTypesKnowledge(typeName, polymorphic);
		} catch (Exception e) {
			throw new ResourceRegistryException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <ME extends ModelElement> List<Type> getType(Class<ME> clazz, int level)
			throws SchemaNotFoundException, ResourceRegistryException {
		try {
			String typeName = TypeUtility.getTypeName(clazz);
			return getTypeFromTypesKnowledge(typeName, level);
		} catch (Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Node<Type> getTypeTreeNode(String typeName) throws SchemaNotFoundException, ResourceRegistryException {
		try { 
			Node<Type> node = null;
			try {
				node = typesKnowledge.getModelKnowledge().getNodeByName(typeName);
			} catch (RuntimeException e) {
				throw new SchemaNotFoundException(e);
			}
			return node;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <ME extends ModelElement> Node<Type> getTypeTreeNode(Class<ME> clazz)
			throws SchemaNotFoundException, ResourceRegistryException {
		try {
			String typeName = TypeUtility.getTypeName(clazz);
			return getTypeTreeNode(typeName);
		} catch (Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
	
	/**
	 * Gets type information directly from the server, bypassing the local cache.
	 * 
	 * @param <ME> the type parameter extending ModelElement
	 * @param clazz the class representing the type to retrieve
	 * @param polymorphic whether to include polymorphic types
	 * @return a list of Type objects
	 * @throws SchemaNotFoundException if the type schema is not found
	 * @throws ResourceRegistryException if an error occurs during retrieval
	 */
	public <ME extends ModelElement> List<Type> getTypeFromServer(Class<ME> clazz, Boolean polymorphic)
			throws SchemaNotFoundException, ResourceRegistryException {
		try {
			String json = getTypeFromServer(TypeUtility.getTypeName(clazz), polymorphic);
			return TypeMapper.deserializeTypeDefinitions(json);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
	
	/**
	 * Gets type schema directly from the server as a JSON string.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/types/{type-name}[?polymorphic={true|false}&includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves type schema definition(s) for the specified type name</li>
	 * <li>When polymorphic=false: returns only the specified type definition</li>
	 * <li>When polymorphic=true: returns the specified type AND all its existing subtypes</li>
	 * <li>Metadata inclusion is controlled by client configuration and user authorization</li>
	 * <li>Response detail level varies based on user authorization (IS/Infrastructure managers get full metadata).</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Type successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Type with the specified name does not exist</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to access type information.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Receive complete type definitions with full metadata when requested</li>
	 * <li><strong>Infrastructure-Manager</strong>: Receive complete type definitions with full metadata when requested</li>
	 * <li><strong>All Other Users</strong>: Receive basic type information with filtered metadata (sensitive information obfuscated).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Read a ContactFacet type definition only
	 * String contactFacetSchema = client.getType("ContactFacet", false);
	 * 
	 * // Read ContactFacet and all its subtypes
	 * String contactFacetAndSubtypes = client.getType("ContactFacet", true);
	 * </pre> 
	 * 
	 * @param typeName the name of the type to retrieve
	 * @param polymorphic Whether to include instances of subtypes (true) or only exact type instances (false). 
	 *                    When true, includes all subtypes of the specified type; when false, returns only instances 
	 *                    of the exact specified type. This parameter is automatically set in the {@code polymorphic}
	 *                    query parameter of the REST API call.
	 * @return the type schema as a JSON string
	 * @throws SchemaNotFoundException if the type schema is not found
	 * @throws ResourceRegistryException if an error occurs during retrieval
	 */
	public String getTypeFromServer(String typeName, Boolean polymorphic)
			throws SchemaNotFoundException, ResourceRegistryException {
		try {
			logger.info("Going to get {} schema", typeName);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.TYPES_PATH_PART);
			gxHTTPStringRequest.path(typeName);
			
			Map<String,String> parameters = new HashMap<>();
			addIncludeMeta(parameters);
			parameters.put(TypePath.POLYMORPHIC_QUERY_PARAMETER, polymorphic.toString());
			gxHTTPStringRequest.queryParams(parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String json = HTTPUtility.getResponse(String.class, httpURLConnection);

			logger.debug("Got schema for {} is {}", typeName, json);
			return json;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} schema for {}", polymorphic ? AccessPath.POLYMORPHIC_PARAM + " " : "",
			//		type, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {}schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <ERElem extends ERElement> List<ERElem> getInstances(Class<ERElem> clazz, Boolean polymorphic)
			throws ResourceRegistryException {
		String type = TypeUtility.getTypeName(clazz);
		String ret = getInstances(type, polymorphic);
		try {
			return (List<ERElem>) ElementMapper.unmarshalList(ERElement.class, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getInstances(String type, Boolean polymorphic) throws ResourceRegistryException {
		try {
			logger.info("Going to get all instances of {} ", type);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			
			Map<String,String> parameters = new HashMap<>();
			addOffset(parameters);
			addLimit(parameters);
			parameters.put(InstancePath.POLYMORPHIC_QUERY_PARAMETER, polymorphic.toString());
			includeAdditionalQueryParameters(gxHTTPStringRequest, parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got instances of {} are {}", type, ret);
			return ret;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} instances", type, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {} instances", type, e);
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public <ERElem extends ERElement> boolean existInstance(Class<ERElem> clazz, UUID uuid)
			throws AvailableInAnotherContextException, ResourceRegistryException {
		String type = TypeUtility.getTypeName(clazz);
		return existInstance(type, uuid);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean existInstance(String type, UUID uuid)
			throws AvailableInAnotherContextException, ResourceRegistryException {
		try {
			logger.info("Going to check if {} with UUID {} exists", type, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());
			
			Map<String,String> queryParams =  new HashMap<>();
			addHierarchicalMode(queryParams);
			gxHTTPStringRequest.queryParams(queryParams);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.head();
			HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("{} with UUID {} exists", type, uuid);
			return true;
		} catch (NotFoundException e) {
			return false;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while checking if {} with UUID {} exists.", type, uuid,
			// e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while checking if {} with UUID {} exists.", type, uuid,
			// e);
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public <ERElem extends ERElement> ERElem getInstance(Class<ERElem> clazz, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		String type = TypeUtility.getTypeName(clazz);
		String ret = getInstance(type, uuid);
		try {
			return ElementMapper.unmarshal(clazz, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	public String getInstance(String type, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		try {
			logger.info("Going to get {} with UUID {}", type, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());
			
			includeAdditionalQueryParameters(gxHTTPStringRequest);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got {} with UUID {} is {}", type, uuid, ret);
			return ret;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} with UUID {}", type, uuid, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {} with UUID {}", type, uuid, e);
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String rawQuery(String query)
			throws InvalidQueryException, ResourceRegistryException {
		return rawQuery(query, false);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String rawQuery(String query, boolean raw)
			throws InvalidQueryException, ResourceRegistryException {
		
		try {
			logger.info("Going to query. {}", query);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.QUERY_PATH_PART);
			
			Map<String,String> parameters = new HashMap<>();
			parameters.put(AccessPath.Q_QUERY_PARAMETER, query);
			parameters.put(AccessPath.RAW_QUERY_PARAMETER, Boolean.toString(raw));
			
			if(raw) {
				addHierarchicalMode(parameters);
				gxHTTPStringRequest.queryParams(parameters);
			} else {
				includeAdditionalQueryParameters(gxHTTPStringRequest, parameters);
			}
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();

			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.trace("Query result is {}", ret);
			return ret;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while querying", e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while querying", e);
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * Gets entities related to a specific reference entity through a specific relation.
	 * 
	 * @param entityType the type of entities to retrieve
	 * @param relationType the type of relation connecting the entities
	 * @param referenceEntityType the type of the reference entity
	 * @param referenceEntity the UUID of the reference entity (null for constraints-based search)
	 * @param direction the direction of the relation (IN/OUT)
	 * @param polymorphic whether to include polymorphic types
	 * @param facetConstraints optional constraints on facet properties
	 * @return JSON string containing the related entities
	 * @throws ResourceRegistryException if an error occurs during retrieval
	 */
	protected String getRelated(String entityType, String relationType, String referenceEntityType,
			UUID referenceEntity, Direction direction, Boolean polymorphic, Map<String,String> facetConstraints)
			throws ResourceRegistryException {
		
		try {
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.QUERY_PATH_PART);
			gxHTTPStringRequest.path(entityType);
			gxHTTPStringRequest.path(relationType);
			gxHTTPStringRequest.path(referenceEntityType);
			
			Map<String,String> parameters = new HashMap<>();
			parameters.put(AccessPath._DIRECTION_QUERY_PARAMETER, direction.name());
			parameters.put(AccessPath._POLYMORPHIC_QUERY_PARAMETER, polymorphic.toString());
			
			if(referenceEntity == null) {
				if(facetConstraints != null && facetConstraints.size() > 0) {
					logger.info("Going to get {} linked by a {} Relation to a {} having {}", entityType, relationType,
							referenceEntityType, facetConstraints);
					parameters.putAll(facetConstraints);
				} else {
					logger.info("Going to get {} linked by a {} Relation to a {}", entityType, relationType,
							referenceEntityType);
				}
			} else {
				logger.info("Going to get {} linked by {} to {} with UUID {}", entityType, relationType,
						referenceEntityType, referenceEntity);
				parameters.put(AccessPath._REFERENCE_QUERY_PARAMETER, referenceEntity.toString());
			}
			
			includeAdditionalQueryParameters(gxHTTPStringRequest, parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();

			String json = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			if(referenceEntity == null) {
				logger.info("{} linked by {} to/from {} having {} are {}", entityType, relationType,
						referenceEntityType, facetConstraints, json);
				
			} else {
				logger.info("{} linked by {} to/from {} with UUID {} are", entityType, relationType,
						referenceEntityType, referenceEntity, json);
			}
			
			return json;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
		
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public <R extends Resource, C extends ConsistsOf<?,?>, F extends Facet> List<R> getResourcesFromReferenceFacet(
			Class<R> resourceClass, Class<C> consistsOfClass, F referenceFacet,
			boolean polymorphic) throws ResourceRegistryException {
		UUID referenceFacetUUID = referenceFacet.getID();
		@SuppressWarnings("unchecked")
		Class<F> facetClass = (Class<F>) referenceFacet.getClass();
		return getResourcesFromReferenceFacet(resourceClass, consistsOfClass, facetClass, referenceFacetUUID,
				polymorphic);
	}
	
	@SuppressWarnings("unchecked")
	public <R extends Resource, C extends ConsistsOf<?,?>, F extends Facet> List<R> getResourcesFromReferenceFacet(
			Class<R> resourceClass, Class<C> consistsOfClass, Class<F> facetClass, UUID referenceFacetUUID,
			boolean polymorphic) throws ResourceRegistryException {
		String resourceType = TypeUtility.getTypeName(resourceClass);
		String consistsOfType = TypeUtility.getTypeName(consistsOfClass);
		String facetType = TypeUtility.getTypeName(facetClass);
		String ret = getResourcesFromReferenceFacet(resourceType, consistsOfType, facetType, referenceFacetUUID,
				polymorphic);
		try {
			return (List<R>) ElementMapper.unmarshalList(Resource.class, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getResourcesFromReferenceFacet(String resourceType, String consistsOfType, String facetType,
			UUID facetUUID, boolean polymorphic) throws ResourceRegistryException {
		return getRelated(resourceType, consistsOfType, facetType, facetUUID, Direction.OUT, polymorphic);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <R extends Resource, C extends ConsistsOf<?,?>, F extends Facet> List<R> getFilteredResources(
			Class<R> resourceClass, Class<C> consistsOfClass, Class<F> facetClass, boolean polymorphic,
			Map<String,String> facetConstraints) throws ResourceRegistryException {
		String resourceType = TypeUtility.getTypeName(resourceClass);
		String consistsOfType = TypeUtility.getTypeName(consistsOfClass);
		String facetType = TypeUtility.getTypeName(facetClass);
		String ret = getFilteredResources(resourceType, consistsOfType, facetType, polymorphic, facetConstraints);
		try {
			return (List<R>) ElementMapper.unmarshalList(Resource.class, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getFilteredResources(String resourceType, String consistsOfType, String facetType,
			boolean polymorphic, Map<String,String> facetConstraints) throws ResourceRegistryException {
		return getRelated(resourceType, consistsOfType, facetType, Direction.OUT, polymorphic, facetConstraints);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public <R extends Resource, I extends IsRelatedTo<?,?>, RR extends Resource> List<R> getRelatedResourcesFromReferenceResource(
			Class<R> resourceClass, Class<I> isRelatedToClass, RR referenceResource,
			Direction direction, boolean polymorphic) throws ResourceRegistryException {
		UUID referenceResourceUUID = referenceResource.getID();
		@SuppressWarnings("unchecked")
		Class<RR> referenceResourceClass = (Class<RR>) referenceResource.getClass();
		return getRelatedResourcesFromReferenceResource(resourceClass, isRelatedToClass, referenceResourceClass,
				referenceResourceUUID, direction, polymorphic);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <R extends Resource, I extends IsRelatedTo<?,?>, RR extends Resource> List<R> getRelatedResourcesFromReferenceResource(
			Class<R> resourceClass, Class<I> isRelatedToClass, Class<RR> referenceResourceClass,
			UUID referenceResourceUUID, Direction direction, boolean polymorphic) throws ResourceRegistryException {
		String resourceType = TypeUtility.getTypeName(resourceClass);
		String isRelatedToType = TypeUtility.getTypeName(isRelatedToClass);
		String referenceResourceType = TypeUtility.getTypeName(referenceResourceClass);
		String ret = getRelatedResourcesFromReferenceResource(resourceType, isRelatedToType, referenceResourceType,
				referenceResourceUUID, direction, polymorphic);
		try {
			return (List<R>) ElementMapper.unmarshalList(Resource.class, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getRelatedResourcesFromReferenceResource(String resourceType, String isRelatedToType,
			String referenceResourceType, UUID referenceResourceUUID, Direction direction, boolean polymorphic)
			throws ResourceRegistryException {
		return getRelated(resourceType, isRelatedToType, referenceResourceType, referenceResourceUUID, direction,
				polymorphic);
	}
	
	@SuppressWarnings("unchecked")
	/**
	 * {@inheritDoc}
	 */
	@Override
	public <R extends Resource, I extends IsRelatedTo<?,?>, RR extends Resource> List<R> getRelatedResources(
			Class<R> resourceClass, Class<I> isRelatedToClass, Class<RR> referenceResourceClass, Direction direction,
			boolean polymorphic) throws ResourceRegistryException {
		String resourceType = TypeUtility.getTypeName(resourceClass);
		String isRelatedToType = TypeUtility.getTypeName(isRelatedToClass);
		String referenceResourceType = TypeUtility.getTypeName(referenceResourceClass);
		String ret = getRelatedResources(resourceType, isRelatedToType, referenceResourceType, direction, polymorphic);
		try {
			return (List<R>) ElementMapper.unmarshalList(Resource.class, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getRelatedResources(String resourceType, String isRelatedToType, String referenceResourceType,
			Direction direction, boolean polymorphic) throws ResourceRegistryException {
		return getRelated(resourceType, isRelatedToType, referenceResourceType, direction, polymorphic, null);
	}
	
	/**
	 * Gets entities related through a specific relation with constraint-based filtering.
	 * 
	 * @param <E> the entity type to retrieve
	 * @param <R> the relation type
	 * @param <RE> the reference entity type
	 * @param entityClass the class of entities to retrieve
	 * @param relationClass the class of the relation
	 * @param referenceEntityClass the class of the reference entity
	 * @param direction the direction of the relation (IN/OUT)
	 * @param polymorphic whether to include polymorphic types
	 * @param map optional constraints on entity properties
	 * @return a list of related entities
	 * @throws ResourceRegistryException if an error occurs during retrieval
	 */
	@SuppressWarnings("unchecked")
	// @Override
	protected <E extends Entity, R extends Relation<?,?>, RE extends Entity> List<E> getRelated(Class<E> entityClass,
			Class<R> relationClass, Class<RE> referenceEntityClass, Direction direction, boolean polymorphic,
			Map<String,String> map) throws ResourceRegistryException {
		String entityType = TypeUtility.getTypeName(entityClass);
		String relationType = TypeUtility.getTypeName(relationClass);
		String referenceEntityType = TypeUtility.getTypeName(referenceEntityClass);
		String ret = getRelated(entityType, relationType, referenceEntityType, direction, polymorphic, map);
		try {
			return (List<E>) ElementMapper.unmarshalList(Resource.class, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * Gets entities related through a specific relation with constraint-based filtering (string-based).
	 * 
	 * @param entityType the type name of entities to retrieve
	 * @param relationType the type name of the relation
	 * @param referenceEntityType the type name of the reference entity
	 * @param direction the direction of the relation (IN/OUT)
	 * @param polymorphic whether to include polymorphic types
	 * @param facetConstraints optional constraints on facet properties
	 * @return JSON string containing the related entities
	 * @throws ResourceRegistryException if an error occurs during retrieval
	 */
	// @Override
	protected String getRelated(String entityType, String relationType, String referenceEntityType, Direction direction,
			boolean polymorphic, Map<String,String> facetConstraints) throws ResourceRegistryException {
		return getRelated(entityType, relationType, referenceEntityType, null, direction, polymorphic, facetConstraints);
	}
	
	/**
	 * Gets entities related to a specific reference entity through a specific relation.
	 * 
	 * @param <E> the entity type to retrieve
	 * @param <R> the relation type
	 * @param <RE> the reference entity type
	 * @param entityClass the class of entities to retrieve
	 * @param relationClass the class of the relation
	 * @param referenceEntityClass the class of the reference entity
	 * @param referenceEntity the specific reference entity instance
	 * @param direction the direction of the relation (IN/OUT)
	 * @param polymorphic whether to include polymorphic types
	 * @return a list of related entities
	 * @throws ResourceRegistryException if an error occurs during retrieval
	 */
	// @Override
	protected  <E extends Entity, R extends Relation<?,?>, RE extends Entity> List<E> getRelated(Class<E> entityClass,
			Class<R> relationClass, Class<RE> referenceEntityClass, RE referenceEntity, Direction direction,
			boolean polymorphic) throws ResourceRegistryException {
		UUID referenceEntityUUID = referenceEntity.getID();
		return getRelated(entityClass, relationClass, referenceEntityClass, referenceEntityUUID, direction,
				polymorphic);
	}
	
	/**
	 * Gets entities related to a specific reference entity through a specific relation using UUID.
	 * 
	 * @param <E> the entity type to retrieve
	 * @param <R> the relation type
	 * @param <RE> the reference entity type
	 * @param entityClass the class of entities to retrieve
	 * @param relationClass the class of the relation
	 * @param referenceEntityClass the class of the reference entity
	 * @param referenceEntityUUID the UUID of the reference entity
	 * @param direction the direction of the relation (IN/OUT)
	 * @param polymorphic whether to include polymorphic types
	 * @return a list of related entities
	 * @throws ResourceRegistryException if an error occurs during retrieval
	 */
	@SuppressWarnings("unchecked")
	// @Override
	protected  <E extends Entity, R extends Relation<?,?>, RE extends Entity> List<E> getRelated(Class<E> entityClass,
			Class<R> relationClass, Class<RE> referenceEntityClass, UUID referenceEntityUUID, Direction direction,
			boolean polymorphic) throws ResourceRegistryException {
		String entityType = TypeUtility.getTypeName(entityClass);
		String relationType = TypeUtility.getTypeName(relationClass);
		String referenceEntityType = TypeUtility.getTypeName(referenceEntityClass);
		String ret = getRelated(entityType, relationType, referenceEntityType, referenceEntityUUID, direction,
				polymorphic);
		try {
			return (List<E>) ElementMapper.unmarshalList(Resource.class, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * Gets entities related to a specific reference entity through a specific relation (string-based with UUID).
	 * 
	 * @param entityType the type name of entities to retrieve
	 * @param relationType the type name of the relation
	 * @param referenceEntityType the type name of the reference entity
	 * @param referenceEntity the UUID of the reference entity
	 * @param direction the direction of the relation (IN/OUT)
	 * @param polymorphic whether to include polymorphic types
	 * @return JSON string containing the related entities
	 * @throws ResourceRegistryException if an error occurs during retrieval
	 */
	// @Override
	protected  String getRelated(String entityType, String relationType, String referenceEntityType, UUID referenceEntity,
			Direction direction, boolean polymorphic) throws ResourceRegistryException {
		return getRelated(entityType, relationType, referenceEntityType, referenceEntity, direction, polymorphic, null);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <ERElem extends ERElement> Map<UUID, String> getInstanceContexts(Class<ERElem> clazz, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		String typeName = TypeUtility.getTypeName(clazz);
		return getInstanceContexts(typeName, uuid);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Map<UUID, String> getInstanceContexts(String type, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		try {
			logger.trace("Going to get contexts of {} with UUID {}", type, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());
			gxHTTPStringRequest.path(AccessPath.CONTEXTS_PATH_PART);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String objectNode = HTTPUtility.getResponse(String.class, httpURLConnection);

			logger.info("Contexts of {} with UUID {} are {}", type, uuid, objectNode);
			
			Map<UUID, String> contexts = ContextUtility.getContextMap(objectNode);
			return contexts;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} with UUID {}", type, uuid, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {} with UUID {}", type, uuid, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<QueryTemplate> getAllQueryTemplates() throws ResourceRegistryException {
		try {
			logger.trace("Going to list {}s", QueryTemplate.NAME);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(QueryTemplatePath.QUERY_TEMPLATES_PATH_PART);
			
			Map<String,String> parameters = new HashMap<>();
			addIncludeMeta(parameters);
			gxHTTPStringRequest.queryParams(parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String all = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got {}s are {}", QueryTemplate.NAME, all);
			
			JavaType type = ElementMapper.getObjectMapper().getTypeFactory().constructCollectionType(ArrayList.class, QueryTemplate.class);
			return ElementMapper.getObjectMapper().readValue(all, type);
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean existQueryTemplate(QueryTemplate queryTemplate)
			throws QueryTemplateNotFoundException, ResourceRegistryException {
		return existQueryTemplate(queryTemplate.getName());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean existQueryTemplate(String queryTemplateName)
			throws QueryTemplateNotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to read {} with name {}", QueryTemplate.NAME, queryTemplateName);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(QueryTemplatePath.QUERY_TEMPLATES_PATH_PART);
			gxHTTPStringRequest.path(queryTemplateName);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.head();
			HTTPUtility.getResponse(String.class, httpURLConnection);
			return true;
		} catch (NotFoundException e) {
			return false;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public QueryTemplate readQueryTemplate(QueryTemplate queryTemplate)
			throws QueryTemplateNotFoundException, ResourceRegistryException {
		return readQueryTemplate(queryTemplate.getName());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public QueryTemplate readQueryTemplate(String queryTemplateName)
			throws QueryTemplateNotFoundException, ResourceRegistryException {
		try {
			String queryTemplate = readQueryTemplateAsString(queryTemplateName);
			return ElementMapper.unmarshal(QueryTemplate.class, queryTemplate);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String readQueryTemplateAsString(String queryTemplateName)
			throws QueryTemplateNotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to read {} with name {}", QueryTemplate.NAME, queryTemplateName);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(QueryTemplatePath.QUERY_TEMPLATES_PATH_PART);
			gxHTTPStringRequest.path(queryTemplateName);
			
			Map<String,String> parameters = new HashMap<>();
			addIncludeMeta(parameters);
			gxHTTPStringRequest.queryParams(parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String c = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got {} is {}", QueryTemplate.NAME, c);
			return c;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}

//	@Override
//	public <E extends Entity> List<E> runQueryTemplate(String name)
//			throws QueryTemplateNotFoundException, ResourceRegistryException {
//		try {
//			String ret = runQueryTemplateGetString(name);
//			JavaType type = ElementMapper.getObjectMapper().getTypeFactory().constructCollectionType(ArrayList.class, Entity.class);
//			return ElementMapper.getObjectMapper().readValue(ret, type);
//		} catch(ResourceRegistryException e) {
//			throw e;
//		} catch(Exception e) {
//			throw new RuntimeException(e);
//		}
//	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String runQueryTemplate(String name)
			throws QueryTemplateNotFoundException, ResourceRegistryException {
		try {
			String ret = runQueryTemplate(name, "");
			return ret;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String runQueryTemplate(QueryTemplate queryTemplate)
			throws QueryTemplateNotFoundException, ResourceRegistryException {
		return runQueryTemplate(queryTemplate.getName());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String runQueryTemplate(String name, String params)
			throws QueryTemplateNotFoundException, ResourceRegistryException {
		try {
			if(params==null || params.compareTo("")==0) {
				logger.trace("Going to run {} using default parameters", QueryTemplate.NAME);
				params = null;
			}else {
				logger.trace("Going to run {} with the following parameters {}", QueryTemplate.NAME, params);
			}
			
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.header(CONTENT_TYPE_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(QueryTemplatePath.QUERY_TEMPLATES_PATH_PART);
			gxHTTPStringRequest.path(name);
			
			includeAdditionalQueryParameters(gxHTTPStringRequest);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.post(params);
			String c = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.trace("The result of the query is {}", c);
			return c;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String runQueryTemplate(String name, JsonNode jsonNode)
			throws QueryTemplateNotFoundException, ResourceRegistryException {
		try {
			ObjectMapper objectMapper = new ObjectMapper();
			String ret = runQueryTemplate(name, objectMapper.writeValueAsString(jsonNode));
			return ret;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String runQueryTemplate(QueryTemplate queryTemplate, JsonNode jsonNode)
			throws QueryTemplateNotFoundException, ResourceRegistryException {
		return runQueryTemplate(queryTemplate.getName(), jsonNode);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String jsonQuery(String query) throws InvalidQueryException, ResourceRegistryException {
		try {
			logger.trace("Going to run the following JSON Query {}", query);
			
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.header(CONTENT_TYPE_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.QUERY_PATH_PART);
			
			includeAdditionalQueryParameters(gxHTTPStringRequest);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.post(query);
			String c = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.trace("The result of the query is {}", c);
			return c;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <E extends Entity> List<E> jsonQuery(JsonNode jsonNode)
			throws InvalidQueryException, ResourceRegistryException {
		try {
			ObjectMapper objectMapper = new ObjectMapper();
			String ret = jsonQuery(objectMapper.writeValueAsString(jsonNode));
			JavaType type = ElementMapper.getObjectMapper().getTypeFactory().constructCollectionType(ArrayList.class, Entity.class);
			return ElementMapper.getObjectMapper().readValue(ret, type);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
}
