package org.gcube.informationsystem.resourceregistry.publisher;

import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import org.gcube.common.authorization.client.Constants;
import org.gcube.common.authorization.library.AuthorizationEntry;
import org.gcube.common.authorization.library.provider.SecurityTokenProvider;
import org.gcube.common.gxhttp.reference.GXConnection;
import org.gcube.common.gxhttp.request.GXHTTPStringRequest;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.context.reference.entities.Context;
import org.gcube.informationsystem.model.impl.properties.HeaderImpl;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.properties.Header;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.model.reference.relations.IsRelatedTo;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.context.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.facet.FacetAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.facet.FacetNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.rest.AccessPath;
import org.gcube.informationsystem.resourceregistry.api.rest.InstancePath;
import org.gcube.informationsystem.resourceregistry.api.rest.SharingPath;
import org.gcube.informationsystem.resourceregistry.api.rest.httputils.HTTPUtility;
import org.gcube.informationsystem.utils.ElementMapper;
import org.gcube.informationsystem.utils.Utility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ResourceRegistryPublisherImpl implements ResourceRegistryPublisher {
	
	private static final Logger logger = LoggerFactory.getLogger(ResourceRegistryPublisherImpl.class);
	
	protected final String address;
	
	private void checkHierarchicalMode(GXHTTPStringRequest gxHTTPStringRequest) throws UnsupportedEncodingException{
		checkHierarchicalMode(gxHTTPStringRequest, null);
	}
	
	private void checkHierarchicalMode(GXHTTPStringRequest gxHTTPStringRequest, Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(ResourceRegistryPublisherFactory.isHierarchicalMode()) {
			if(queryParams==null) {
				queryParams = new HashMap<>();
			}
			queryParams.put(AccessPath.HIERARCHICAL_MODE_PARAM, Boolean.toString(true));
		}
		gxHTTPStringRequest.queryParams(queryParams);
	}
	
	public ResourceRegistryPublisherImpl(String address) {
		this.address = address;
	}
	
	private static String getCurrentContext() {
		String token = SecurityTokenProvider.instance.get();
		AuthorizationEntry authorizationEntry = null;
		try {
			authorizationEntry = Constants.authorizationService().get(token);
		} catch(Exception e) {
			return ScopeProvider.instance.get();
		}
		return authorizationEntry.getContext();
	}
	
	private UUID getCurrentContextUUID() throws ResourceRegistryException {
		logger.debug("Going to read current {} ({}) definition", Context.NAME, getCurrentContext());
		try {
			logger.info("Going to get current {} ", Context.NAME);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.CURRENT_CONTEXT);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			Context context = HTTPUtility.getResponse(Context.class, httpURLConnection);
			
			logger.debug("Got Context is {}", ElementMapper.marshal(context));
			return context.getHeader().getUUID();
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {}schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw new RuntimeException(e);
		}
	}
	
	protected String create(String identifiableElementTypeName, String json, UUID uuid)
			throws AlreadyPresentException, ResourceRegistryException {
		try {
			logger.trace("Going to create {} : {}", identifiableElementTypeName, json);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.header("Content-type", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(InstancePath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(identifiableElementTypeName);
			gxHTTPStringRequest.path(uuid.toString());
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.put(json);
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.trace("{} successfully created", ret);
			return ret;
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	public <IE extends IdentifiableElement> String internalCreate(IE identifiableElement) throws AlreadyPresentException, ResourceRegistryException {
		try {
			String identifiableElementTypeName = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(identifiableElement);
			String json = ElementMapper.marshal(identifiableElement);
			Header header = identifiableElement.getHeader();
			if(header==null) {
				header = new HeaderImpl(UUID.randomUUID());
				identifiableElement.setHeader(header);
			}
			UUID uuid = identifiableElement.getHeader().getUUID();
			return create(identifiableElementTypeName, json, uuid);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	
	@SuppressWarnings("unchecked")
	@Override
	public <IE extends IdentifiableElement> IE create(IE identifiableElement) throws AlreadyPresentException, ResourceRegistryException {
		try {
			String ret = internalCreate(identifiableElement);
			return (IE) ElementMapper.unmarshal(IdentifiableElement.class, ret);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String create(String identifiableElement) throws AlreadyPresentException, ResourceRegistryException {
		try {
			IdentifiableElement e = ElementMapper.unmarshal(IdentifiableElement.class, identifiableElement);
			return internalCreate(e);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public <IE extends IdentifiableElement> boolean exists(Class<IE> clazz, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		String type = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(clazz);
		return exists(type, uuid);
	}
	
	@Override
	public <IE extends IdentifiableElement> boolean exists(IE identifiableElement)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		String type = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(identifiableElement);
		UUID uuid = identifiableElement.getHeader().getUUID();
		return exists(type, uuid);
	}
	
	@Override
	public boolean exists(String type, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		try {
			logger.info("Going to check if {} with UUID {} exists", type, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address); 
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());
			
			checkHierarchicalMode(gxHTTPStringRequest);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.head();
			HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("{} with UUID {} exists", type, uuid);
			return true;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while checking if {} with UUID {} exists.", type, uuid,
			// e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while checking if {} with UUID {} exists.", type, uuid,
			// e);
			throw new RuntimeException(e);
		}
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <IE extends IdentifiableElement> IE read(IE identifiableElement) throws NotFoundException, ResourceRegistryException {
		try {
			String identifiableElementTypeName = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(identifiableElement);
			UUID uuid = identifiableElement.getHeader().getUUID();
			String ret = read(identifiableElementTypeName, uuid);
			return (IE) ElementMapper.unmarshal(IdentifiableElement.class, ret);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String read(String identifiableElementTypeName, UUID uuid) throws NotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to read {} with UUID {}", identifiableElementTypeName, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(InstancePath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(identifiableElementTypeName);
			gxHTTPStringRequest.path(uuid.toString());
			
			checkHierarchicalMode(gxHTTPStringRequest);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got {} with UUID {} is {}", identifiableElementTypeName, uuid, ret);
			return ret;
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	protected String update(String identifiableElementTypeName, String json, UUID uuid)
			throws AlreadyPresentException, ResourceRegistryException {
		try {
			logger.trace("Going to create {} : {}", identifiableElementTypeName, json);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.header("Content-type", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(InstancePath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(identifiableElementTypeName);
			gxHTTPStringRequest.path(uuid.toString());
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.put(json);
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.trace("{} with UUID {} successfully created : {}", identifiableElementTypeName, uuid, ret);
			return ret;
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <IE extends IdentifiableElement> IE update(IE identifiableElement) throws NotFoundException, ResourceRegistryException {
		try {
			String identifiableElementTypeName = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(identifiableElement);
			String json = ElementMapper.marshal(identifiableElement);
			UUID uuid = identifiableElement.getHeader().getUUID();
			String ret = update(identifiableElementTypeName, json, uuid);
			return (IE) ElementMapper.unmarshal(Element.class, ret);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String update(String identifiableElementTypeName, String json) throws NotFoundException, ResourceRegistryException {
		try {
			UUID uuid = Utility.getUUIDFromJSONString(json);
			return update(identifiableElementTypeName, json, uuid);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String update(String json) throws NotFoundException, ResourceRegistryException {
		try {
			String identifiableElementTypeName = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getClassFromJsonString(json);
			return update(identifiableElementTypeName, json);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public <IE extends IdentifiableElement> boolean delete(IE identifiableElement) throws NotFoundException, ResourceRegistryException {
		try {
			String identifiableElementTypeName = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(identifiableElement);
			UUID uuid = identifiableElement.getHeader().getUUID();
			return delete(identifiableElementTypeName, uuid);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public boolean delete(String identifiableElementTypeName, UUID uuid) throws NotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to delete {} with UUID {}", identifiableElementTypeName, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(InstancePath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(identifiableElementTypeName);
			gxHTTPStringRequest.path(uuid.toString());
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.delete();
			HTTPUtility.getResponse(String.class, httpURLConnection);
			
			boolean deleted = true;
			
			logger.info("{} with UUID {} {}", identifiableElementTypeName, uuid, deleted ? " successfully deleted" : "was NOT deleted");
			return deleted;
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public <F extends Facet> F createFacet(F facet) throws FacetAlreadyPresentException, ResourceRegistryException {
		return create(facet);
	}
	
	@Override
	public String createFacet(String facet) throws FacetAlreadyPresentException, ResourceRegistryException {
		return create(facet);
	}
	
	@Override
	public <F extends Facet> F readFacet(F facet) throws FacetNotFoundException, ResourceRegistryException {
		return read(facet);
	}
	
	@Override
	public String readFacet(String facetTypeName, UUID uuid) throws FacetNotFoundException, ResourceRegistryException {
		return read(facetTypeName, uuid);
	}
	
	@Override
	public <F extends Facet> F updateFacet(F facet) throws FacetNotFoundException, ResourceRegistryException {
		return update(facet);
	}
	
	@Override
	public String updateFacet(String facet) throws FacetNotFoundException, ResourceRegistryException {
		return update(facet);
	}
	
	@Override
	public <F extends Facet> boolean deleteFacet(F facet) throws FacetNotFoundException, ResourceRegistryException {
		return delete(facet);
	}
	
	@Override
	public boolean deleteFacet(String facetTypeName, UUID uuid) throws FacetNotFoundException, ResourceRegistryException {
		return delete(facetTypeName, uuid);
	}
	
	@Override
	public <R extends Resource> R createResource(R resource)
			throws ResourceAlreadyPresentException, ResourceRegistryException {
		return create(resource);
	}
	
	@Override
	public String createResource(String resource) throws ResourceAlreadyPresentException, ResourceRegistryException {
		return create(resource);
	}
	
	@Override
	public <R extends Resource> R readResource(R resource) throws ResourceNotFoundException, ResourceRegistryException {
		return read(resource);
	}
	
	@Override
	public String readResource(String resourceTypeName, UUID uuid)
			throws ResourceNotFoundException, ResourceRegistryException {
		return read(resourceTypeName, uuid);
	}
	
	@Override
	public <R extends Resource> R updateResource(R resource)
			throws ResourceNotFoundException, ResourceRegistryException {
		return update(resource);
	}
	
	@Override
	public String updateResource(String resource) throws ResourceNotFoundException, ResourceRegistryException {
		return update(resource);
	}
	
	@Override
	public <R extends Resource> boolean deleteResource(R resource)
			throws ResourceNotFoundException, ResourceRegistryException {
		return delete(resource);
	}
	
	@Override
	public boolean deleteResource(String resourceTypeName, UUID uuid)
			throws ResourceNotFoundException, ResourceRegistryException {
		return delete(resourceTypeName, uuid);
	}
	
	@Override
	public <C extends ConsistsOf<? extends Resource,? extends Facet>> C createConsistsOf(C consistsOf)
			throws NotFoundException, ResourceRegistryException {
		return create(consistsOf);
	}
	
	@Override
	public String createConsistsOf(String consistsOf) throws NotFoundException, ResourceRegistryException {
		return create(consistsOf);
	}
	
	@Override
	public <C extends ConsistsOf<? extends Resource,? extends Facet>> C readConsistsOf(C consistsOf)
			throws NotFoundException, ResourceRegistryException {
		return read(consistsOf);
	}
	
	@Override
	public String readConsistsOf(String consistsOfTypeName, UUID uuid) throws NotFoundException, ResourceRegistryException {
		return read(consistsOfTypeName, uuid);
	}
	
	@Override
	public <C extends ConsistsOf<? extends Resource,? extends Facet>> C updateConsistsOf(C consistsOf)
			throws NotFoundException, ResourceRegistryException {
		return update(consistsOf);
	}
	
	@Override
	public String updateConsistsOf(String consistsOf) throws NotFoundException, ResourceRegistryException {
		return update(consistsOf);
	}
	
	@Override
	public <C extends ConsistsOf<? extends Resource,? extends Facet>> boolean deleteConsistsOf(C consistsOf)
			throws ResourceRegistryException {
		return delete(consistsOf);
	}
	
	@Override
	public boolean deleteConsistsOf(String consistsOfTypeName, UUID uuid) throws ResourceRegistryException {
		return delete(consistsOfTypeName, uuid);
	}
	
	@Override
	public <I extends IsRelatedTo<? extends Resource,? extends Resource>> I createIsRelatedTo(I isRelatedTo)
			throws ResourceNotFoundException, ResourceRegistryException {
		return create(isRelatedTo);
	}
	
	@Override
	public String createIsRelatedTo(String isRelatedTo) throws ResourceNotFoundException, ResourceRegistryException {
		return create(isRelatedTo);
	}
	
	@Override
	public <I extends IsRelatedTo<? extends Resource,? extends Resource>> I readIsRelatedTo(I isRelatedTo)
			throws NotFoundException, ResourceRegistryException {
		return read(isRelatedTo);
	}
	
	@Override
	public String readIsRelatedTo(String isRelatedToTypeName, UUID uuid)
			throws NotFoundException, ResourceRegistryException {
		return read(isRelatedToTypeName, uuid);
	}
	
	@Override
	public <I extends IsRelatedTo<? extends Resource,? extends Resource>> I updateIsRelatedTo(I isRelatedTo)
			throws NotFoundException, ResourceRegistryException {
		return update(isRelatedTo);
	}
	
	@Override
	public String updateIsRelatedTo(String isRelatedTo) throws NotFoundException, ResourceRegistryException {
		return update(isRelatedTo);
	}
	
	@Override
	public <I extends IsRelatedTo<? extends Resource,? extends Resource>> boolean deleteIsRelatedTo(I isRelatedTo)
			throws ResourceRegistryException {
		return delete(isRelatedTo);
	}
	
	@Override
	public boolean deleteIsRelatedTo(String isRelatedToTypeName, UUID uuid) throws ResourceRegistryException {
		return delete(isRelatedToTypeName, uuid);
	}
	
	@Override
	public boolean addToContext(UUID contextUUID, String identifiableElementTypeName, UUID instanceUUID)
			throws NotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to add {} with UUID {} to {} with UUID {} ", identifiableElementTypeName, instanceUUID, Context.NAME,
					contextUUID);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.path(SharingPath.SHARING_PATH_PART);
			gxHTTPStringRequest.path(SharingPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(contextUUID.toString());
			gxHTTPStringRequest.path(identifiableElementTypeName);
			gxHTTPStringRequest.path(instanceUUID.toString());
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.put();
			boolean added = HTTPUtility.getResponse(Boolean.class, httpURLConnection);
			
			logger.info("{} with UUID {} {} to {} with UUID {}", identifiableElementTypeName, instanceUUID,
					added ? " successfully added" : "was NOT added", Context.NAME, contextUUID);
			return added;
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public <IE extends IdentifiableElement> boolean addToContext(UUID contextUUID, IE identifiableElement)
			throws NotFoundException, ResourceRegistryException {
		try {
			String identifiableElementTypeName = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(identifiableElement);
			UUID instanceUUID = identifiableElement.getHeader().getUUID();
			return addToContext(contextUUID, identifiableElementTypeName, instanceUUID);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public boolean addToCurrentContext(String identifiableElementTypeName, UUID instanceUUID)
			throws NotFoundException, ResourceRegistryException {
		UUID contextUUID = getCurrentContextUUID();
		return addToContext(contextUUID, identifiableElementTypeName, instanceUUID);
	}
	
	@Override
	public <IE extends IdentifiableElement> boolean addToCurrentContext(IE identifiableElement) throws NotFoundException, ResourceRegistryException {
		UUID contextUUID = getCurrentContextUUID();
		return addToContext(contextUUID, identifiableElement);
	}
	
	@Override
	public boolean removeFromContext(UUID contextUUID, String identifiableElementTypeName, UUID instanceUUID)
			throws NotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to add {} with UUID {} to {} with UUID {} ", identifiableElementTypeName, instanceUUID, Context.NAME,
					contextUUID);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
			gxHTTPStringRequest.from(ResourceRegistryPublisher.class.getSimpleName());
			gxHTTPStringRequest.path(SharingPath.SHARING_PATH_PART);
			gxHTTPStringRequest.path(SharingPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(contextUUID.toString());
			gxHTTPStringRequest.path(identifiableElementTypeName);
			gxHTTPStringRequest.path(instanceUUID.toString());
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.delete();
			boolean removed = HTTPUtility.getResponse(Boolean.class, httpURLConnection);
			
			logger.info("{} with UUID {} {} to {} with UUID {}", identifiableElementTypeName, instanceUUID,
					removed ? " successfully removed" : "was NOT removed", Context.NAME, contextUUID);
			return removed;
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public <IE extends IdentifiableElement> boolean removeFromContext(UUID contextUUID, IE identifiableElement)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		try {
			String identifiableElementTypeName = org.gcube.informationsystem.resourceregistry.api.utils.Utility.getTypeName(identifiableElement);
			UUID instanceUUID = identifiableElement.getHeader().getUUID();
			return removeFromContext(contextUUID, identifiableElementTypeName, instanceUUID);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error Creating {}", facet, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error Creating {}", facet, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public boolean removeFromCurrentContext(String identifiableElementTypeName, UUID instanceUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		UUID contextUUID = getCurrentContextUUID();
		return removeFromContext(contextUUID, identifiableElementTypeName, instanceUUID);
	}
	
	@Override
	public <IE extends IdentifiableElement> boolean removeFromCurrentContext(IE identifiableElement)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		UUID contextUUID = getCurrentContextUUID();
		return removeFromContext(contextUUID, identifiableElement);
	}
	
	@Override
	public boolean addResourceToContext(UUID contextUUID, String resourceTypeName, UUID resourceUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToContext(contextUUID, resourceTypeName, resourceUUID);
	}
	
	@Override
	public <R extends Resource> boolean addResourceToContext(UUID contextUUID, R resource)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToContext(contextUUID, resource);
	}
	
	@Override
	public boolean addResourceToCurrentContext(String resourceTypeName, UUID resourceUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToCurrentContext(resourceTypeName, resourceUUID);
	}
	
	@Override
	public <R extends Resource> boolean addResourceToCurrentContext(R resource)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToCurrentContext(resource);
	}
	
	@Override
	public boolean removeResourceFromContext(UUID contextUUID, String resourceTypeName, UUID resourceUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromContext(contextUUID, resourceTypeName, resourceUUID);
	}
	
	@Override
	public <R extends Resource> boolean removeResourceFromContext(UUID contextUUID, R resource)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromContext(contextUUID, resource);
	}
	
	@Override
	public boolean removeResourceFromCurrentContext(String resourceTypeName, UUID resourceUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromCurrentContext(resourceTypeName, resourceUUID);
	}
	
	@Override
	public <R extends Resource> boolean removeResourceFromCurrentContext(R resource)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromCurrentContext(resource);
	}
	
	@Override
	public boolean addFacetToContext(UUID contextUUID, String facetTypeName, UUID facetUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToContext(contextUUID, facetTypeName, facetUUID);
	}
	
	@Override
	public <F extends Facet> boolean addFacetToContext(UUID contextUUID, F facet)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToContext(contextUUID, facet);
	}
	
	@Override
	public boolean addFacetToCurrentContext(String facetTypeName, UUID facetUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToCurrentContext(facetTypeName, facetUUID);
	}
	
	@Override
	public <F extends Facet> boolean addFacetToCurrentContext(F facet)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return addToCurrentContext(facet);
	}
	
	@Override
	public boolean removeFacetFromContext(UUID contextUUID, String facetTypeName, UUID facetUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromContext(contextUUID, facetTypeName, facetUUID);
	}
	
	@Override
	public <F extends Facet> boolean removeFacetFromContext(UUID contextUUID, F facet)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromContext(contextUUID, facet);
	}
	
	@Override
	public boolean removeFacetFromCurrentContext(String facetTypeName, UUID facetUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromCurrentContext(facetTypeName, facetUUID);
	}
	
	@Override
	public <F extends Facet> boolean removeFacetFromCurrentContext(F facet)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException {
		return removeFromCurrentContext(facet);
	}
	
}
