package org.gcube.informationsystem.resourceregistry.schema;

import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.gcube.common.gxhttp.reference.GXConnection;
import org.gcube.common.gxhttp.request.GXHTTPStringRequest;
import org.gcube.common.security.providers.SecretManagerProvider;
import org.gcube.common.security.secrets.Secret;
import org.gcube.informationsystem.model.knowledge.ModelKnowledge;
import org.gcube.informationsystem.model.reference.ModelElement;
import org.gcube.informationsystem.model.reference.properties.Metadata;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.contexts.ContextAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.contexts.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.types.SchemaException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.types.SchemaNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.rest.AccessPath;
import org.gcube.informationsystem.resourceregistry.api.rest.TypePath;
import org.gcube.informationsystem.resourceregistry.api.rest.httputils.HTTPUtility;
import org.gcube.informationsystem.tree.Node;
import org.gcube.informationsystem.types.TypeMapper;
import org.gcube.informationsystem.types.knowledge.TypeInformation;
import org.gcube.informationsystem.types.knowledge.TypesKnowledge;
import org.gcube.informationsystem.types.reference.Type;
import org.gcube.informationsystem.utils.TypeUtility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class ResourceRegistrySchemaClientImpl implements ResourceRegistrySchemaClient {
	
	private static final Logger logger = LoggerFactory.getLogger(ResourceRegistrySchemaClientImpl.class);
	
	private static final String ACCEPT_HTTP_HEADER_KEY = "Accept";
	private static final String CONTENT_TYPE_HTTP_HEADER_KEY = "Content-Type";
	
	/**
	 * The base address URL for the Resource Registry Schema service endpoint.
	 */
	protected final String address;
	
	/**
	 * HTTP headers to be included in requests to the Resource Registry Schema service.
	 */
	protected Map<String, String> headers;
	
	/**
	 * Cached knowledge about types and their relationships within the Information System.
	 */
	protected TypesKnowledge typesKnowledge;
	
	/**
	 * Track if the client must request to include {@link Metadata} 
	 */
	protected boolean includeMeta;
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean includeMeta() {
		return includeMeta;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setIncludeMeta(boolean includeMeta) {
		this.includeMeta = includeMeta;
	}

	private void addOptionalQueryParameters(Map<String,String> queryParams) throws UnsupportedEncodingException {
		addIncludeMeta(queryParams);
	}
	
	private GXHTTPStringRequest includeAdditionalQueryParameters(GXHTTPStringRequest gxHTTPStringRequest) throws UnsupportedEncodingException{
		Map<String,String> queryParams = new HashMap<>();
		return includeAdditionalQueryParameters(gxHTTPStringRequest, queryParams);
	}
	
	private GXHTTPStringRequest includeAdditionalQueryParameters(GXHTTPStringRequest gxHTTPStringRequest, Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(queryParams==null) {
			queryParams = new HashMap<>();
		}
		addOptionalQueryParameters(queryParams);
		return gxHTTPStringRequest.queryParams(queryParams);
	}
	
	private void addIncludeMeta(Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(includeMeta) {
			queryParams.put(AccessPath.INCLUDE_META_QUERY_PARAMETER, Boolean.toString(includeMeta));
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addHeader(String name, String value) {
		headers.put(name, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addHeader(String name, boolean value) {
		addHeader(name, Boolean.toString(value));
	}
	
	/**
	 * Creates and configures a GXHTTPStringRequest with authentication headers and custom headers.
	 * This method sets up the basic HTTP request infrastructure needed for all REST API calls.
	 * 
	 * @return a configured GXHTTPStringRequest ready for API calls
	 */
	protected GXHTTPStringRequest getGXHTTPStringRequest() {
		GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
		
		/* It is a gcube request */
		Secret secret = SecretManagerProvider.get();
		Map<String, String> authorizationHeaders = secret.getHTTPAuthorizationHeaders();
		for(String key : authorizationHeaders.keySet()) {
			gxHTTPStringRequest.header(key, authorizationHeaders.get(key));
		}

		gxHTTPStringRequest.from(this.getClass().getSimpleName());
		for(String name : headers.keySet()) {
			gxHTTPStringRequest.header(name, headers.get(name));
		}
		return gxHTTPStringRequest;
	}
	
	/**
	 * Creates a new ResourceRegistrySchemaClient instance with shared model knowledge.
	 * This constructor delegates to {@link #ResourceRegistrySchemaClientImpl(String, boolean)} with sharedModelKnowledge=true.
	 * 
	 * @param address the base address of the Resource Registry service
	 */
	public ResourceRegistrySchemaClientImpl(String address) {
		this(address, true);
		
	}
	
	/**
	 * Creates a new ResourceRegistrySchemaClient instance.
	 * 
	 * @param address the base address of the Resource Registry service
	 * @param sharedModelKnowledge whether to use shared TypesKnowledge instance (true) or create a new one (false)
	 */
	public ResourceRegistrySchemaClientImpl(String address, boolean sharedModelKnowledge) {
		this.address = address;
		this.headers = new HashMap<>();
		this.includeMeta = false;
		
		if(sharedModelKnowledge) {
			this.typesKnowledge = TypesKnowledge.getInstance();
		}else {
			this.typesKnowledge = new TypesKnowledge();
		}
		typesKnowledge.setTypesDiscoverer(new RRCCTypesDiscoverer(this));
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public ModelKnowledge<Type, TypeInformation> getModelKnowledge() {
		return typesKnowledge.getModelKnowledge();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void renewModelKnowledge() {
		typesKnowledge.renew();
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #create(String)} for implementation details.
	 */
	@Override
	public <ME extends ModelElement> Type create(Class<ME> clz)
			throws SchemaException, ResourceRegistryException {
		try {
			String typeDefinition = TypeMapper.serializeType(clz);
			// String type = AccessType.getAccessType(clz).getName();
			String res = create(typeDefinition);
			return TypeMapper.deserializeTypeDefinition(res);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String create(String typeDefinitition) throws ContextAlreadyPresentException, ResourceRegistryException {
		try {
			logger.trace("Going to create: {}", typeDefinitition);
			Type typeDefinitionObj = TypeMapper.deserializeTypeDefinition(typeDefinitition);
			
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.header(CONTENT_TYPE_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(TypePath.TYPES_PATH_PART);
			gxHTTPStringRequest.path(typeDefinitionObj.getName());
			
			includeAdditionalQueryParameters(gxHTTPStringRequest);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.put(typeDefinitition);
			String c = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			Type t = TypeMapper.deserializeTypeDefinition(c);
			typesKnowledge.getModelKnowledge().addType(t);
			
			logger.trace("{} successfully created", c);
			return c;
			
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #exist(String)} for implementation details.
	 */
	@Override
	public <ME extends ModelElement> boolean exist(Class<ME> clazz) throws ResourceRegistryException {
		return exist(TypeUtility.getTypeName(clazz));
	}	
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean exist(String typeName) throws ResourceRegistryException {
		try {
			return typesKnowledge.getModelKnowledge().getTypeByName(typeName) != null;
		}catch (RuntimeException e) {
			return false;
		}
	}
	
	/**
	 * Retrieves type information from the local TypesKnowledge cache.
	 * This method supports both polymorphic and level-based filtering.
	 * 
	 * @param typeName the name of the type to retrieve
	 * @param polymorphic whether to include subtypes
	 * @return list of Type objects matching the criteria
	 * @throws SchemaNotFoundException if the type is not found
	 * @throws ResourceRegistryException for other retrieval errors
	 */
	public List<Type> getTypeFromTypesKnowledge(String typeName, Boolean polymorphic)
			throws SchemaNotFoundException, ResourceRegistryException {
		return getTypeFromTypesKnowledge(typeName, polymorphic, -1);
	}
	
	/**
	 * Retrieves type information from the local TypesKnowledge cache with level-based filtering.
	 * This method delegates to {@link #getTypeFromTypesKnowledge(String, Boolean, int)} with polymorphic=true.
	 * 
	 * @param typeName the name of the type to retrieve
	 * @param level the maximum hierarchy level to include
	 * @return list of Type objects matching the criteria
	 * @throws SchemaNotFoundException if the type is not found
	 * @throws ResourceRegistryException for other retrieval errors
	 */
	public List<Type> getTypeFromTypesKnowledge(String typeName, int level)
			throws SchemaNotFoundException, ResourceRegistryException {
		return getTypeFromTypesKnowledge(typeName, true, level);
	}
	
	/**
	 * Recursively adds child types to the result list up to the specified maximum level.
	 * This method is used internally to build polymorphic type hierarchies.
	 * 
	 * @param node the current node in the type hierarchy
	 * @param types the accumulator list for type results
	 * @param currentLevel the current recursion level
	 * @param maxLevel the maximum level to recurse to (-1 for unlimited)
	 * @return the updated list of types including children
	 */
	protected List<Type> addChildren(Node<Type> node, List<Type> types, int currentLevel, int maxLevel) {
		if(maxLevel>=0 && maxLevel <= currentLevel) {
			return types;
		}
		
		Set<Node<Type>> children = node.getChildren();
		if(children!=null && children.size()>0) {
			for(Node<Type> child : children) {
				types.add(child.getNodeElement());
				types = addChildren(child, types, ++currentLevel, maxLevel);
			}
		}
		
		return types;
	}
	
	/**
	 * Retrieves type information from the local TypesKnowledge cache with full filtering options.
	 * This is the main implementation method that supports both polymorphic and level-based filtering.
	 * 
	 * @param typeName the name of the type to retrieve
	 * @param polymorphic whether to include subtypes
	 * @param level the maximum hierarchy level to include (-1 for unlimited)
	 * @return list of Type objects matching the criteria
	 * @throws SchemaNotFoundException if the type is not found
	 * @throws ResourceRegistryException for other retrieval errors
	 */
	public List<Type> getTypeFromTypesKnowledge(String typeName, Boolean polymorphic, int level)
			throws SchemaNotFoundException, ResourceRegistryException {
		
		Node<Type> node = getTypeTreeNode(typeName);
		
		List<Type> types = new ArrayList<>();
		types.add(node.getNodeElement());
		
		if (polymorphic) {
			addChildren(node, types, 0, level);
		} 
		
		return types;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String read(String typeName, Boolean polymorphic) throws SchemaNotFoundException, ResourceRegistryException {
		try { 
			List<Type> types = getTypeFromTypesKnowledge(typeName, polymorphic);
			return TypeMapper.serializeTypeDefinitions(types);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #getTypeFromTypesKnowledge(String, Boolean)} for implementation details.
	 */
	@Override
	public <ME extends ModelElement> List<Type> read(Class<ME> clazz, Boolean polymorphic)
			throws SchemaNotFoundException, ResourceRegistryException {
		try {
			String typeName = TypeUtility.getTypeName(clazz);
			return getTypeFromTypesKnowledge(typeName, polymorphic);
		} catch (Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String read(String typeName, int level) throws SchemaNotFoundException, ResourceRegistryException {
		try { 
			List<Type> types = getTypeFromTypesKnowledge(typeName, level);
			return TypeMapper.serializeTypeDefinitions(types);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #getTypeFromTypesKnowledge(String, int)} for implementation details.
	 */
	@Override
	public <ME extends ModelElement> List<Type> read(Class<ME> clazz, int level)
			throws SchemaNotFoundException, ResourceRegistryException {
		try {
			String typeName = TypeUtility.getTypeName(clazz);
			return getTypeFromTypesKnowledge(typeName, level);
		} catch (Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Node<Type> getTypeTreeNode(String typeName) throws SchemaNotFoundException, ResourceRegistryException {
		try { 
			Node<Type> node = null;
			try {
				node = typesKnowledge.getModelKnowledge().getNodeByName(typeName);
			} catch (RuntimeException e) {
				throw new SchemaNotFoundException(e);
			}
			return node;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * This method is a type-safe wrapper that delegates to {@link #getTypeTreeNode(String)} for implementation details.
	 */
	@Override
	public <ME extends ModelElement> Node<Type> getTypeTreeNode(Class<ME> clazz)
			throws SchemaNotFoundException, ResourceRegistryException {
		try {
			String typeName = TypeUtility.getTypeName(clazz);
			return getTypeTreeNode(typeName);
		} catch (Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
	
	/**
	 * Checks if a type exists directly on the server using a HEAD request.
	 * This method bypasses the local cache and queries the server directly.
	 * 
	 * @param typeName the name of the type to check
	 * @return true if the type exists on the server, false otherwise
	 * @throws ResourceRegistryException for server communication errors
	 */
	public boolean existTypeFromServer(String typeName) throws ResourceRegistryException {
		try {
			logger.info("Going to get {} schema", typeName);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(TypePath.TYPES_PATH_PART);
			gxHTTPStringRequest.path(typeName);
			
			Map<String,String> parameters = new HashMap<>();
			parameters.put(TypePath.POLYMORPHIC_QUERY_PARAMETER, Boolean.FALSE.toString());
			gxHTTPStringRequest.queryParams(parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.head();
			HTTPUtility.getResponse(String.class, httpURLConnection);
			
			return true;
		} catch (NotFoundException e) {
			return false;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * Retrieves type definitions directly from the server using a model element class.
	 * This method bypasses the local cache and queries the server directly.
	 * This method is a type-safe wrapper that delegates to {@link #getTypeFromServer(String, Boolean)} for implementation details.
	 * 
	 * @param <ME> the model element type parameter
	 * @param clz the class of the model element type to retrieve
	 * @param polymorphic whether to include subtypes
	 * @return list of Type objects retrieved from the server
	 * @throws SchemaNotFoundException if the type is not found on the server
	 * @throws ResourceRegistryException for server communication errors
	 */
	public <ME extends ModelElement> List<Type> getTypeFromServer(Class<ME> clz, Boolean polymorphic)
			throws SchemaNotFoundException, ResourceRegistryException {
		try {
			String typeName = TypeUtility.getTypeName(clz);
			String json = getTypeFromServer(typeName, polymorphic);
			return TypeMapper.deserializeTypeDefinitions(json);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
	
	/**
	 * Retrieves type definitions directly from the server by type name.
	 * This method bypasses the local cache and queries the server directly using REST API.
	 * 
	 * @param typeName the name of the type to retrieve
	 * @param polymorphic whether to include subtypes in the response
	 * @return JSON string containing the type definition(s) from the server
	 * @throws ContextNotFoundException if the context is not found
	 * @throws ResourceRegistryException for server communication errors
	 */
	public String getTypeFromServer(String typeName, Boolean polymorphic) throws ContextNotFoundException, ResourceRegistryException {
		try {
			logger.info("Going to get {} schema", typeName);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(TypePath.TYPES_PATH_PART);
			gxHTTPStringRequest.path(typeName);
			
			Map<String,String> parameters = new HashMap<>();
			if(polymorphic != null) {
				parameters.put(TypePath.POLYMORPHIC_QUERY_PARAMETER, polymorphic.toString());
			}
			includeAdditionalQueryParameters(gxHTTPStringRequest, parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String json = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got schema for {} is {}", typeName, json);
			return json;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
}
