package org.gcube.informationsystem.resourceregistry.types.relations;

import java.util.HashMap;

import org.gcube.informationsystem.base.reference.AccessType;
import org.gcube.informationsystem.model.reference.relations.Relation;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.context.ContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.relation.RelationNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.schema.SchemaAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.schema.SchemaNotFoundException;
import org.gcube.informationsystem.resourceregistry.contexts.ContextUtility;
import org.gcube.informationsystem.resourceregistry.contexts.security.SecurityContext;
import org.gcube.informationsystem.resourceregistry.dbinitialization.DatabaseEnvironment;
import org.gcube.informationsystem.resourceregistry.instances.base.ElementManagement;
import org.gcube.informationsystem.resourceregistry.instances.base.relations.RelationElementManagement;
import org.gcube.informationsystem.resourceregistry.types.entities.EntityTypeDefinitionManagement;
import org.gcube.informationsystem.resourceregistry.types.entities.ResourceTypeDefinitionManagement;
import org.gcube.informationsystem.resourceregistry.utils.Utility;
import org.gcube.informationsystem.types.reference.entities.EntityType;
import org.gcube.informationsystem.types.reference.entities.ResourceType;
import org.gcube.informationsystem.types.reference.relations.RelationType;

import com.orientechnologies.orient.core.db.document.ODatabaseDocument;
import com.orientechnologies.orient.core.record.OEdge;
import com.orientechnologies.orient.core.record.OVertex;
import com.orientechnologies.orient.core.sql.executor.OResult;
import com.orientechnologies.orient.core.sql.executor.OResultSet;

public abstract class RelationTypeDefinitionManagement<T extends EntityTypeDefinitionManagement<TT>, TT extends EntityType>
		extends RelationElementManagement<ResourceTypeDefinitionManagement, T> {
	
	protected String name;
	
	public RelationTypeDefinitionManagement(Class<TT> clz) {
		super(AccessType.RELATION_TYPE, ResourceType.class, clz);
		this.elementType = RelationType.NAME;
	}
	
	public RelationTypeDefinitionManagement(SecurityContext securityContext, ODatabaseDocument oDatabaseDocument, Class<TT> clz) throws ResourceRegistryException {
		this(clz);
		this.oDatabaseDocument = oDatabaseDocument;
		setWorkingContext(securityContext);
	}
	
	@Override
	protected SecurityContext getWorkingContext() throws ResourceRegistryException {
		if(workingContext == null) {
			this.workingContext = ContextUtility.getInstance()
					.getSecurityContextByUUID(DatabaseEnvironment.SCHEMA_SECURITY_CONTEXT_UUID);
		}
		return workingContext;
	}
	
	public void setName(String name) {
		this.name = name;
	}
	
	public String getName() {
		if(name == null) {
			if(element == null) {
				if(jsonNode != null) {
					name = jsonNode.get(RelationType.NAME_PROPERTY).asText();
				}
			} else {
				name = element.getProperty(RelationType.NAME_PROPERTY);
			}
		}
		return name;
	}
	
	@Override
	protected OEdge reallyCreate() throws ResourceRegistryException {
		logger.debug("Going to create {} for {}", RelationType.NAME, getName());
		if(sourceEntityManagement == null) {
			if(!jsonNode.has(Relation.SOURCE_PROPERTY)) {
				throw new ResourceRegistryException("Error while creating relation. No source definition found");
			}
			
			sourceEntityManagement = newSourceEntityManagement();
			// sourceEntityManagement.setElementType(EntityTypeDefinition.NAME);
			sourceEntityManagement.setJsonNode(jsonNode.get(Relation.SOURCE_PROPERTY));
		}
		
		if(targetEntityManagement == null) {
			if(!jsonNode.has(Relation.TARGET_PROPERTY)) {
				throw new ResourceRegistryException(
						"Error while creating " + elementType + ". No target definition found");
			}
			
			targetEntityManagement = newTargetEntityManagement();
			// targetEntityManagement.setElementType(EntityTypeDefinition.NAME);
			targetEntityManagement.setJsonNode(jsonNode.get(Relation.TARGET_PROPERTY));
		}
		
		logger.trace("Creating {} beetween {} -> {}", elementType, getSourceEntityManagement().serialize(),
				getTargetEntityManagement().serialize());
		
		OVertex source = (OVertex) getSourceEntityManagement().getElement();
		OVertex target = (OVertex) getTargetEntityManagement().getElement();
		
		element = oDatabaseDocument.newEdge(source, target, elementType);
		
		ElementManagement.updateProperties(oClass, element, jsonNode, ignoreKeys, ignoreStartWithKeys);
		
		return element;
	}
	
	@Override
	protected OEdge reallyUpdate() throws NotFoundException, ResourceRegistryException {
		logger.debug("Going to update {} for {}", RelationType.NAME, getName());
		OEdge relationTypeDefinition = getElement();
		relationTypeDefinition = (OEdge) ElementManagement.updateProperties(oClass, relationTypeDefinition, jsonNode,
				ignoreKeys, ignoreStartWithKeys);
		return relationTypeDefinition;
		
	}
	
	@Override
	protected boolean reallyDelete() throws RelationNotFoundException, ResourceRegistryException {
		logger.debug("Going to remove {} for {}", RelationType.NAME, getName());
		getElement().delete();
		return true;
	}
	
	@Override
	public OEdge getElement() throws NotFoundException, ResourceRegistryException {
		if(element == null) {
			try {
				element = retrieveElement();
			} catch(NotFoundException e) {
				throw e;
			} catch(ResourceRegistryException e) {
				throw e;
			} catch(Exception e) {
				throw new ResourceRegistryException(e);
			}
			
		} else {
			if(reload) {
				element.reload();
			}
		}
		return element;
	}
	
	@Override
	public OEdge retrieveElement() throws NotFoundException, ResourceRegistryException {
		try {
			if(getName() == null) {
				throw new NotFoundException("null name does not allow to retrieve the Element");
			}
			
			String select = "SELECT FROM " + elementType + " WHERE " + RelationType.NAME_PROPERTY + " = \""
					+ getName() + "\"";
			
			OResultSet resultSet = oDatabaseDocument.query(select, new HashMap<>());
			
			if(resultSet == null || !resultSet.hasNext()) {
				String error = String.format("No %s with name %s was found", elementType, getName());
				logger.info(error);
				throw new NotFoundException(error);
			}
			
			OResult oResult = resultSet.next();
			OEdge element = (OEdge) ElementManagement.getElementFromOptional(oResult.getElement());
			
			logger.trace("{} with id {} is : {}", elementType, getName(), Utility.toJsonString(element, true));
			
			if(resultSet.hasNext()) {
				throw new ResourceRegistryException("Found more than one " + elementType + " with name " + getName()
						+ ". This is a fatal error please contact Admnistrator");
			}
			
			return element;
		} catch(NotFoundException e) {
			throw getSpecificElementNotFoundException(e);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
	
	@Override
	public String reallyGetAll(boolean polymorphic) throws ResourceRegistryException {
		throw new UnsupportedOperationException();
	}
	
	@Override
	protected boolean reallyAddToContext(SecurityContext targetSecurityContext)
			throws ContextException, ResourceRegistryException {
		throw new UnsupportedOperationException();
	}
	
	@Override
	protected boolean reallyRemoveFromContext(SecurityContext targetSecurityContext)
			throws ContextException, ResourceRegistryException {
		throw new UnsupportedOperationException();
	}
	
	@Override
	protected NotFoundException getSpecificElementNotFoundException(NotFoundException e) {
		return new SchemaNotFoundException(e.getMessage(), e.getCause());
	}
	
	@Override
	protected AlreadyPresentException getSpecificERAlreadyPresentException(String message) {
		return new SchemaAlreadyPresentException(message);
	}
	
	@Override
	protected AvailableInAnotherContextException getSpecificERAvailableInAnotherContextException(String message) {
		throw new UnsupportedOperationException();
	}
	
	@Override
	protected ResourceTypeDefinitionManagement newSourceEntityManagement() throws ResourceRegistryException {
		return new ResourceTypeDefinitionManagement(getWorkingContext(), oDatabaseDocument);
	}
	
}
