/****************************************************************************
 *  This software is part of the gCube Project.
 *  Site: http://www.gcube-system.org/
 ****************************************************************************
 * The gCube/gCore software is licensed as Free Open Source software
 * conveying to the EUPL (http://ec.europa.eu/idabc/eupl).
 * The software and documentation is provided by its authors/distributors
 * "as is" and no expressed or
 * implied warranty is given for its use, quality or fitness for a
 * particular case.
 ****************************************************************************
 * Filename: ServiceProxyImpl.java
 ****************************************************************************
 * @author <a href="mailto:daniele.strollo@isti.cnr.it">Daniele Strollo</a>
 ***************************************************************************/

package org.gcube.portlets.admin.resourcemanagement.server;

import java.io.File;
import java.security.InvalidParameterException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.servlet.http.HttpSession;

import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.informationsystem.client.ISClient;
import org.gcube.common.core.informationsystem.client.QueryParameter;
import org.gcube.common.core.informationsystem.client.XMLResult;
import org.gcube.common.core.informationsystem.client.queries.GCUBEGenericQuery;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.portlets.admin.resourcemanagement.client.remote.ServiceProxy;
import org.gcube.portlets.admin.resourcemanagement.server.gcube.services.StatusHandler;
import org.gcube.portlets.admin.resourcemanagement.server.gcube.services.configuration.ConfigurationLoader;
import org.gcube.portlets.admin.resourcemanagement.shared.exceptions.InvalidPermissionsException;
import org.gcube.resourcemanagement.support.client.utils.CurrentStatus;
import org.gcube.resourcemanagement.support.client.views.ResourceTypeDecorator;
import org.gcube.resourcemanagement.support.server.gcube.CacheManager;
import org.gcube.resourcemanagement.support.server.gcube.ISClientRequester;
import org.gcube.resourcemanagement.support.server.gcube.queries.QueryLoader;
import org.gcube.resourcemanagement.support.server.gcube.queries.QueryLocation;
import org.gcube.resourcemanagement.support.server.managers.resources.GenericResourceManager;
import org.gcube.resourcemanagement.support.server.managers.resources.ManagementUtils;
import org.gcube.resourcemanagement.support.server.managers.resources.ResourceFactory;
import org.gcube.resourcemanagement.support.server.managers.scope.ScopeManager;
import org.gcube.resourcemanagement.support.server.types.AllowedResourceTypes;
import org.gcube.resourcemanagement.support.server.utils.ServerConsole;
import org.gcube.resourcemanagement.support.shared.operations.SupportedOperations;
import org.gcube.resourcemanagement.support.shared.plugins.GenericResourcePlugin;
import org.gcube.resourcemanagement.support.shared.plugins.TMPluginFormField;
import org.gcube.resourcemanagement.support.shared.types.Tuple;
import org.gcube.resourcemanagement.support.shared.types.UserGroup;
import org.gcube.resourcemanagement.support.shared.types.datamodel.CompleteResourceProfile;
import org.gcube.resourcemanagement.support.shared.types.datamodel.ResourceDescriptor;
import org.gcube.resourcemanagement.support.shared.util.Assertion;
import org.gcube.vremanagement.resourcemanager.stubs.reporting.ReportingPortType;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.server.rpc.RemoteServiceServlet;

/**
 * @author Daniele Strollo
 * @author Massimiliano Assante (ISTI-CNR)
 * 
 *
 */
@SuppressWarnings("serial")
public class ServiceProxyImpl extends RemoteServiceServlet implements ServiceProxy {
	private static final String LOG_PREFIX = "[SERVICE-IMPL]";

	public final void emptyCache() {
		ISClientRequester.emptyCache();
	}

	public final void setUseCache(final boolean flag) {
		this.getCurrentStatus().setUseCache(flag);
	}

	public final void setSuperUser(final boolean superUser) {
		if (superUser && this.getCurrentStatus().getCredentials() == UserGroup.DEBUG) {
			this.getCurrentStatus().setCredentials(UserGroup.ADMIN);
		} else {
			this.getCurrentStatus().setCredentials(UserGroup.DEBUG);
		}
		initScopes(true);
	}

	/**
	 * Called by portlet at module instantiation
	 */
	public final CurrentStatus initStatus() {
		ServerConsole.trace(LOG_PREFIX, "[INIT-STATUS] initializing config parameters");

		ServerConsole.info(LOG_PREFIX, "Clearing status");
		StatusHandler.clearStatus(getHttpSession());

		ConfigurationLoader.setConfigurationFile(getPropertiesFSPath() + File.separator + "resourcemanagement.properties");
		return this.getCurrentStatus();
	}

	private HttpSession getHttpSession() {
		return this.getThreadLocalRequest().getSession();
	}

	public final void initScopes(final boolean doClean) {
		ServerConsole.trace(LOG_PREFIX, "[INIT-SCOPES] initializing scopes from: " + this.getScopeDataPath());

		// Updates the scopes
		try {
			ScopeManager.setScopeConfigFile(this.getScopeDataPath());
			if (doClean) {
				ScopeManager.clear();
			}
			ScopeManager.update();
		} catch (Exception e) {
			ServerConsole.error(LOG_PREFIX, "During scope caching", e);
		}

	}

	private String getServletFSPath() {
		return this.getServletContext().getRealPath("") + File.separator + "WEB-INF";
	}

	private String getPropertiesFSPath() {
		return this.getServletContext().getRealPath("") + File.separator + "conf";
	}

	private String getScopeDataPath() { 
		String startDirectory = getServletFSPath();
		return startDirectory + File.separator + "xml" + File.separator +
				((this.getCurrentStatus().getCredentials() == UserGroup.ADMIN) ?
						"scopedata_admin.xml" :
						"scopedata.xml");
	}

	private String getRepositoryUpdgradeDirPath() {
		String startDirectory = getServletFSPath();
		return startDirectory + File.separator + "reports";
	}

	private String getXML2HTMLMapping() {
		String startDirectory = getServletFSPath();
		return startDirectory + File.separator + "xml" + File.separator + "xmlverbatim.xsl";
	}


	public final Tuple<String> addResourcesToScope(
			final String resType,
			final List<String> resourceIDs,
			final String scope)
					throws Exception {
		Assertion<Exception> checker = new Assertion<Exception>();

		CurrentStatus status = this.getCurrentStatus();

		checker.validate(SupportedOperations.ADD_TO_SCOPE.isAllowed(status.getCredentials()),
				new Exception("User not allowed to request this operation"));
		checker.validate(resType != null && resType.trim().length() > 0, new Exception("Invalid parameter type"));
		checker.validate(resourceIDs != null && resourceIDs.size() > 0, new Exception("Invalid parameter type"));
		checker.validate(scope != null && scope.trim().length() > 0, new Exception("Invalid parameter type"));

		AllowedResourceTypes type = AllowedResourceTypes.valueOf(resType);
		GCUBEScope targetScope = ScopeManager.getScope(scope);
		GCUBEScope sourceScope = ScopeManager.getScope(status.getCurrentScope());

		String reportID = ManagementUtils.addToExistingScope(type,
				resourceIDs.toArray(new String[]{}),
				sourceScope,
				targetScope);

		ReportingPortType manager =
				ResourceFactory.createResourceManager(type).getReportResourceManager(targetScope);

		String xmlReport = manager.getReport(reportID);
		String mappingPath = this.getXML2HTMLMapping();
		String htmlReport = ISClientRequester.XML2HTML(xmlReport, mappingPath);
		return new Tuple<String>(
				reportID,
				resType,
				xmlReport,
				htmlReport);
	}
	/**
	 * REMOVE FROM SCOPE
	 */
	public final Tuple<String> removeResourcesFromScope(
			final String resType,
			final List<String> resourceIDs,
			final String scope)
					throws Exception {
		Assertion<Exception> checker = new Assertion<Exception>();

		CurrentStatus status = this.getCurrentStatus();

		checker.validate(SupportedOperations.ADD_TO_SCOPE.isAllowed(status.getCredentials()),
				new Exception("User not allowed to request this operation"));
		checker.validate(resType != null && resType.trim().length() > 0, new Exception("Invalid parameter type"));
		checker.validate(resourceIDs != null && resourceIDs.size() > 0, new Exception("Invalid parameter type"));
		checker.validate(scope != null && scope.trim().length() > 0, new Exception("Invalid parameter type"));

		AllowedResourceTypes type = AllowedResourceTypes.valueOf(resType);
		GCUBEScope targetScope = ScopeManager.getScope(scope);
		GCUBEScope sourceScope = ScopeManager.getScope(status.getCurrentScope());

		return new Tuple<String>(ManagementUtils.removeFromExistingScope(type,
				resourceIDs.toArray(new String[]{}),
				sourceScope,
				targetScope));
	}

	public final String deploy(final List<String> ghnsID, final List<String> servicesID) throws Exception {
		try {
			GCUBEScope sourceScope = ScopeManager.getScope(this.getCurrentStatus().getCurrentScope());
			String[] param1 = ghnsID.toArray(new String[0]);
			String[] param2 = servicesID.toArray(new String[0]);
			return ManagementUtils.deploy(sourceScope, param1, param2);
		} catch (Exception e) {
			ServerConsole.error(LOG_PREFIX, "During service deployment", e);
			throw new Exception(e.getMessage());
		}
	}

	public final Tuple<String> checkDeployStatus(final String scope, final String deployID) throws Exception {
		if (SupportedOperations.SERVICE_GET_REPORT.isAllowed(this.getCurrentStatus().getCredentials())) {
			GCUBEScope sourceScope = ScopeManager.getScope(scope);
			ReportingPortType manager =
					ResourceFactory.createResourceManager(AllowedResourceTypes.Service).getReportResourceManager(sourceScope);
			String xmlReport = manager.getReport(deployID);
			String mappingPath = this.getXML2HTMLMapping();
			String htmlReport = ISClientRequester.XML2HTML(xmlReport, mappingPath);
			return new Tuple<String>(deployID, xmlReport, htmlReport);
		} else {
			throw new Exception("The current user is not allowed to request the operation");
		}
	}

	public final List<String> getAvailableScopes() {
		ServerConsole.trace(LOG_PREFIX, "[GET-SCOPES] getting available scopes");
		GWT.log("The CONTEXT PATH is: " + this.getScopeDataPath());
		Vector<String> retval = new Vector<String>();

		try {
			Map<String, GCUBEScope> scopes = ScopeManager.getAvailableScopes();
			for (GCUBEScope scope : scopes.values()) {
				//if ((scope.getType().compareTo(Type.INFRASTRUCTURE) == 0)
				//		|| (scope.getType().compareTo(Type.VO) == 0)) {
				retval.add(scope.toString());
				//}
			}
			return retval;
		} catch (Exception e) {
			retval.add("/gcube");
			retval.add("/gcube/devsec");
		}
		return retval;
	}

	public final List<String> getAvailableAddScopes() {
		List<String> retval = new Vector<String>();
		try {
			GCUBEScope currScope = ScopeManager.getScope(this.getCurrentStatus().getCurrentScope());
			List<String> scopes = this.getAvailableScopes();
			for (String scope : scopes) {
				GCUBEScope trgScope = ScopeManager.getScope(scope);
				if (trgScope.isEnclosedIn(currScope)) {
					retval.add(trgScope.toString());
				}
			}
			return retval;
		} catch (Exception e) {
			ServerConsole.error(LOG_PREFIX, e);
		}
		return null;
	}

	public final HashMap<String, ArrayList<String>> getResourceTypeTree(final String scope) throws Exception {
		try {
			GCUBEScope gscope = ScopeManager.getScope(scope);
			HashMap<String, ArrayList<String>>  results = ISClientRequester.getResourcesTree(getCacheManager(this.getCurrentStatus()), gscope);
			return results;
		} catch (Exception e) {
			ServerConsole.error(LOG_PREFIX, "applying resource get", e);
			throw e;
		}
	}

	public final List<String> getRelatedResources(final String type, final String id, final String scope)
	{
		try {
			return ISClientRequester.getRelatedResources(
					getCacheManager(this.getCurrentStatus()),
					type,
					id,
					ScopeManager.getScope(scope)
					);
		} catch (Exception e) {
			ServerConsole.error(LOG_PREFIX, e);
			return null;
		}
	}

	public final List<String> getResourcesByType(final String scope, final String type)
	{
		try {
			return ISClientRequester.getResourcesByType(
					getCacheManager(this.getCurrentStatus()),
					ScopeManager.getScope(scope), type, null);
		} catch (Exception e) {
			ServerConsole.error(LOG_PREFIX, e);
			return null;
		}
	}

	public final ResourceDescriptor getGenericResourceDescriptor(final String scope, final String resID)
			throws Exception {
		return this.getDescriptor(ScopeManager.getScope(scope), resID);
	}

	private ResourceDescriptor getDescriptor(final GCUBEScope scope, final String resID)
			throws Exception {
		ISClient client = GHNContext.getImplementation(ISClient.class);
		GCUBEGenericQuery isQuery = null;
		isQuery = client.getQuery(GCUBEGenericQuery.class);

		isQuery.setExpression(QueryLoader.getQuery(QueryLocation.GET_RESOURCE_BYID));
		isQuery.addParameters(
				new QueryParameter("RES_ID", resID),
				new QueryParameter("RES_TYPE", ResourceTypeDecorator.GenericResource.name())
				);

		List<XMLResult> results = client.execute(isQuery, scope);

		if (results != null && results.size() > 0) {
			ResourceDescriptor retval = new ResourceDescriptor();
			XMLResult node = results.get(0);

			List<String> currValue = null;

			currValue = node.evaluate("/Resource/ID/text()");
			if (currValue != null && currValue.size() > 0) {
				retval.setID(currValue.get(0));
			}
			currValue = node.evaluate("/Resource/Profile/Name/text()");
			if (currValue != null && currValue.size() > 0) {
				retval.setName(currValue.get(0));
			}

			currValue = node.evaluate("/Resource/Profile/Body/node()");
			if (currValue != null && currValue.size() > 0) {
				StringBuilder bodytext = new StringBuilder();
				for (String line : currValue) {
					bodytext.append(line);
				}
				retval.set("Body", bodytext.toString().trim());
			} else {
				retval.set("Body", "");
			}


			currValue = node.evaluate("/Resource/Profile/Description/text()");
			if (currValue != null && currValue.size() > 0) {
				retval.set("Description", currValue.get(0));
			} else {
				retval.set("Description", "");
			}


			currValue = node.evaluate("/Resource/Profile/SecondaryType/text()");
			if (currValue != null && currValue.size() > 0) {
				retval.set("SecondaryType", currValue.get(0));
			} else {
				retval.set("SecondaryType", "");
			}

			return retval;
		}
		return null;
	}

	public final List<ResourceDescriptor> getResourcesModel(final String scope, final String type, final String subType, final List<Tuple<String>> additionalMaps)
			throws Exception {
		if (scope == null || type == null) {
			return null;
		}
		return ISClientRequester.getResourceModels(ScopeManager.getScope(scope), type, subType, additionalMaps);
	}


	public final List<String> getWSResources(final String scope) {
		try {
			return ISClientRequester.getWSResources(ScopeManager.getScope(scope));
		} catch (Exception e) {
			ServerConsole.error(LOG_PREFIX, e);
			return null;
		}
	}

	public final List<String> getResourcesBySubType(final String scope, final String type, final String subtype) {
		try {
			return ISClientRequester.getResourcesByType(
					getCacheManager(this.getCurrentStatus()),
					ScopeManager.getScope(scope), type, subtype);
		} catch (Exception e) {
			ServerConsole.error(LOG_PREFIX, e);
			return null;
		}
	}

	public final CompleteResourceProfile getResourceByID(final String scope, final String type, final String resID) {
		try {
			CompleteResourceProfile profile = ISClientRequester.getResourceByID(
					this.getXML2HTMLMapping(),
					ScopeManager.getScope(scope),
					type,
					resID);
			return profile;
		} catch (Exception e) {
			ServerConsole.error(LOG_PREFIX, e);
			return null;
		}
	}

	public final String createGenericResource(
			final String ID,
			final String name,
			final String description,
			final String body,
			final String subType)
					throws Exception {
		// Check permissions
		Assertion<InvalidPermissionsException> checker = new Assertion<InvalidPermissionsException>();
		checker.validate(
				SupportedOperations.GENERIC_RESOURCE_CREATE.isAllowed(this.getCurrentStatus().getCredentials()),
				new InvalidPermissionsException("The user is not allowed to execute the following operation"));

		return GenericResourceManager.create(
				ID,
				ScopeManager.getScope(this.getCurrentStatus().getCurrentScope()),
				name,
				description,
				body,
				subType);
	}

	public final void updateGenericResource(
			final String ID,
			final String name,
			final String description,
			final String body,
			final String subType)
					throws Exception {
		// Check permissions
		Assertion<InvalidPermissionsException> checker = new Assertion<InvalidPermissionsException>();
		checker.validate(
				SupportedOperations.GENERIC_RESOURCE_EDIT.isAllowed(this.getCurrentStatus().getCredentials()),
				new InvalidPermissionsException("The user is not allowed to execute the following operation"));

		GCUBEScope sourceScope = ScopeManager.getScope(this.getCurrentStatus().getCurrentScope());
		GenericResourceManager resource = new GenericResourceManager(ID);
		resource.update(name, description, body, subType, sourceScope);
	}

	public final void setCurrentScope(final String scope) {
		this.getCurrentStatus().setCurrentScope(scope);
	}

	public final Map<String, GenericResourcePlugin> getGenericResourcePlugins() throws Exception {
		return ISClientRequester.getGenericResourcePlugins(ScopeManager.getScope(this.getCurrentStatus().getCurrentScope()));		
	}
	

	public final HashMap<String, ArrayList<TMPluginFormField>> getGenericResourceTreeMagnagerPlugins() throws Exception {
		return ISClientRequester.getGenericResourceTreeManagerPlugins(ScopeManager.getScope(this.getCurrentStatus().getCurrentScope()));
	}

	private CurrentStatus getCurrentStatus() {
		return StatusHandler.getStatus(this.getHttpSession());
	}
	
	private CacheManager getCacheManager(CurrentStatus status) {
		CacheManager cm = new CacheManager();
		cm.setUseCache(status.useCache());
		return cm;
	}
	/***********************************************************
	 * RESOURCE OPERATIONS
	 * @throws InvalidParameterException
	 **********************************************************/
	public final void doOperation(
			final SupportedOperations opCode,
			final String scope,
			final List<ResourceDescriptor> resources)
					throws Exception {
		try {
			ResourceCommands.doOperation(
					this.getCurrentStatus(),
					opCode, scope, resources);
		} catch (final Exception e) {
			ServerConsole.error(LOG_PREFIX, e);
			throw new Exception(e.getMessage());
		}
	}
}
