package org.gcube.portlets.user.homelibrary.jcr.workspace;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import javax.jcr.ItemExistsException;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.lock.LockException;
import javax.jcr.nodetype.ConstraintViolationException;
import javax.jcr.version.VersionException;

import org.apache.commons.lang.Validate;
import org.apache.jackrabbit.util.Text;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.portlets.user.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.portlets.user.homelibrary.home.workspace.Properties;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.DocumentMetadata;
import org.gcube.portlets.user.homelibrary.jcr.content.JCRRepository;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.JCRDocumentMetadata;

public class JCRProperties implements Properties {
	
	private static GCUBELog logger = new GCUBELog(JCRProperties.class);
	private static final String PROPERTIES = "hl:metadata";
	private String identifier;
	
	public JCRProperties(Node itemNode) throws RepositoryException {
		this.identifier = itemNode.getIdentifier();
	}
	
	@Override
	public String getId() throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return node.getNode(PROPERTIES).getIdentifier();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}	
	}

	@Override
	public String getPropertyValue(String propertyName) throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try {
			Node itemNode = session.getNodeByIdentifier(identifier);
			return itemNode.getNode(PROPERTIES).getProperty(propertyName).getString();
		} catch (PathNotFoundException e) {
			return null;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public Map<String, String> getProperties() throws InternalErrorException {
		
		Map<String,String> map = new HashMap<String,String>();
		
		Session session = JCRRepository.getSession();
		Node itemNode = null;
		try {
			itemNode = session.getNodeByIdentifier(identifier);
			
			Node propertiesNode = itemNode.getNode(PROPERTIES);
			
			for (PropertyIterator iterator = propertiesNode.getProperties();
					iterator.hasNext();) {
				Property property = iterator.nextProperty();
				if(!property.getName().startsWith("jcr:")) {
					String unescapeName = Text.unescape(property.getName());
					map.put(unescapeName,
							property.getValue().toString());
				}
			}
			
			return Collections.unmodifiableMap(map);
		} catch (PathNotFoundException e) {
			if (itemNode!= null) {
				try {
					itemNode.addNode(PROPERTIES);
					session.save();
					return map; 
				} catch (RepositoryException e1) {
					logger.error("Error to add properties ",e1);
					throw new InternalErrorException(e1.getMessage());
				}
			} else {
				throw new InternalErrorException(e.getMessage());
			}
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public void addProperty(String name, String value) throws InternalErrorException {
		
		Validate.notNull(name,"Name property must be not null");
		Validate.notNull(value,"Value property must be not null");
		
		Session session = JCRRepository.getSession();
		try {
			Node itemNode = session.getNodeByIdentifier(identifier);
			String escapeName = Text.escapeIllegalJcrChars(name);
			itemNode.getNode(PROPERTIES).setProperty(escapeName, value);
			session.save();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

}
