package org.gcube.portlets.user.joinvre.service.impl;

import java.util.ArrayList;
import java.util.List;

import org.gcube.portlets.user.joinvre.model.TabbedPage;
import org.gcube.portlets.user.joinvre.model.UserBelonging;
import org.gcube.portlets.user.joinvre.model.VRE;
import org.gcube.portlets.user.joinvre.model.VRECategory;
import org.gcube.portlets.user.joinvre.service.JoinVREService;

import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.model.Group;
import com.liferay.portal.kernel.model.GroupConstants;
import com.liferay.portal.kernel.model.User;
import com.liferay.portal.kernel.model.LayoutSet;
import com.liferay.portal.kernel.service.GroupLocalService;
import com.liferay.portal.kernel.service.LayoutLocalService;
import com.liferay.portal.kernel.service.LayoutSetLocalService;
import com.liferay.portal.kernel.service.UserLocalService;
import com.liferay.portal.kernel.util.Portal;
import com.liferay.portal.kernel.util.LocaleUtil;
import com.liferay.portal.kernel.util.PortalUtil;
import com.liferay.portal.kernel.webserver.WebServerServletTokenUtil;

import java.util.HashSet;
import java.util.Set;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * JoinVRE Service Implementation - Migrated from 6.2 to 7.4
 * Uses hybrid SG3/SG4 strategy with mock implementations for unavailable dependencies
 * 
 * @author Massimiliano Assante, ISTI-CNR (original)
 * @author netfarm-m2 (migration to 7.4)
 */
@Component(
	immediate = true,
	service = JoinVREService.class
)
public class JoinVREServiceImpl implements JoinVREService {
	
	private static final Log _log = LogFactoryUtil.getLog(JoinVREServiceImpl.class);
	
	// Liferay 7.4 Services
	@Reference
	private GroupLocalService _groupLocalService;
	
	@Reference
	private UserLocalService _userLocalService;
	
	@Reference
	private Portal _portal;
	
	@Reference
	private LayoutSetLocalService _layoutSetLocalService;
	
	// Constants from original implementation
	public static final String PREFIX_PUBLIC_URL = "/web";
	public static final String TABBED_LAYOUT_ATTRIBUTE = "TabbedLayout";
	public static final String TAB_NAMES_ATTRIBUTE = "TabName";
	public static final String ORGANIZATION_NAMES_ATTRIBUTE = "OrganisationName";
	public static final String ALLVRES_SESSION_ATTRIBUTE = "ALLVRES_SESSION";
	
	@Override
	public String joinVRE(Long vreID, User user) {
		try {
			_log.info("User " + user.getScreenName() + " joining VRE: " + vreID);
			
			// Get the group/site for the VRE
			Group group = _groupLocalService.getGroup(vreID);
			
			// TODO: Implement actual join logic with gCube APIs when SG4 is available
			// For now, return the group's friendly URL
			return "/web" + group.getFriendlyURL();
			
		} catch (Exception e) {
			_log.error("Error joining VRE " + vreID + " for user " + user.getScreenName(), e);
			return "";
		}
	}
	
	@Override
	public Object readInvite(String inviteId, long siteId) {
		try {
			_log.info("Reading invite: " + inviteId + " for site: " + siteId);
			
			// TODO: Implement with gCube databook when SG4 is available
			// Mock implementation for now
			return createMockUserInfo(inviteId, siteId);
			
		} catch (Exception e) {
			_log.error("Error reading invite " + inviteId, e);
			return null;
		}
	}
	
	@Override
	public String getTermsOfUse(long siteId) {
		try {
			_log.debug("Getting Terms of Use for site: " + siteId);
			
			// TODO: Implement with gCube Terms of Use when SG4 is available
			// Mock implementation for now
			Group group = _groupLocalService.getGroup(siteId);
			return "Terms of Use for " + group.getName() + " (Mock implementation)";
			
		} catch (Exception e) {
			_log.error("Error getting Terms of Use for site " + siteId, e);
			return null;
		}
	}
	
	@Override
	public List<TabbedPage> isTabbedPanel(long groupId) {
		try {
			_log.debug("Checking if group " + groupId + " is tabbed panel");
			
			// TODO: Implement with gCube group manager when SG4 is available
			// Mock implementation for now
			List<TabbedPage> tabs = new ArrayList<>();
			tabs.add(new TabbedPage("Home", "/web/guest", "Home page", true));
			tabs.add(new TabbedPage("Workspace", "/group/workspace", "Workspace", false));
			
			return tabs;
			
		} catch (Exception e) {
			_log.error("Error checking tabbed panel for group " + groupId, e);
			return null;
		}
	}
	
	@Override
	public List<VRE> getAllVREs(User user) {
		try {
			_log.info("Getting all VREs for user: " + user.getScreenName());
			
			List<VRE> vres = new ArrayList<>();
			
			// Get all active sites/groups from Liferay
			List<Group> allGroups = _groupLocalService.getActiveGroups(
				_portal.getDefaultCompanyId(), true);
			
			// Get user's groups to determine membership
			List<Group> userGroups = _groupLocalService.getUserGroups(user.getUserId());
			
			for (Group group : allGroups) {
				// Filter for sites that could be VREs (not control panel, not global, etc.)
				if (group.isSite() && 
					!group.isControlPanel() && 
					!group.isCompany() &&
					!group.isGuest() &&
					!group.isUser() &&
					!group.isUserGroup() &&
					group.isActive() &&
					group.getGroupId() != user.getGroupId()) { // Exclude user's personal site
					
					// Get VRE data from Liferay group properties
					String vreName = group.getDescriptiveName(LocaleUtil.getDefault());
					String vreDescription = getGroupDescription(group);
					String vreCategory = getGroupCategory(group);
					
					VRE vre = new VRE();
					vre.setId(group.getGroupId());
					vre.setName(vreName);
					vre.setDescription(vreDescription);
					vre.setUrl("/web" + group.getFriendlyURL());
					vre.setPublic(hasPublicPages(group));
					vre.setOpen(isOpenMembership(group));
					vre.setMembersCount(getUsersCount(group));
					vre.setCategoryName(vreCategory);
					
					// Determine user belonging
					UserBelonging belonging = new UserBelonging();
					belonging.setUserId(user.getUserId());
					belonging.setVreId(group.getGroupId());
					
					if (userGroups.contains(group)) {
						belonging.setMembershipType(UserBelonging.MembershipType.MEMBER);
						belonging.setCanJoin(false); // Already a member
						belonging.setCanRequest(false);
					} else {
						belonging.setMembershipType(UserBelonging.MembershipType.NOT_MEMBER);
						belonging.setCanJoin(isOpenMembership(group));
						belonging.setCanRequest(true);
					}
					
					vre.setUserBelonging(belonging);
					
					// Set thumbnail URL from group logo
					try {
						String logoURL = getGroupLogoURL(group);
						vre.setThumbnailURL(logoURL);
					} catch (Exception e) {
						// Fallback: no thumbnail
						vre.setThumbnailURL(null);
						_log.debug("Could not get thumbnail for group " + group.getName(), e);
					}
					
					vres.add(vre);
					_log.debug("Added VRE: " + vre.getName() + " (ID: " + vre.getId() + ")");
				}
			}
			
			_log.info("Found " + vres.size() + " VREs for user: " + user.getScreenName());
			return vres;
			
		} catch (Exception e) {
			_log.error("Error getting VREs for user " + user.getScreenName(), e);
			return new ArrayList<>();
		}
	}
	
	@Override
	public List<VRE> getAvailableVREs() {
		try {
			_log.info("Getting available VREs (public access)");
			
			List<VRE> vres = new ArrayList<>();
			
			// Get all active sites/groups from Liferay
			List<Group> allGroups = _groupLocalService.getActiveGroups(
				_portal.getDefaultCompanyId(), true);
			
			for (Group group : allGroups) {
				// Filter for sites that could be VREs (not control panel, not global, etc.)
				if (group.isSite() && 
					!group.isControlPanel() && 
					!group.isCompany() &&
					!group.isGuest() &&
					!group.isUser() &&
					!group.isUserGroup() &&
					group.isActive()) {
					
					// Get VRE data from Liferay group properties
					String vreName = group.getName();
					String vreDescription = getGroupDescription(group);
					String vreCategory = getGroupCategory(group);
					boolean publicPages = hasPublicPages(group);
					boolean openMembership = isOpenMembership(group);
					String friendlyURL = "/web" + group.getFriendlyURL();
					
					// Determine membership type
					String membershipType = "PRIVATE";
					if (openMembership) {
						membershipType = "OPEN";
					} else if (publicPages) {
						membershipType = "RESTRICTED";
					}
					
					// Create default UserBelonging for public access
					UserBelonging defaultBelonging = new UserBelonging();
					defaultBelonging.setMembershipType(UserBelonging.MembershipType.NOT_MEMBER);
					defaultBelonging.setCanJoin(openMembership);
					defaultBelonging.setCanRequest(true);
					
					// Create VRE with all fields
					VRE vre = new VRE(
						group.getGroupId(),
						vreName,
						vreDescription,
						getGroupLogoURL(group),
						friendlyURL,
						publicPages,
						openMembership,
						getUsersCount(group),
						vreCategory,
						defaultBelonging,
						membershipType,
						publicPages,
						friendlyURL
					);
					
					vres.add(vre);
					_log.debug("Added VRE: " + vre.getName() + " (ID: " + vre.getId() + ")");
				}
			}
			
			_log.info("Found " + vres.size() + " available VREs");
			return vres;
			
		} catch (Exception e) {
			_log.error("Error getting available VREs", e);
			return new ArrayList<>();
		}
	}
	
	/**
	 * Get group description, fallback to generated description if empty
	 */
	private String getGroupDescription(Group group) {
		String description = group.getDescription();
		if (description != null && !description.trim().isEmpty()) {
			return description;
		}
		return generateVREDescription(group.getName(), null);
	}
	
	/**
	 * Get group category from type settings or generate one
	 */
	private String getGroupCategory(Group group) {
		try {
			String category = group.getTypeSettingsProperty("category");
			if (category != null && !category.isEmpty()) {
				return category;
			}
		} catch (Exception e) {
			_log.debug("No category found for group: " + group.getName());
		}
		return generateVRECategory(group.getName());
	}
	
	/**
	 * Check if group has public pages
	 */
	private boolean hasPublicPages(Group group) {
		try {
			return group.getPublicLayoutsPageCount() > 0;
		} catch (Exception e) {
			_log.debug("Error checking public pages for group: " + group.getName());
			return false;
		}
	}
	
	/**
	 * Check if group has open membership
	 */
	private boolean isOpenMembership(Group group) {
		try {
			return group.getType() == 1; // Open site
		} catch (Exception e) {
			_log.debug("Error checking membership type for group: " + group.getName());
			return false;
		}
	}
	
	/**
	 * Get actual users count for the group
	 */
	private int getUsersCount(Group group) {
		try {
			return _userLocalService.getGroupUsersCount(group.getGroupId());
		} catch (Exception e) {
			_log.debug("Error getting users count for group: " + group.getName());
			return 0;
		}
	}
	
	/**
	 * Get group logo URL
	 */
	private String getGroupLogoURL(Group group) {
		try {
			long logoId = 0L;
			LayoutSet publicLayoutSet = _layoutSetLocalService.getLayoutSet(group.getGroupId(), false);
			if (publicLayoutSet != null) {
				logoId = publicLayoutSet.getLogoId();
			}
			if (logoId <= 0) {
				LayoutSet privateLayoutSet = _layoutSetLocalService.getLayoutSet(group.getGroupId(), true);
				if (privateLayoutSet != null) {
					logoId = privateLayoutSet.getLogoId();
				}
			}
			if (logoId > 0) {
				String token = WebServerServletTokenUtil.getToken(logoId);
				String pathImage = PortalUtil.getPathImage();
				return pathImage + "/group_logo?img_id=" + logoId + "&t=" + token;
			}
		} catch (Exception e) {
			_log.debug("Error getting logo URL for group: " + group.getName(), e);
		}
		return null;
	}
	
	@Override
	public List<VRECategory> getVRECategories() {
		try {
			_log.debug("Getting VRE categories");
			
			// Get categories from actual VREs in the system
			List<VRECategory> categories = new ArrayList<>();
			List<Group> allGroups = _groupLocalService.getActiveGroups(
				_portal.getDefaultCompanyId(), true);
			
			// Collect unique categories from existing VREs
			Set<String> categoryNames = new HashSet<>();
			
			for (Group group : allGroups) {
				if (group.isSite() && !group.isControlPanel() && 
					!group.isCompany() && !group.isGuest() && 
					!group.isUser() && !group.isUserGroup() && 
					group.isActive()) {
					
					String categoryName = getGroupCategory(group);
					categoryNames.add(categoryName);
				}
			}
			
			// Create VRECategory objects
			for (String categoryName : categoryNames) {
				VRECategory category = new VRECategory();
				category.setName(categoryName);
				category.setDescription(getCategoryDescription(categoryName));
				category.setVres(new ArrayList<>()); // VREs will be populated separately if needed
				categories.add(category);
			}
			
			// Sort categories alphabetically
			categories.sort((c1, c2) -> c1.getName().compareTo(c2.getName()));
			
			_log.debug("Found " + categories.size() + " VRE categories");
			return categories;
			
		} catch (Exception e) {
			_log.error("Error getting VRE categories", e);
			return new ArrayList<>();
		}
	}
	
	/**
	 * Get description for a category
	 */
	private String getCategoryDescription(String categoryName) {
		switch (categoryName) {
			case "Testing & Validation":
				return "Environments for testing and validating research methodologies";
			case "Environmental Sciences":
				return "Research environments focused on environmental and sustainability studies";
			case "Development & Innovation":
				return "Platforms for developing next-generation research tools";
			case "Data Management":
				return "Comprehensive data management and analysis environments";
			case "Life Sciences":
				return "Research environments for biological and life science studies";
			case "Earth Sciences":
				return "Platforms for geological and earth science research";
			case "Social Sciences":
				return "Collaborative environments for social science research";
			default:
				return "Research and development environments for " + categoryName.toLowerCase();
		}
	}
	
	@Override
	public UserBelonging getUserBelonging(User user, long vreId) {
		try {
			_log.debug("Getting user belonging for user " + user.getScreenName() + " and VRE " + vreId);
			
			// Check if user is member of the group
			boolean isMember = _groupLocalService.hasUserGroup(user.getUserId(), vreId);
			
			UserBelonging belonging = new UserBelonging();
			belonging.setUserId(user.getUserId());
			belonging.setVreId(vreId);
			
			if (isMember) {
				belonging.setMembershipType(UserBelonging.MembershipType.MEMBER);
				belonging.setCanJoin(false);
				belonging.setCanRequest(false);
			} else {
				belonging.setMembershipType(UserBelonging.MembershipType.NOT_MEMBER);
				belonging.setCanJoin(true);
				belonging.setCanRequest(true);
			}
			
			return belonging;
			
		} catch (Exception e) {
			_log.error("Error getting user belonging", e);
			return null;
		}
	}
	
	@Override
	public boolean requestMembership(User user, long vreId, String comments) {
		try {
			_log.info("User " + user.getScreenName() + " requesting membership to VRE " + vreId);
			
			// Get the group/site
			Group group = _groupLocalService.getGroup(vreId);
			
			// Check if group allows membership requests
			if (!group.isActive()) {
				_log.warn("Cannot request membership to inactive group: " + vreId);
				return false;
			}
			
			// For open groups, add user directly
			if (isOpenMembership(group)) {
				_userLocalService.addGroupUser(vreId, user.getUserId());
				_log.info("User " + user.getScreenName() + " added directly to open VRE " + vreId);
				return true;
			} else {
				// For restricted/private groups, create a membership request
				// TODO: Implement proper membership request workflow when gCube APIs are available
				// For now, log the request for manual processing
				_log.info("Membership request logged for manual processing: User " + 
					user.getScreenName() + " -> VRE " + group.getName() + " (" + vreId + ")");
				if (comments != null && !comments.trim().isEmpty()) {
					_log.info("Request comments: " + comments);
				}
				return true; // Request submitted successfully
			}
			
		} catch (Exception e) {
			_log.error("Error requesting membership for user " + user.getScreenName() + " to VRE " + vreId, e);
			return false;
		}
	}
	
	/**
	 * Create user info from invite data
	 */
	private Object createMockUserInfo(String inviteId, long siteId) {
		try {
			// TODO: Replace with actual gCube invite processing when SG4 is available
			// For now, try to extract user information from Liferay if possible
			
			// Try to find user by invite ID (could be email or username)
			User invitedUser = null;
			try {
				// Try to find user by email
				invitedUser = _userLocalService.getUserByEmailAddress(
					_portal.getDefaultCompanyId(), inviteId);
			} catch (Exception e) {
				try {
					// Try to find user by screen name
					invitedUser = _userLocalService.getUserByScreenName(
						_portal.getDefaultCompanyId(), inviteId);
				} catch (Exception e2) {
					// User not found in Liferay
					_log.debug("User not found for invite ID: " + inviteId);
				}
			}
			
			final User finalUser = invitedUser;
			final Group site = _groupLocalService.getGroup(siteId);
			
			return new Object() {
				public String getUsername() { 
					return finalUser != null ? finalUser.getScreenName() : inviteId; 
				}
				public String getFullname() { 
					return finalUser != null ? finalUser.getFullName() : "Invited User"; 
				}
				public String getUserAvatarURL() { 
					if (finalUser != null) {
						try {
							return finalUser.getPortraitURL(null);
						} catch (PortalException e) {
							_log.debug("Error getting portrait URL for user: " + finalUser.getScreenName());
							return "/images/user_portrait.png";
						}
					}
					return "/images/user_portrait.png"; 
				}
				public String getAccountURL() { 
					return finalUser != null ? "/web/" + finalUser.getScreenName() : ""; 
				}
				public String getTermsOfUse() { 
					return JoinVREServiceImpl.this.getTermsOfUse(siteId); 
				}
				public boolean isOnline() { 
					return finalUser != null && finalUser.isActive(); 
				}
				public boolean isAway() { 
					return false; 
				}
			};
			
		} catch (Exception e) {
			_log.error("Error creating user info for invite " + inviteId, e);
			return createFallbackUserInfo(inviteId, siteId);
		}
	}
	
	/**
	 * Create fallback user info when user lookup fails
	 */
	private Object createFallbackUserInfo(String inviteId, long siteId) {
		return new Object() {
			public String getUsername() { return inviteId; }
			public String getFullname() { return "Invited User"; }
			public String getUserAvatarURL() { return "/images/user_portrait.png"; }
			public String getAccountURL() { return ""; }
			public String getTermsOfUse() { return JoinVREServiceImpl.this.getTermsOfUse(siteId); }
			public boolean isOnline() { return false; }
			public boolean isAway() { return false; }
		};
	}
	
	/**
	 * Generate a realistic VRE description based on the name and existing description
	 */
	private String generateVREDescription(String vreName, String existingDescription) {
		if (existingDescription != null && !existingDescription.trim().isEmpty() && 
			!existingDescription.equals(vreName)) {
			return existingDescription;
		}
		
		// Generate description based on VRE name patterns
		String lowerName = vreName.toLowerCase();
		
		if (lowerName.contains("gateway") || lowerName.contains("test")) {
			return "A test environment for exploring and validating research methodologies, " +
				   "providing researchers with tools and resources for experimental activities " +
				   "in a controlled setting.";
		} else if (lowerName.contains("cloud") || lowerName.contains("green")) {
			return "A development environment focused on cloud computing and sustainable " +
				   "technologies, supporting research in environmental sciences and " +
				   "green computing initiatives.";
		} else if (lowerName.contains("next") || lowerName.contains("dev")) {
			return "A development platform for next-generation research tools and " +
				   "methodologies, providing cutting-edge resources for innovative " +
				   "scientific projects.";
		} else if (lowerName.contains("data") || lowerName.contains("catalogue")) {
			return "A comprehensive data management environment providing tools for " +
				   "data discovery, cataloguing, and analysis across multiple research " +
				   "domains and disciplines.";
		} else {
			return "A collaborative research environment providing specialized tools " +
				   "and resources for scientific research and data analysis in the " +
				   vreName + " domain.";
		}
	}
	
	/**
	 * Generate a realistic VRE category based on the name
	 */
	private String generateVRECategory(String vreName) {
		String lowerName = vreName.toLowerCase();
		
		if (lowerName.contains("gateway") || lowerName.contains("test")) {
			return "Testing & Validation";
		} else if (lowerName.contains("cloud") || lowerName.contains("green")) {
			return "Environmental Sciences";
		} else if (lowerName.contains("next") || lowerName.contains("dev")) {
			return "Development & Innovation";
		} else if (lowerName.contains("data") || lowerName.contains("catalogue")) {
			return "Data Management";
		} else if (lowerName.contains("bio") || lowerName.contains("life")) {
			return "Life Sciences";
		} else if (lowerName.contains("geo") || lowerName.contains("earth")) {
			return "Earth Sciences";
		} else if (lowerName.contains("social") || lowerName.contains("humanities")) {
			return "Social Sciences";
		} else {
			return "Research & Development";
		}
	}
}