/**
 * 
 */
package org.gcube.resourcemanagement.model.reference.entities.facets;

import java.util.List;

import org.gcube.com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import org.gcube.com.fasterxml.jackson.databind.node.ArrayNode;
import org.gcube.com.fasterxml.jackson.databind.node.ObjectNode;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.queries.templates.reference.properties.QueryTemplateReference;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.types.reference.Change;
import org.gcube.informationsystem.types.reference.TypeMetadata;
import org.gcube.informationsystem.utils.Version;
import org.gcube.resourcemanagement.model.impl.entities.facets.DiscoveryFacetImpl;
/**
 * 
 * 
 * @author Luca Frosini (ISTI - CNR)
 */
@JsonDeserialize(as=DiscoveryFacetImpl.class)
@TypeMetadata(
	name = DiscoveryFacet.NAME, 
	description = "This facet defines a group of resources to be discovered and potentially selected for inclusion in a context. It enables dynamic resource discovery through queries and supports selection constraints.",
	version = Version.MINIMAL_VERSION_STRING
)
@Change(version = Version.MINIMAL_VERSION_STRING, description = Version.MINIMAL_VERSION_DESCRIPTION)
public interface DiscoveryFacet extends Facet {
	
	/** The name associated with this facet */
	public static final String NAME = "DiscoveryFacet"; // DiscoveryFacet.class.getSimpleName();
	
	/** The name of 'group' property */
	public static final String GROUP_PROPERTY = "group";
	/** The name of 'description' property */
	public static final String DESCRIPTION_PROPERTY = "description";
	/** The name of 'max' property */
	public static final String MAX_PROPERTY = "max";
	/** The name of 'min' property */
	public static final String MIN_PROPERTY = "min";
	/** The name of 'queries' property */
	public static final String QUERIES_PROPERTY = "queries";
	/** The name of 'queryTemplates' property */
	public static final String QUERY_TEMPLATES_PROPERTY = "queryTemplates";

	/**
	 * Returns a default instance of DiscoveryFacet
	 * @return a default instance of DiscoveryFacet
	 */
	public static DiscoveryFacet getDefaultInstance() {	
		return new DiscoveryFacetImpl();
	}

	/**
	 * The name identifier of the resource group to discover. This serves as a unique identifier for organizing and categorizing discovered resources.
	 * @return the group name
	 */
	@ISProperty(name = GROUP_PROPERTY, description = "The name identifier of the resource group to discover. This serves as a unique identifier for organizing and categorizing discovered resources.", mandatory = true, nullable = false)
	public String getGroup();

	/**
	 * Sets the name identifier of the resource group to discover. This serves as a unique identifier for organizing and categorizing discovered resources.
	 * @param group the group name
	 */
	public void setGroup(String group);

	/**
	 * Human-readable description displayed to users explaining what type of resources will be discovered in this group
	 * @return the description
	 */
	@ISProperty(name = DESCRIPTION_PROPERTY, description = "Human-readable description displayed to users explaining what type of resources will be discovered in this group")
	public String getDescription();

	/**
	 * Sets a human-readable description displayed to users explaining what type of resources will be discovered in this group
	 * @param description the description
	 */
	public void setDescription(String description);

	/**
	 * Minimum number of resources that must be selected from this group. Default is 0 (optional selection).
	 * @return the min
	 */
	@ISProperty(name = MIN_PROPERTY, description = "Minimum number of resources that must be selected from this group. Default is 0 (optional selection).", readonly = false, mandatory = true, nullable = false, defaultValue = "0")
	public int getMin();
	
	/**
	 * Sets the minimum number of resources that must be selected from this group. Default is 0 (optional selection).
	 * @param min the min
	 */
	public void setMin(int min);

	/**
	 * Maximum number of resources that can be selected from this group. Null means unlimited selection.
	 * @return the max
	 */
	@ISProperty(name = MAX_PROPERTY, description = "Maximum number of resources that can be selected from this group. Null means unlimited selection.", readonly = false, mandatory = true, nullable = true, defaultValue = "null")
	public Integer getMax();
	
	/**
	 * Sets the maximum number of resources that can be selected from this group. Null means unlimited selection.
	 * @param max the max
	 */
	public void setMax(Integer max);

	/**
	 * Array of JSON query objects used to discover resources. Each query is executed and results are combined (union of all query results).
	 * @return the queries
	 */
	@ISProperty(name = QUERIES_PROPERTY, description = "Array of JSON query objects used to discover resources. Each query is executed and results are combined (union of all query results).", readonly = false, mandatory = true, nullable = true, defaultValue = "null")
	public ArrayNode getQueries();

	/**
	 * Sets the array of JSON query objects used to discover resources. Each query is executed and results are combined (union of all query results).
	 * @param queries the queries
	 */
	public void setQueries(ArrayNode queries);

	/**
	 * Adds a JSON query object used to discover resources.
	 * @param query the query
	 */
	public void addQuery(ObjectNode query);

	/**
	 * Array of query template references with parameters. These are named, reusable query definitions that can accept variables for dynamic resource discovery.
	 * @return the query templates
	 */
	@ISProperty(name = QUERY_TEMPLATES_PROPERTY, description = "Array of query template references with parameters. These are named, reusable query definitions that can accept variables for dynamic resource discovery.", readonly = false, mandatory = true, nullable = true, defaultValue = "null")
	public List<QueryTemplateReference> getQueryTemplates();

	/**
	 * Sets the array of query template references with parameters. These are named, reusable query definitions that can accept variables for dynamic resource discovery.
	 * @param queryTemplates the query templates
	 */
	public void setQueryTemplates(List<QueryTemplateReference> queryTemplates);

	/**
	 * Adds a query template reference with parameters. These are named, reusable query definitions that can accept variables for dynamic resource discovery.
	 * @param queryTemplates the query template to add
	 */
	public void addQueryTemplates(QueryTemplateReference queryTemplates);

}