/**
 * 
 */
package org.gcube.resourcemanagement.model.reference.entities.facets;

import org.gcube.com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import org.gcube.com.fasterxml.jackson.databind.node.ObjectNode;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.types.reference.Change;
import org.gcube.informationsystem.types.reference.Changelog;
import org.gcube.informationsystem.types.reference.TypeMetadata;
import org.gcube.informationsystem.utils.Version;
import org.gcube.resourcemanagement.model.impl.entities.facets.LocationFacetImpl;

/**
 * LocationFacet captures information on a physical area characterising the resource it
 * is associated with. This should not be confused with {@link CoverageFacet}
 * The LocationFacet provides information of a location (eventually using 
 * latitude and longitude), instead {@link CoverageFacet} provide a way to to define the spatial 
 * or the temporal extent the resource represent.
 * 
 * It is mainly used to locate a data centre or to the geographic references of a 
 * legal body playing the role of an actor in the infrastructure.
 * 
 * https://wiki.gcube-system.org/gcube/GCube_Model#Location_Facet
 * 
 * @author Luca Frosini (ISTI - CNR)
 * @author Francesco Mangiacrapa (ISTI - CNR)
 */
@JsonDeserialize(as=LocationFacetImpl.class)
@TypeMetadata(
	name = LocationFacet.NAME, 
	description = "LocationFacet captures information about the geographic location characterizing the associated resource. It should not be confused with {@link CoverageFacet}. LocationFacet describes where a resource is located, optionally using GeoJSON spatial geometry. It is typically used to locate a data centre or to represent the geographic reference of a legal or organizational body acting as an actor within the infrastructure.",
	version = LocationFacet.VERSION
)
@Changelog({
	@Change(version = LocationFacet.VERSION, description = "General revision of the facet to improve clarity and usability; removed 'latitude' and 'longitude' in favor of 'spatialGeometry' for greater flexibility; removed 'country' to avoid collisions with 'location' when the intended location represents a country; 'location' is no longer limited to a city name and can represent any human-readable location description."),
	@Change(version = Version.MINIMAL_VERSION_STRING, description = Version.MINIMAL_VERSION_DESCRIPTION)
})
public interface LocationFacet extends Facet {

	/** The name associated with this facet */
	public static final String NAME = "LocationFacet"; // LocationFacet.class.getSimpleName();
	/** The version associated with this facet */
	public static final String VERSION = "1.1.0";
	
	/** The name of 'location' property */
	public static final String LOCATION_PROPERTY = "location";
	/** The name of 'spatialGeometry' property */
	public static final String SPATIAL_GEOMETRY_PROPERTY = "spatialGeometry";

	/**
	 * Returns a default instance of LocationFacet
	 * @return a default instance of LocationFacet
	 */
	public static LocationFacet getDefaultInstance() {
		return new LocationFacetImpl();
	}

	/**
	 * A human english name for the location, e.g., a city name, a country name, a site name, a region name, etc.
	 * @return the location
	 */
	@ISProperty(name = LocationFacet.LOCATION_PROPERTY, description = "A human english name for the location, e.g., a city name, a country name, a site name, a region name, etc.")
	public String getLocation();
	
	/**
	 * Sets A human english name for the location, e.g., a city name, a country name, a site name, a region name, etc.
	 * @param location the location
	 */
	public void setLocation(String location);

	/**
	 * Spatial Geometry in GeoJSON format
	 * @return the spatial geometry
	 */	
	@ISProperty(name = LocationFacet.SPATIAL_GEOMETRY_PROPERTY, description = "Spatial Geometry in GeoJSON format")
	public ObjectNode getSpatialGeometry();

	/**
	 * Sets the Spatial Geometry in GeoJSON format
	 * @param spatialGeometry the spatial geometry
	 */
	public void setSpatialGeometry(ObjectNode spatialGeometry);
	
}