package org.gcube.resourcemanagement.analyser;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.core.JsonProcessingException;
import org.gcube.com.fasterxml.jackson.databind.node.ArrayNode;
import org.gcube.com.fasterxml.jackson.databind.node.ObjectNode;
import org.gcube.informationsystem.base.reference.AccessType;
import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.model.impl.properties.PropagationConstraintImpl;
import org.gcube.informationsystem.model.reference.ERElement;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.properties.PropagationConstraint;
import org.gcube.informationsystem.model.reference.properties.PropagationConstraint.AddConstraint;
import org.gcube.informationsystem.model.reference.properties.PropagationConstraint.DeleteConstraint;
import org.gcube.informationsystem.model.reference.properties.PropagationConstraint.RemoveConstraint;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.serialization.ElementMapper;
import org.gcube.informationsystem.tree.Tree;
import org.gcube.informationsystem.types.impl.validator.ObjectNodeValidator;
import org.gcube.informationsystem.types.knowledge.TypesKnowledge;
import org.gcube.informationsystem.types.reference.Type;
import org.gcube.resourcemanagement.model.impl.entities.resources.ConfigurationImpl;
import org.gcube.resourcemanagement.model.impl.entities.resources.ConfigurationTemplateImpl;
import org.gcube.resourcemanagement.model.impl.relations.isrelatedto.IsCustomizedByImpl;
import org.gcube.resourcemanagement.model.impl.relations.isrelatedto.IsDerivationOfImpl;
import org.gcube.resourcemanagement.model.reference.entities.facets.TemplateFacet;
import org.gcube.resourcemanagement.model.reference.entities.resources.Configuration;
import org.gcube.resourcemanagement.model.reference.entities.resources.ConfigurationTemplate;
import org.gcube.resourcemanagement.model.reference.entities.resources.Service;
import org.gcube.resourcemanagement.model.reference.relations.isrelatedto.IsCustomizedBy;
import org.gcube.resourcemanagement.model.reference.relations.isrelatedto.IsDerivationOf;

import jakarta.ws.rs.BadRequestException;
import jakarta.ws.rs.InternalServerErrorException;
import jakarta.ws.rs.WebApplicationException;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class ConfigurationTemplateAnalyser extends GCubeAnalyser<ConfigurationTemplate> {

    public ConfigurationTemplateAnalyser() {
        super();
    }

    public ConfigurationTemplateAnalyser(String type, UUID instanceUUID) {
        super(type, instanceUUID);
    }

    @Override
    public Class<ConfigurationTemplate> getTypeClass() {
        return ConfigurationTemplate.class;
    }

    @Override
    public boolean polymorphic() {
        return true;
    }

    protected List<TemplateFacet> getTemplateFacets() throws WebApplicationException, ResourceRegistryException {
        return resource.getFacets(TemplateFacet.class);
    }

    protected List<IdentifiableElement> getTemplateFacetsAsIdentifiableElements() throws WebApplicationException, ResourceRegistryException {
        List<TemplateFacet> templateFacets = getTemplateFacets();
        List<IdentifiableElement> identifiableElements = new ArrayList<>(templateFacets.size());
        for (TemplateFacet facet : templateFacets) {
            
            TemplateFacet clone = null;
            try {
                String s = ElementMapper.marshal(facet);
                clone = ElementMapper.unmarshal(TemplateFacet.class, s);
            } catch (IOException e) {
                throw new InternalServerErrorException(e);
            }

            clone.setID(null);
            identifiableElements.add(clone);
        }
        return identifiableElements;
    }

    protected boolean validateInstanceAgainstTemplate(TemplateFacet tf, Facet newFacet) throws WebApplicationException, ResourceRegistryException {
        String expectedTargetType = tf.getTargetType();

        TypesKnowledge tk = TypesKnowledge.getInstance();
        Tree<Type> facetTree = tk.getModelKnowledge().getTree(AccessType.FACET);

        String newFacetType = newFacet.getTypeName();
        if(newFacetType.compareTo(expectedTargetType)!=0 && !facetTree.isChildOf(expectedTargetType, newFacetType)){
            return false;
        }

        ObjectNode node = ElementMapper.getObjectMapper().valueToTree(newFacet);
        ObjectNodeValidator validator = new ObjectNodeValidator(node, tf.getProperties());
        boolean valid = validator.validate();
        if(valid){
            logger.trace("The instance {} is valid against the template facet {}", newFacet, tf);
        } else {
            logger.trace("The instance {} is not valid against the template facet {}. The errors are: {}", newFacet, tf, validator.getValidatorReport().getErrors());
        }
        
        return valid;
    }

    @Override
    protected List<IdentifiableElement> validateInstancesToCreate(List<IdentifiableElement> newInstances) throws WebApplicationException, ResourceRegistryException {
        List<IdentifiableElement> elements = new ArrayList<>();

        UUID configurationUUID = UUID.randomUUID();
        Configuration configuration = new ConfigurationImpl();
        configuration.setID(configurationUUID);
        
        ConfigurationTemplate configurationTemplate = new ConfigurationTemplateImpl();
        configurationTemplate.setID(this.getResource().getID());
        
        // Creating IsDerivationOf relation
        PropagationConstraint propagationConstraint = new PropagationConstraintImpl();
        propagationConstraint.setAddConstraint(AddConstraint.unpropagate);
        propagationConstraint.setRemoveConstraint(RemoveConstraint.keep);
        propagationConstraint.setDeleteConstraint(DeleteConstraint.keep);
        IsDerivationOf<Configuration, ConfigurationTemplate> isDerivationOf = new IsDerivationOfImpl<>(configuration, configurationTemplate, propagationConstraint);
        configuration.attachResource(isDerivationOf);

        Set<IdentifiableElement> validInstances = new HashSet<>();
        int templateFacetCount = 0;

        // Cloning ConfigurationTemplate to Configuration
        ConfigurationTemplate clone = null;
        try {
            String resourceString = ElementMapper.marshal(this.getResource());
            clone = ElementMapper.unmarshal(ConfigurationTemplate.class, resourceString);
        } catch (IOException e) {
            throw new InternalServerErrorException(e);
        }

        for(ConsistsOf<? extends Resource, ? extends Facet> consistsOf : clone.getConsistsOf()){
            @SuppressWarnings("unchecked")
            ConsistsOf<Resource, Facet> co = (ConsistsOf<Resource, Facet>) consistsOf;
            co.setSource(configuration);
            co.setID(null);
            Facet f = co.getTarget();
            f.setID(null);

            /*
             * If the facet is a TemplateFacet, then we need to find a valid instance for it.
             * The algorithm is the following:
             * - we find all the instance matching the TemplateFacet;
             * - we add all the isntanced matching the TemplateFacet to the validInstances set;
             * - if there is no instance we rainse an exception;
             * If the number of valid instances is minor of the number of TemplateFacet we raise an exception (see below at the end of the for).
             */
            if(f instanceof TemplateFacet){
                templateFacetCount++;
                boolean found = false;
                TemplateFacet tf = (TemplateFacet) f;
                for(IdentifiableElement ie : newInstances){
                    Facet newFacet = (Facet) ie;
                    if(validateInstanceAgainstTemplate(tf, newFacet)){
                        found = true;
                        boolean added = validInstances.add(newFacet);
                        if(!added){
                            logger.debug("Valid instance found (i.e. {}) for the template facet {}. Please note that this instance match also other template facets.", newFacet, tf);
                        }else{
                            logger.info("Valid instance found (i.e. {}) for the template facet {} ", newFacet, tf);
                        }
                    }
                }

                if(!found){
                    throw new BadRequestException("No valid instances found for the template facet " + tf.toString());
                }

            }else{
                configuration.addFacet(co);
            }

        }

        /*
         * If the number of valid instances is minor of the number of TemplateFacet we raise an exception.
         */
        if(templateFacetCount<validInstances.size()){
            throw new BadRequestException("The number of valid instances found " + validInstances.size()+ " is minor of the number of template facets " + templateFacetCount);
        }

        for(IdentifiableElement ie : validInstances){
            configuration.addFacet((Facet) ie);
        }

        elements.add(configuration);

        if(ancestor instanceof Service){
            PropagationConstraint pc = new PropagationConstraintImpl();
            pc.setAddConstraint(AddConstraint.unpropagate);
            pc.setRemoveConstraint(RemoveConstraint.cascade);
            pc.setDeleteConstraint(DeleteConstraint.cascadeWhenOrphan);

            ObjectNode node = ElementMapper.getObjectMapper().valueToTree(ancestor);
            ((ArrayNode) node.get(Resource.CONSISTS_OF_PROPERTY)).removeAll();
            ((ArrayNode) node.get(Resource.IS_RELATED_TO_PROPERTY)).removeAll();
            node.remove(ERElement.METADATA_PROPERTY);
            Service service;
            try {
                service = ElementMapper.getObjectMapper().treeToValue(node, Service.class);
            } catch (JsonProcessingException e) {
                throw new InternalServerErrorException(e);
            }
            
            Configuration conf = new ConfigurationImpl();
            conf.setID(configurationUUID);

            IsCustomizedBy<Service, Configuration> isCustomizedBy = new IsCustomizedByImpl<>(service, conf, pc);
             elements.add(isCustomizedBy);
        }

        return elements;
    }

    @Override
    protected  List<IdentifiableElement> getNewInstances() throws WebApplicationException, ResourceRegistryException {
        List<IdentifiableElement> elements = getTemplateFacetsAsIdentifiableElements();
        return elements;
    }

}
