package org.gcube.vremanagement.executor.plugin;

import java.util.Date;
import java.util.Map;

import org.gcube.vremanagement.executor.persistence.Persistence;
import org.gcube.vremanagement.executor.persistence.PersistenceConnector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This interface represent the contract for a plugin runnable by the executor.
 * @author Luca Frosini (ISTI - CNR) http://www.lucafrosini.com/
 *
 */
public abstract class Plugin<T extends PluginDeclaration> {
	
	/**
	 * Logger
	 */
	private static Logger logger = LoggerFactory.getLogger(Plugin.class);
	
	/**
	 * Instance of the class which log the plugin state evolution
	 */
	private Persistence<?  extends PersistenceConnector> persistence;
	
	private T pluginDeclaration;
	
	public Plugin(T pluginDeclaration, Persistence<?  extends PersistenceConnector> persistence){
		this.pluginDeclaration = pluginDeclaration;
		this.persistence = persistence;
	}
	
	/**
	 * @return the pluginDeclaration
	 */
	public T getPluginDeclaration() {
		return pluginDeclaration;
	}
	
	/**
	 * It is up to the plugin update the State of the Running Plugin using 
	 * this facilities function.
	 * @param pluginState
	 * @throws Exception 
	 */
	public void setState(PluginState pluginState) {
		long timestamp =  new Date().getTime();
		try {
			persistence.addEvolution(timestamp, pluginState);
		} catch(Exception e) {
			logger.error(String.format("Unable to persist State : %d,%s", timestamp, pluginState.name()));
		}
	}
	
	/**
	 * Launch the plugin with the provided input.
	 * @param inputs
	 * @throws Exception if the launch fails
	 */
	public abstract void launch(Map<String,Object> inputs) throws Exception;
	
	/**
	 * This function is used to correctly stop the plugin 
	 * @throws Exception if the launch fails
	 */
	protected abstract void onStop() throws Exception;
	
	/**
	 * Stop the Plugin setting state to {@link PluginState#SUSPENDED}
	 * @throws Exception
	 */
	public void stop() throws Exception {
		setState(PluginState.SUSPENDED);
		onStop();
	}
	
	/**
	 * Used to retrieve the state of the execution
	 * @return the {@link PluginState}
	 * @throws Exception
	 */
	public PluginState getState() throws Exception {
		return persistence.getState();
	}
	
}
