package org.gcube.common.security;

import java.util.concurrent.Callable;

import org.gcube.common.security.providers.SecretManagerProvider;
import org.gcube.common.security.secrets.Secret;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A utility class for managing and executing tasks with specific authorization contexts.
 * This class provides static methods to bind {@link Callable} and {@link Runnable}
 * tasks to a specific {@link Secret} context. It ensures that the correct secret
 * is set before the task runs and is properly reset afterward, preventing
 * secrets from leaking into unintended scopes.
 */
public class AuthorizedTasks {

	private static Logger logger = LoggerFactory.getLogger(AuthorizedTasks.class);

	/**
	 * Binds a {@link Callable} task to the current authorization context.
	 *
	 * This method captures the current secret from {@link SecretManagerProvider}
	 * and returns a new {@code Callable} that will automatically set that secret
	 * before executing the original task and reset it afterward. This is
	 * useful for delegating tasks to a different thread or for asynchronous
	 * execution while maintaining the correct authorization context.
	 *
	 * @param task The original {@link Callable} task to be bound.
	 * @param <V> The return type of the task.
	 * @return An equivalent {@link Callable} task bound to the current
	 * authorization context.
	 */
	static public <V> Callable<V> bind(final Callable<V> task) {
		final Secret secret = SecretManagerProvider.get();

		return new Callable<V>() {
			@Override
			public V call() throws Exception {
				SecretManagerProvider.reset();
				SecretManagerProvider.set(secret);
				try {
					logger.info("Setting authorized task context for: {}", secret.getContext());
					return task.call();
				} finally {
					SecretManagerProvider.reset();
				}
			}
		};
	}

	/**
	 * Binds a {@link Runnable} task to the current authorization context.
	 *
	 * Similar to the {@code bind(Callable)} method, this captures the current
	 * secret and returns a new {@code Runnable} that will set the secret
	 * before execution and reset it upon completion.
	 *
	 * @param task The original {@link Runnable} task to be bound.
	 * @return An equivalent {@link Runnable} task bound to the current
	 * authorization context.
	 */
	static public Runnable bind(final Runnable task) {
		final Secret secret = SecretManagerProvider.get();

		return new Runnable() {
			@Override
			public void run() {
				SecretManagerProvider.reset();
				SecretManagerProvider.set(secret);
				try {
					logger.info("Setting authorized task context for: {}", secret.getContext());
					task.run();
				} finally {
					SecretManagerProvider.reset();
				}
			}
		};
	}
	
	/**
	 * Executes a {@link Runnable} task immediately within a specific secret context.
	 *
	 * This method temporarily sets a specified secret, executes the task,
	 * and then restores the previous secret, guaranteeing that the original
	 * secret is always put back in place, even if the task throws an exception.
	 *
	 * @param task The {@link Runnable} task to be executed.
	 * @param secret The {@link Secret} that must be used during the execution of the task.
	 */
	static public void executeSafely(final Runnable task, final Secret secret) {
		Secret previousSecret = SecretManagerProvider.get();
		try {
			SecretManagerProvider.reset();
			SecretManagerProvider.set(secret);	
			task.run();
		} finally {
			SecretManagerProvider.reset();
			SecretManagerProvider.set(previousSecret);	
		}
	}
	
	
	/**
	 * Executes a {@link Callable} task immediately within a specific secret context.
	 *
	 * Similar to the {@code executeSafely(Runnable)} method, this sets a specified secret,
	 * executes the {@link Callable} task, and restores the previous secret. It also
	 * handles propagating any exceptions thrown by the task.
	 *
	 * @param task The {@link Callable} task to be executed.
	 * @param secret The {@link Secret} that must be used during the execution of the task.
	 * @param <T> The return type of the task.
	 * @return The result of the {@link Callable} task.
	 * @throws Throwable if the task execution throws an exception.
	 */
	static public <T> T executeSafely(final Callable<T> task, final Secret secret) throws Throwable {
		Secret previousSecret = SecretManagerProvider.get();
		try {
			SecretManagerProvider.reset();
			SecretManagerProvider.set(secret);	
			return task.call();
		} finally {
			SecretManagerProvider.reset();
			SecretManagerProvider.set(previousSecret);	
		}
	}
}