package org.gcube.keycloak.protocol.oidc.tip;

import java.io.FileInputStream;
import java.io.InputStream;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.gcube.keycloak.protocol.oidc.Utils;
import org.gcube.keycloak.protocol.oidc.tip.TIPConfiguration.Config.LoggingConfig;
import org.gcube.keycloak.protocol.oidc.tip.TIPConfiguration.Config.ServerConfig;
import org.gcube.keycloak.protocol.oidc.tip.TIPConfiguration.Config.TipConfig;
import org.gcube.keycloak.protocol.oidc.tip.TIPConfiguration.Config.TipConfig.RemoteIssuer;
import org.jboss.logging.Logger;
import org.yaml.snakeyaml.LoaderOptions;
import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.constructor.Constructor;

/**
 * @author <a href="mailto:mauro.mugnaini@nubisware.com">Mauro Mugnaini</a>
 */
public class TIPConfiguration {

    private static final Logger logger = Logger.getLogger(TIPConfiguration.class);

    private final Config config;

    public TIPConfiguration(Config config) {
        this.config = config;
        ExecutorService executor = Executors.newSingleThreadExecutor();
        executor.submit(() -> TIPConfiguration.checkIssuerIntrospectionEndpoints(this.config));
        executor.shutdown();
    }

    public static TIPConfiguration loadFromYAML(String yamlConfigFile) throws TIPConfigurationException {
        try {
            return loadFromYAML(new FileInputStream(yamlConfigFile));
        } catch (Exception e) {
            throw new TIPConfigurationException("Cannot load YAML confguration from file " + yamlConfigFile, e);
        }
    }

    public static TIPConfiguration loadFromYAML(InputStream yamlConfigUnputStream) {
        Yaml yaml = new Yaml(new Constructor(Config.class, new LoaderOptions()));
        return new TIPConfiguration(yaml.loadAs(yamlConfigUnputStream, Config.class));
    }

    protected static void checkIssuerIntrospectionEndpoints(Config config) {
        TipConfig tip = config.getTip();
        if (tip.getFallback_issuer_unknown_token_issuer() != null &&
                tip.getFallback_issuer_unknown_token_issuer().getIntrospection_endpoint() == null) {

            String issuerUrl = tip.getFallback_issuer_unknown_token_issuer().getIssuer_url();
            logger.debugf("Discovering introspection endpoint for unknown_token_issuer fallback at: %s", issuerUrl);
            String discoveredEndpoint = Utils.getIntrospectionEndpoint(issuerUrl);
            logger.debugf("Discovered unknown_token_issuer endpoint is: %s", discoveredEndpoint);
            tip.getFallback_issuer_unknown_token_issuer().setIntrospection_endpoint(discoveredEndpoint);
        }

        if (tip.getFallback_issuer_unsupported_token_issuer() != null &&
                tip.getFallback_issuer_unsupported_token_issuer().getIntrospection_endpoint() == null) {

            String issuerUrl = tip.getFallback_issuer_unsupported_token_issuer().getIssuer_url();
            logger.debugf("Discovering introspection endpoint for unsupported_token_issuer fallback at: %s", issuerUrl);
            String discoveredEndpoint = Utils.getIntrospectionEndpoint(issuerUrl);
            logger.debugf("Discovered unsupported_token_issuer endpoint is: %s", discoveredEndpoint);
            tip.getFallback_issuer_unsupported_token_issuer().setIntrospection_endpoint(discoveredEndpoint);
        }
        for (RemoteIssuer remoteIssuer : tip.getRemote_issuers()) {
            if (remoteIssuer.getIntrospection_endpoint() == null) {
                String remoteIssuerIssuerUrl = remoteIssuer.getIssuer_url();
                logger.debugf("Discovering introspection endpoint for remote issuer: %s", remoteIssuerIssuerUrl);
                String discoveredEndpoint = Utils.getIntrospectionEndpoint(remoteIssuerIssuerUrl);
                logger.debugf("Discovered remote issuer endpoint is: %s", discoveredEndpoint);
                remoteIssuer.setIntrospection_endpoint(discoveredEndpoint);
            }
        }
    }

    public ServerConfig getServer() {
        return config.getServer();
    }

    public LoggingConfig getLogging() {
        return config.getLogging();
    }

    public TipConfig getTip() {
        return config.getTip();
    }

    public static class Config {

        private ServerConfig server;
        private LoggingConfig logging;
        private TipConfig tip;

        public ServerConfig getServer() {
            return server;
        }

        public void setServer(ServerConfig server) {
            this.server = server;
        }

        public LoggingConfig getLogging() {
            return logging;
        }

        public void setLogging(LoggingConfig logging) {
            this.logging = logging;
        }

        public TipConfig getTip() {
            return tip;
        }

        public void setTip(TipConfig tip) {
            this.tip = tip;
        }

        // -------- Nested Classes --------

        public static class ServerConfig {
            private int port;
            private TlsConfig tls;

            public int getPort() {
                return port;
            }

            public void setPort(int port) {
                this.port = port;
            }

            public TlsConfig getTls() {
                return tls;
            }

            public void setTls(TlsConfig tls) {
                this.tls = tls;
            }

            public static class TlsConfig {

                private Boolean enabled;
                private Boolean redirect_http;
                private Boolean use_custom_port;
                private String cert;
                private String key;

                public Boolean getEnabled() {
                    return enabled;
                }

                public void setEnabled(Boolean enabled) {
                    this.enabled = enabled;
                }

                public Boolean getRedirect_http() {
                    return redirect_http;
                }

                public void setRedirect_http(Boolean redirect_http) {
                    this.redirect_http = redirect_http;
                }

                public Boolean getUse_custom_port() {
                    return use_custom_port;
                }

                public void setUse_custom_port(Boolean use_custom_port) {
                    this.use_custom_port = use_custom_port;
                }

                public String getCert() {
                    return cert;
                }

                public void setCert(String cert) {
                    this.cert = cert;
                }

                public String getKey() {
                    return key;
                }

                public void setKey(String key) {
                    this.key = key;
                }
            }
        }

        public static class LoggingConfig {

            private AccessLog access;
            private InternalLog internal;

            public AccessLog getAccess() {
                return access;
            }

            public void setAccess(AccessLog access) {
                this.access = access;
            }

            public InternalLog getInternal() {
                return internal;
            }

            public void setInternal(InternalLog internal) {
                this.internal = internal;
            }

            public static class AccessLog {

                private String dir;
                private boolean stderr;

                public String getDir() {
                    return dir;
                }

                public void setDir(String dir) {
                    this.dir = dir;
                }

                public boolean isStderr() {
                    return stderr;
                }

                public void setStderr(boolean stderr) {
                    this.stderr = stderr;
                }
            }

            public static class InternalLog {

                private String dir;
                private boolean stderr;
                private String level;
                private SmartLog smart;

                public String getDir() {
                    return dir;
                }

                public void setDir(String dir) {
                    this.dir = dir;
                }

                public boolean isStderr() {
                    return stderr;
                }

                public void setStderr(boolean stderr) {
                    this.stderr = stderr;
                }

                public String getLevel() {
                    return level;
                }

                public void setLevel(String level) {
                    this.level = level;
                }

                public SmartLog getSmart() {
                    return smart;
                }

                public void setSmart(SmartLog smart) {
                    this.smart = smart;
                }

                public static class SmartLog {
                    private boolean enabled;
                    private String dir;

                    public boolean isEnabled() {
                        return enabled;
                    }

                    public void setEnabled(boolean enabled) {
                        this.enabled = enabled;
                    }

                    public String getDir() {
                        return dir;
                    }

                    public void setDir(String dir) {
                        this.dir = dir;
                    }
                }
            }
        }

        public static class TipConfig {

            private IssuerConfig linked_issuer;
            private RemoteIssuer fallback_issuer_unknown_token_issuer;
            private RemoteIssuer fallback_issuer_unsupported_token_issuer;
            private List<RemoteIssuer> remote_issuers;

            public IssuerConfig getLinked_issuer() {
                return linked_issuer;
            }

            public void setLinked_issuer(IssuerConfig linked_issuer) {
                this.linked_issuer = linked_issuer;
            }

            public RemoteIssuer getFallback_issuer_unknown_token_issuer() {
                return fallback_issuer_unknown_token_issuer;
            }

            public void setFallback_issuer_unknown_token_issuer(RemoteIssuer fallback_issuer_unknown_token_issuer) {
                this.fallback_issuer_unknown_token_issuer = fallback_issuer_unknown_token_issuer;
            }

            public RemoteIssuer getFallback_issuer_unsupported_token_issuer() {
                return fallback_issuer_unsupported_token_issuer;
            }

            public void setFallback_issuer_unsupported_token_issuer(
                    RemoteIssuer fallback_issuer_unsupported_token_issuer) {
                this.fallback_issuer_unsupported_token_issuer = fallback_issuer_unsupported_token_issuer;
            }

            public List<RemoteIssuer> getRemote_issuers() {
                return remote_issuers;
            }

            public void setRemote_issuers(List<RemoteIssuer> remote_issuers) {
                this.remote_issuers = remote_issuers;
            }

            public static class IssuerConfig {

                private String issuer_url;
                private String native_introspection_endpoint;

                public String getIssuer_url() {
                    return issuer_url;
                }

                public void setIssuer_url(String issuer_url) {
                    this.issuer_url = issuer_url;
                }

                public String getNative_introspection_endpoint() {
                    return native_introspection_endpoint;
                }

                public void setNative_introspection_endpoint(String native_introspection_endpoint) {
                    this.native_introspection_endpoint = native_introspection_endpoint;
                }
            }

            public static class RemoteIssuer {

                private String issuer_url;
                private String introspection_endpoint;
                private String client_id;
                private String client_secret;
                private List<String> drop_claims;
                private Map<String, String> claim_renaming;
                private ClaimMapping claim_mapping;

                public String getIssuer_url() {
                    return issuer_url;
                }

                public void setIssuer_url(String issuer_url) {
                    this.issuer_url = issuer_url;
                }

                public String getIntrospection_endpoint() {
                    return introspection_endpoint;
                }

                public void setIntrospection_endpoint(String introspection_endpoint) {
                    this.introspection_endpoint = introspection_endpoint;
                }

                public String getClient_id() {
                    return client_id;
                }

                public void setClient_id(String client_id) {
                    this.client_id = client_id;
                }

                public String getClient_secret() {
                    return client_secret;
                }

                public void setClient_secret(String client_secret) {
                    this.client_secret = client_secret;
                }

                public List<String> getDrop_claims() {
                    return drop_claims;
                }

                public void setDrop_claims(List<String> drop_claims) {
                    this.drop_claims = drop_claims;
                }

                public Map<String, String> getClaim_renaming() {
                    return claim_renaming;
                }

                public void setClaim_renaming(Map<String, String> claim_renaming) {
                    this.claim_renaming = claim_renaming;
                }

                public ClaimMapping getClaim_mapping() {
                    return claim_mapping;
                }

                public void setClaim_mapping(ClaimMapping claim_mapping) {
                    this.claim_mapping = claim_mapping;
                }

                public static class ClaimMapping {

                    private Map<String, Map<String, String>> strings;
                    private Map<String, Map<String, List<String>>> string_arrays;

                    public Map<String, Map<String, String>> getStrings() {
                        return strings;
                    }

                    public void setStrings(Map<String, Map<String, String>> strings) {
                        this.strings = strings;
                    }

                    public Map<String, Map<String, List<String>>> getString_arrays() {
                        return string_arrays;
                    }

                    public void setString_arrays(Map<String, Map<String, List<String>>> string_arrays) {
                        this.string_arrays = string_arrays;
                    }
                }
            }
        }
    }

}
