/**
 * 
 */
package org.gcube.resourcemanagement.model.reference.entities.facets;

import org.gcube.com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.types.reference.Change;
import org.gcube.informationsystem.types.reference.Changelog;
import org.gcube.informationsystem.types.reference.TypeMetadata;
import org.gcube.informationsystem.utils.Version;
import org.gcube.resourcemanagement.model.impl.entities.facets.ContactFacetImpl;

/**
 * ContactFacet captures information on a point of contact for the resource, 
 * i.e., a person or a department serving as the coordinator or focal point 
 * of information concerning the resource.
 * 
 * https://wiki.gcube-system.org/gcube/GCube_Model#Contact_Facet
 * 
 *  @author Luca Frosini (ISTI - CNR)
 */
@JsonDeserialize(as=ContactFacetImpl.class)
@TypeMetadata(
	name = ContactFacet.NAME, 
	description = "ContactFacet captures information on a point of contact for the resource, "
			+ "i.e., a person or a department serving as the coordinator or focal point  "
			+ "of information concerning the resource.",
	version = ContactFacet.VERSION
)
@Changelog({
	@Change(version = ContactFacet.VERSION, description = "Added id property and removed the mandatory constraint on the email property to provide more flexibility and comply with privacy regulations (e.g., GDPR)."),
	@Change(version = Version.MINIMAL_VERSION_STRING, description = Version.MINIMAL_VERSION_DESCRIPTION)
})
public interface ContactFacet extends Facet {

	/** The name associated with this facet */
	public static final String NAME = "ContactFacet"; // ContactFacet.class.getSimpleName();
	/** The version associated with this facet */
	public static final String VERSION = "1.1.0";

	/** The name of 'eMail' property */
	public static final String EMAIL_PROPERTY = "eMail";
	/** The pattern for email validation */
	public static final String EMAIL_PATTERN = "^[a-z0-9._%+-]{1,128}@[a-z0-9.-]{1,128}$";

	/**
	 * Returns a default instance of ContactFacet
	 * @return a default instance of ContactFacet
	 */
	public static ContactFacet getDefaultInstance() {
		return new ContactFacetImpl();
	}

	/**
	 * An identifier for the point of contact.
	 * @return the id
	 */
	@ISProperty(description = "An identifier for the point of contact.")
	public String getId();

	/**
	 * Sets the id of the point of contact.
	 * @param id the id
	 */
	public void setId(String id);

	/**
	 * A name describing the profession or marital status of the point of contact. e.g., Dr, Mrs, Mr.
	 * @return the title
	 */
	@ISProperty(description = "A name describing the profession or marital status of the point of contact. e.g., Dr, Mrs, Mr.")
	public String getTitle();

	/**
	 * Sets the title of the point of contact.
	 * @param title the title
	 */
	public void setTitle(String title);

	/**
	 * The first name of the point of contact.
	 * @return the name
	 */
	@ISProperty(description = "First Name", mandatory=true, nullable=false)
	public String getName();

	/**
	 * Sets the first name of the point of contact.
	 * @param name the name
	 */
	public void setName(String name);

	/**
	 * The middle name of the point of contact.
	 * @return the middle name
	 */
	@ISProperty(description = "Middle Name")
	public String getMiddleName();

	/**
	 * Sets the middle name of the point of contact.
	 * @param middleName the middle name
	 */
	public void setMiddleName(String middleName);

	/**
	 * The surname of the point of contact.
	 * @return the surname
	 */
	@ISProperty(description = "Surname", mandatory=true, nullable=false)
	public String getSurname();

	/**
	 * Sets the surname of the point of contact.
	 * @param surname the surname
	 */
	public void setSurname(String surname);

	/**
	 * The email address of the point of contact.
	 * @return the email
	 */
	@ISProperty(description = "Email address", name=EMAIL_PROPERTY, nullable=false, regexpr=EMAIL_PATTERN)
	public String getEMail();

	/**
	 * Sets the email address of the point of contact.
	 * @param eMail the email
	 */
	public void setEMail(String eMail);

}
