package org.gcube.common.security.factories;

import java.util.Set;

import org.gcube.common.security.credentials.Credentials;
import org.gcube.common.security.secrets.Secret;


/**
 * Provides a contract for retrieving authorization secrets and credentials.
 * This interface is designed to support different authorization contexts.
 */
public interface AuthorizationProvider {

	/**
	 * Retrieves the set of all available authorization contexts.
	 * Each context typically corresponds to a specific service or
	 * environment for which a secret is available.
	 *
	 * @return a {@code Set<String>} containing all known contexts.
	 * Returns an empty set if no contexts are available.
	 */
	Set<String> getContexts();
	
	/**
	 * Retrieves the secret for a given authorization context.
	 * The secret is provided as a {@link Secret} object, which
	 * encapsulates the sensitive data in a secure way.
	 *
	 * @param context the specific authorization context for which
	 * the secret is required.
	 * @return a {@link Secret} object for the specified context.
	 * Returns {@code null} if the context does not exist
	 * or no secret is associated with it.
	 */
	Secret getSecretForContext(String context);
	
	/**
	 * Retrieves generic credentials for authorization.
	 * This method is **deprecated** and should be avoided in new code.
	 * The preferred approach is to use {@link #getSecretForContext(String)}
	 * to retrieve secrets on a per-context basis for better security
	 * and specificity.
	 *
	 * @return a {@link Credentials} object.
	 * @deprecated Use {@link #getSecretForContext(String)} instead.
	 * This method will be removed in a future version.
	 */
	@Deprecated
	Credentials getCredentials();

}

