/**
 * (c) 2014 FAO / UN (project: fi-security-common)
 */
package org.fao.fi.security.common.utilities.pgp;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.NoSuchProviderException;
import java.security.Provider;
import java.util.Iterator;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.PGPCompressedData;
import org.bouncycastle.openpgp.PGPEncryptedDataList;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPLiteralData;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPOnePassSignatureList;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKeyEncryptedData;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRingCollection;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPDigestCalculatorProviderBuilder;
import org.bouncycastle.openpgp.operator.jcajce.JcePBESecretKeyDecryptorBuilder;
import org.bouncycastle.openpgp.operator.jcajce.JcePublicKeyDataDecryptorFactoryBuilder;
import org.fao.fi.security.common.encryption.pgp.exceptions.KeyringAccessException;
import org.fao.fi.security.common.encryption.pgp.exceptions.KeyringException;
import org.fao.fi.security.common.utilities.FileUtils;

/**
 * Place your class / interface description here.
 *
 * History:
 *
 * ------------- --------------- -----------------------
 * Date			 Author			 Comment
 * ------------- --------------- -----------------------
 * 30 Apr 2014   Fiorellato     Creation.
 *
 * @version 1.0
 * @since 30 Apr 2014
 */
public class PGPDecryptor extends AbstractPGPProcessor {
	/**
	 * Class constructor
	 */
	public PGPDecryptor() {
	}
	
	public byte[] decryptBytes(byte[] encodedData, File privateKeyringFile, String passPhrase) throws IOException, KeyringException, PGPException, NoSuchProviderException {
		return this.decryptBytes(encodedData, new FileInputStream(privateKeyringFile), passPhrase);
	}
	
	public byte[] decryptBytes(byte[] encodedData, InputStream privateKeyringStream, String passPhrase) throws IOException, KeyringException, PGPException, NoSuchProviderException {
		ByteArrayInputStream encodedDataStream = null;
		ByteArrayOutputStream decodedDataStream = null;
		
		try {
			this.doDecryptStream(encodedDataStream = new ByteArrayInputStream(encodedData), 
								 decodedDataStream = new ByteArrayOutputStream(), 
								 privateKeyringStream, 
								 passPhrase.toCharArray());
			
			decodedDataStream.flush();
			decodedDataStream.close();
			
			return decodedDataStream.toByteArray();
		} finally {
			encodedDataStream.close();
		}
	}
	
	public void decryptStream(InputStream encryptedStream, OutputStream decryptedStream, File privateKeyFile, char[] passphrase) throws IOException, KeyringException, PGPException, NoSuchProviderException {
		FileInputStream privateKeyStream = new FileInputStream(privateKeyFile);
		
		try {
			this.decryptStream(encryptedStream, decryptedStream, privateKeyStream, passphrase);
		} finally {
			privateKeyStream.close();
		}
	}
	
	public void decryptStream(InputStream encryptedStream, OutputStream decryptedStream, InputStream privateKeyStream, char[] passphrase) throws IOException, KeyringException, PGPException, NoSuchProviderException {
		this.doDecryptStream(encryptedStream, decryptedStream, privateKeyStream, passphrase);
	}

	private long doDecryptStream(InputStream encryptedStream, OutputStream decryptedStream, InputStream privateKeyStream, char[] passphrase) throws IOException, KeyringException, PGPException, NoSuchProviderException {
		try {
			PGPObjectFactory plainObjectFactory = new PGPObjectFactory(this.getPlainStream(encryptedStream, privateKeyStream, passphrase));
			
			Object message = null;
			
			try {
				message = plainObjectFactory.nextObject();
			} catch(IOException IOe) {
				if(IOe.getMessage().contains("unknown object"))
					throw new PGPException("Unable to decrypt stream: " + IOe.getClass().getSimpleName() + " [ " + IOe.getMessage() + " ]");
			}
			
			if (message instanceof PGPCompressedData) {
				PGPCompressedData compressedData = (PGPCompressedData) message;
				
				PGPObjectFactory pgpObjectFactory = new PGPObjectFactory(compressedData.getDataStream());
				
				message = pgpObjectFactory.nextObject();
			}
			
			if(message instanceof PGPLiteralData) {
				PGPLiteralData literalData = (PGPLiteralData)message;
				
				return FileUtils.pipeStreams(literalData.getInputStream(), decryptedStream);
			} else if (message instanceof PGPOnePassSignatureList) {
				throw new PGPException("Encrypted message contains a signed message - not literal data");
			} else {
				throw new PGPException("Message is not a simple encrypted file - type unknown");
			}
		} catch (PGPException e) {
			this._log.error("{}", ( e.getUnderlyingException() != null ? e.getUnderlyingException() : e ).getMessage(), e);
			
			throw e;
		}
	}

	private InputStream getPlainStream(InputStream encryptedStream, InputStream privateKeyStream, char[] passphrase) throws IOException, KeyringException, PGPException, NoSuchProviderException {
		PGPObjectFactory objectFactory = new PGPObjectFactory(encryptedStream);
		
		PGPEncryptedDataList encryptedDataList;
		
		Object object = null;
		
		try {
			object = objectFactory.nextObject();
		} catch(IOException IOe) {
			if(IOe.getMessage().contains("unknown object"))
				throw new PGPException("Unable to decrypt stream: " + IOe.getClass().getSimpleName() + " [ " + IOe.getMessage() + " ]");
			
			throw IOe;
		}

		if(object == null)
			throw new PGPException("Unable to decrypt stream: NULL object returned by object factory");
		
		if (object instanceof PGPEncryptedDataList) {
			encryptedDataList = (PGPEncryptedDataList)object;
		} else {
			encryptedDataList = (PGPEncryptedDataList)objectFactory.nextObject();
		}

		@SuppressWarnings("unchecked")
		Iterator<PGPPublicKeyEncryptedData> publicKeyEncryptedDataIterator = (Iterator<PGPPublicKeyEncryptedData>)encryptedDataList.getEncryptedDataObjects();
		
		PGPPrivateKey privateKey = null;
		PGPPublicKeyEncryptedData publicKeyEncryptedData = null;
		
		while(privateKey == null && publicKeyEncryptedDataIterator.hasNext()) {
			publicKeyEncryptedData = (PGPPublicKeyEncryptedData)publicKeyEncryptedDataIterator.next();
			
			this._log.debug(" * Public key encrypted data key id: {}", publicKeyEncryptedData.getKeyID());
			
			privateKey = this.extractPrivateKey(privateKeyStream, publicKeyEncryptedData.getKeyID(), passphrase);
		}
		
		if (privateKey == null) {
			throw new KeyringAccessException("Unable to extract private key for message");
		}

		if (publicKeyEncryptedData.isIntegrityProtected()) {
			if (!publicKeyEncryptedData.verify()) {
				this._log.error("[KO] Message failed integrity check");
			} else {
				this._log.debug("[OK]  message integrity check passed");
			}
		} else {
			this._log.debug("[!!] No message integrity check");
		}
		
		Provider provider = new BouncyCastleProvider();
		
		return
			publicKeyEncryptedData.getDataStream(
				new JcePublicKeyDataDecryptorFactoryBuilder().
					setProvider(provider).
					setContentProvider(provider).
					build(privateKey)
			);
	}
	
	//
	// Private class method findSecretKey
	//
	private PGPPrivateKey extractPrivateKey(InputStream privateKeyStream, long keyID, char[] passphrase) throws IOException, PGPException, NoSuchProviderException {
		PGPSecretKeyRingCollection secretKeyRingCollection = new PGPSecretKeyRingCollection(PGPUtil.getDecoderStream(privateKeyStream));
		PGPSecretKey secretKey = secretKeyRingCollection.getSecretKey(keyID);
		
		if (secretKey == null) {
			return null;
		}
		
		Provider provider = new BouncyCastleProvider();
		
		return secretKey.
			extractPrivateKey(
				new JcePBESecretKeyDecryptorBuilder(
					new JcaPGPDigestCalculatorProviderBuilder().
						setProvider(provider).build()
					).setProvider(provider).build(passphrase));
	}
}