/**
 * (c) 2014 FAO / UN (project: fi-security-server)
 */
package org.fao.fi.security.server.javax.filters.bandwidth;

import java.io.IOException;

import javax.annotation.PostConstruct;
import javax.annotation.Priority;
import javax.inject.Inject;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Priorities;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;
import net.sf.ehcache.config.CacheConfiguration;
import net.sf.ehcache.store.MemoryStoreEvictionPolicy;

import org.fao.fi.security.server.javax.filters.AbstractRequestValidatorFilter;
import org.fao.fi.security.server.javax.filters.bandwidth.configuration.BandwidthLimitedConfiguration;
import org.fao.fi.security.server.javax.filters.bandwidth.configuration.BandwidthLimitedSimpleConfiguration;
import org.fao.fi.security.server.javax.filters.bandwidth.support.BandwidthLimitedConstants;
import org.fao.fi.security.server.javax.filters.bandwidth.support.BandwidthLimitedResource;
import org.fao.fi.security.server.javax.filters.bandwidth.support.BandwidthLimiter;

/**
 * Place your class / interface description here.
 *
 * History:
 *
 * ------------- --------------- -----------------------
 * Date			 Author			 Comment
 * ------------- --------------- -----------------------
 * 30 Apr 2014   Fiorellato     Creation.
 *
 * @version 1.0
 * @since 30 Apr 2014
 */
@BandwidthLimitedResource @Priority(value=Priorities.AUTHENTICATION + 1)
public class BandwidthLimitedResourceRequestValidatorFilter extends AbstractRequestValidatorFilter {
	static final protected String REQUESTS_BY_IP_CACHE_ID = "requests.cache";
	
	static final protected int MAX_IPS_IN_CACHE = 1000;
	static final protected int MAX_REQUESTS = 50;
	static final protected int TIMEFRAME    = 10;
	
	@Context private HttpServletRequest _servletRequest;

	private BandwidthLimitedConfiguration _configuration;
	
	protected CacheManager _cacheManager;
	protected Cache _requestsByIPCache;
	
	/**
	 * Class constructor
	 */
	@Inject public BandwidthLimitedResourceRequestValidatorFilter(@BandwidthLimiter BandwidthLimitedConfiguration configuration) {
		final String $THIS = this.getClass().getSimpleName() + "#" + this.hashCode();
		
		this._configuration = configuration;
		
		if(this._configuration == null) {
			this._log.warn("No bandwidth limiter configuration provided for {}: using defaults...", this);
			
			this._configuration = new BandwidthLimitedSimpleConfiguration(MAX_REQUESTS, TIMEFRAME);
		}
		
		this._cacheManager = CacheManager.getInstance();
		
		if(this._cacheManager == null)
			this._cacheManager = CacheManager.create();
		
		String cacheId = REQUESTS_BY_IP_CACHE_ID;
		
		this._log.info("Initializing {} with {} and cache ID {}", $THIS, this._cacheManager, cacheId);
		
		if(!this._cacheManager.cacheExists(cacheId)) {
			CacheConfiguration cacheConfiguration = new CacheConfiguration(cacheId, MAX_IPS_IN_CACHE);
			cacheConfiguration.eternal(false).
							   timeToIdleSeconds(this._configuration.getTimeframe()).
							   timeToLiveSeconds(this._configuration.getTimeframe()).
							   memoryStoreEvictionPolicy(MemoryStoreEvictionPolicy.LRU).
							   overflowToOffHeap(false);
	
			this._requestsByIPCache = new Cache(cacheConfiguration);
			
			this._cacheManager.addCacheIfAbsent(this._requestsByIPCache);
		}

		this._log.info("{} has been initialized with {}", $THIS, this._configuration.getClass());
	}
	
	@PostConstruct 
	private void completeInitialization() {
		final String $THIS = this.getClass().getSimpleName() + "#" + this.hashCode();
		
		this._log.info("{} has been initialized with {}", $THIS, this._configuration);
	}
	
	/* (non-Javadoc)
	 * @see org.fao.fi.security.server.javax.filters.AbstractRequestValidatorFilter#getSecurityScheme()
	 */
	@Override
	protected String getSecurityScheme() {
		return BandwidthLimitedConstants.BANDWIDTH_LIMITER_SECURITY_TYPE_HEADER;
	}
	
	@Override
	public void filter(ContainerRequestContext requestContext) throws IOException {
		try {
			String host = this._servletRequest.getRemoteHost();
			String xff = requestContext.getHeaderString("X-FORWARDED-FOR");
			
			String actualHost = xff == null ? host : xff;
			
			if(actualHost != null)
				actualHost = actualHost.replaceAll(":[0-9]+$", "");
			
			this._log.debug("Intercepted a {} request for {} coming from host {} [ XFF: {} ]", requestContext.getMethod(), requestContext.getUriInfo().getAbsolutePath(), host, xff == null ? "NOT SET" : xff);

			RequestsByIP requestsByIP = null;
			
			final long now = System.currentTimeMillis();
			
			synchronized(this._requestsByIPCache) {
				Element currentRequestsElement = this._requestsByIPCache.get(actualHost);
				
				if(currentRequestsElement == null) {
					requestsByIP = RequestsByIP.newRequest();
				} else {
					requestsByIP = ((RequestsByIP)currentRequestsElement.getObjectValue());
					requestsByIP.increaseRequests();
				}
				
				currentRequestsElement = new Element(actualHost, requestsByIP);

				this._requestsByIPCache.put(currentRequestsElement);
			}

			if(requestsByIP._currentRequests > this._configuration.getMaxRequests()) {
				requestContext.abortWith(
					this.errorResponse(Response.Status.FORBIDDEN, "You have been prevented to access this resource")
				);
				
				this._log.warn("Blocked a {} request for {} coming from host {} [ XFF: {} ] having accounted for {} requests in the last {} seconds", 
						requestContext.getMethod(), 
						requestContext.getUriInfo().getAbsolutePath(), 
						host, xff == null ? "NOT SET" : xff,
						requestsByIP._currentRequests,
						Math.round(( now - requestsByIP._firstRequest ) * 0.001D));
			}
		} catch(Throwable t) {
			this._log.error("Unexpected {} caught: {}", t.getClass(), t.getMessage(), t);
			
			requestContext.abortWith(
				this.errorResponse(Response.Status.INTERNAL_SERVER_ERROR, t.getMessage())
			);
		}
	}
	
	static private class RequestsByIP {
		private long _firstRequest;
		private int  _currentRequests;
			
		/**
		 * Class constructor
		 */
		@SuppressWarnings("unused")
		public RequestsByIP() {
			this(System.currentTimeMillis(), 0);
		}

		/**
		 * Class constructor
		 *
		 * @param firstRequest
		 * @param currentRequests
		 */
		public RequestsByIP(long firstRequest, int currentRequests) {
			super();
			this._firstRequest = firstRequest;
			this._currentRequests = currentRequests;
		}
		
		static public RequestsByIP newRequest() {
			return new RequestsByIP(System.currentTimeMillis(), 1);
		}

		public long increaseRequests() {
			return ++this._currentRequests;
		}
	}	
}