package org.gcube.accounting.ut;

import java.net.URI;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.TimeZone;
import java.util.UUID;
import java.util.Vector;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.apache.log4j.Logger;
import org.gcube.accounting.common.helper.TimeHelper;
import org.gcube.accounting.datamodel.AbstractDelegateUsageRecord;
import org.gcube.accounting.datamodel.Consumer;
import org.gcube.accounting.datamodel.RawUsageRecord;
import org.gcube.accounting.datamodel.ResourceScope;
import org.gcube.accounting.datamodel.ResourceType;
import org.gcube.accounting.datamodel.ResourceTypes;
import org.gcube.accounting.datamodel.UsageRecord;
import org.gcube.accounting.datamodel.User;
import org.gcube.accounting.datamodel.adapter.EntryFilterRating;
import org.gcube.accounting.datamodel.reports.Report;
import org.gcube.accounting.datamodel.reports.aggregation.AggregatedResult;
import org.gcube.accounting.datamodel.reports.aggregation.FilterRating;
import org.gcube.accounting.exception.InvalidValueException;
import org.gcube.accounting.security.authn.HTTPBasicAuthentication;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.config.DefaultClientConfig;

/**
 * A client-side class for accessing the Usage Tracker
 * 
 * @author Paolo Fabriani
 * @author Ermanno Travaglino
 * 
 */

public class UsageTrackerRestAPI {

	/**
	 * A logger for this class
	 */
	private static Logger logger = Logger.getLogger(UsageTrackerRestAPI.class);

	private WebResource service;
	//	private static Client client;
	private Client client;
	//protected ObjectMapper mapper;

	private String uri;

	/**
	 * 
	 * @param uri the URI of the Usage Tracker
	 */
	public UsageTrackerRestAPI(String uri) { 
		ClientConfig config = new DefaultClientConfig();
		client = Client.create(config);
		this.uri=uri;
		this.service = client.resource(getBaseURI(this.uri));

	}

	private static URI getBaseURI(String uri) {
		return UriBuilder.fromUri(uri).build();
	}

	/**
	 * Insert a set of usage records
	 * 
	 * @param records
	 */
	public void insertUsageRecords(Collection<UsageRecord> records) {

		service=client.resource(this.uri);
		logger.debug("bulk upload of " + records.size() + " usage records...");
		for (UsageRecord ur : records) {
			this.insertUsageRecord(ur);
		}
		logger.debug("bulk upload completed");
	}

	/**
	 * Insert the new given usage record
	 * 
	 * @param ur
	 */
	public void insertUsageRecord(UsageRecord ur) {

		service=client.resource(this.uri);
		logger.debug("uploading usage record " + ur.getId() + "...");
		service.path("usagerecords").accept(MediaType.TEXT_PLAIN).type(
				MediaType.APPLICATION_XML).post(ur);
		logger.debug("uploaded");
	}

	/**
	 * Insert the new given usage record
	 * 
	 * @param ur
	 */
	public void insertUsageRecord(AbstractDelegateUsageRecord ur) {
		this.insertUsageRecord(ur.getDelegateUR());
	}



	/**
	 * Insert the given storage-specific usage record
	 * @param ur
	 */
	public void insertUsageRecord(RawUsageRecord ur) {

		service=client.resource(this.uri);
		logger.debug("uploading usage record " + ur.getId() + "...");
		//		System.out.println(ur.getResourceType());
		//		System.out.println(ur.getResourceScope());
		//		client.addFilter(new LoggingFilter(System.out));
		service.path("usagerecords").accept(
				MediaType.TEXT_PLAIN).type(MediaType.APPLICATION_XML).post(ur);
		logger.debug("uploaded");
	}


	//	/**
	//	 * Update an existing usage record. The match with the esisting record in
	//	 * the usage tracker is based on the record id
	//	 * 
	//	 * @param ur
	//	 */
	//	public void updateUsageRecord(UsageRecord ur) {
	//
	//		service=client.resource(this.uri);
	//		logger.debug("updating usage record " + ur.getId() + "...");
	//		service.path("usagerecords").path(ur.getId()).accept(
	//				MediaType.TEXT_PLAIN).type(MediaType.APPLICATION_XML).put(ur);
	//		logger.debug("updated");
	//	}
	//
	//	/**
	//	 * Update an existing usage record. The match with the esisting record in
	//	 * the usage tracker is based on the record id
	//	 * 
	//	 * @param ur
	//	 */
	//	public void updateUsageRecord(AbstractDelegateUsageRecord ur) {
	//		this.updateUsageRecord(ur.getDelegateUR());
	//	}

	/**
	 * Return the usage records matching the given query and aggregation fields for sorting/pagination purposes
	 * 
	 * Example: resourceType=job&resourceScope=testing&sortField=startTime&paginationStartPage=1&paginationRecords=6
	 * 
	 * Aggregation fields
	 * ======================
	 * sortField : field on which you want to order the results
	 * sortOrder : sort criteria (asc/desc)
	 * paginationStartRecord : start record of the pagination
	 * paginationRecords : number of records per page
	 * 
	 * @param query must have the form: name=value&name>value&kname=value&...
	 * @return
	 */
	public Collection<UsageRecord> getUsageRecords(String query) {
		service=client.resource(this.uri);
		//old query
		//RawUsageRecord[] records = service.path("usagerecords").path("raw").queryParam("query", query).get(RawUsageRecord[].class);
		RawUsageRecord[] records = null;
		if(query == null || query.equals("")){
			records = service.path("usagerecords").get(RawUsageRecord[].class);
		}
		else{
			records = service.path("usagerecords").queryParam(
					"query", query).get(RawUsageRecord[].class);
		}
		logger.debug("retrieved " + records.length + " usage records");
		Collection<UsageRecord> out = new Vector<UsageRecord>();
		for (UsageRecord r : records) {
			out.add(r);
		}
		return out;
	}



	/**
	 * Return the usage records matching the given query and aggregation fields
	 * 
	 * Example: to do
	 * 
	 * Aggregation fields
	 * ======================
	 * aggregationInterval : to do
	 * aggregationUnit : to do
	 * 
	 * @param query must have the form: name=value&name>value&kname=value&...
	 * @return
	 */
	//	public AggregatedInformation getAggregationInfo(String query) {
	//		service=client.resource(this.uri);
	//		//old query
	//		//RawUsageRecord[] records = service.path("usagerecords").path("raw").queryParam("query", query).get(RawUsageRecord[].class);
	//		AggregatedInformation information = null;
	//		
	//		 information = service.path("usagerecords").queryParam(
	//				"query", query).get(AggregatedInformation.class);
	//		
	//		return information;
	//	}

	//	/**
	//	 * Return the usage records matching the given query.
	//	 * 
	//	 * @param query must have the form: name=value&name>value&kname=value&...
	//	 * @return
	//	 */
	//	public Collection<UsageRecord> getUsageRecordsWithAggregation(String query, String aggregation) {
	//		service=client.resource(this.uri);
	//		RawUsageRecord[] records = service.path("usagerecords").path("raw").queryParam(
	//				"query", query).get(RawUsageRecord[].class);
	//		logger.debug("retrieved " + records.length + " usage records");
	//		Collection<UsageRecord> out = new Vector<UsageRecord>();
	//		for (UsageRecord r : records) {
	//			out.add(r);
	//		}
	//		return out;
	//	}




	//	/**
	//	 * Return the usage records matching the given query.
	//	 * 
	//	 * @deprecated use getUsageRecords(query) instead
	//	 * 
	//	 * @param query must have the form: name=value&name>value&kname=value&...
	//	 * @return
	//	 */
	//	public Collection<UsageRecord> getUsageRecordsByQuery(String query) {
	//		return this.getUsageRecords(query);
	//	}




	/**
	 * Query the Usage Tracker for update information from each infrastructure (resource owner)
	 * 
	 * @return
	 */
	public Collection<UpdateInfo> getUpdateInfo() {

		logger.debug("querying the Usage Tracker for update information...");
		service=client.resource(this.uri);
		UpdateInfo[] infos = service.path("meta").path("update_info").get(
				UpdateInfo[].class);
		logger.debug("done");
		Collection<UpdateInfo> out = new Vector<UpdateInfo>();
		for (UpdateInfo info : infos) {
			out.add(info);
		}
		return out;
	}

	/**
	 * Return the list of users with an associated usage record
	 * 
	 * @return
	 * @deprecated this operation is for development/testing purposes only. It
	 *             will be removed in future releases
	 */
	public Collection<User> getUsers() {
		logger.debug("retrieving tracked users...");
		service=client.resource(this.uri);
		User[] users = service.path("meta").path("users").get(User[].class);
		logger.debug("retrieved " + users.length + " users");
		Collection<User> out = new Vector<User>();
		for (User user : users) {
			out.add(user);
		}
		return out;
	}

	public Collection<Consumer> getTrackedConsumers() {
		service=client.resource(this.uri);
		Consumer[] consumers = service.path("meta").path("consumers").get(Consumer[].class);
		Collection<Consumer> out = new Vector<Consumer>();
		for(Consumer c:consumers)
			out.add(c);
		return out;
	}

	public Collection<ResourceType> getTrackedResourceTypes() {
		service=client.resource(this.uri);
		ResourceType[] resouceTypes = service.path("meta").path("resource_types").get(ResourceType[].class);
		Collection<ResourceType> out = new Vector<ResourceType>();
		for(ResourceType c:resouceTypes)
			out.add(c);
		return out;
	}

	public Collection<ResourceScope> getTrackedResourceScopes() {
		service=client.resource(this.uri);
		ResourceScope[] resourceScopes = service.path("meta").path("resource_scopes").get(ResourceScope[].class);
		Collection<ResourceScope> out = new Vector<ResourceScope>();
		for(ResourceScope c:resourceScopes)
			out.add(c);
		return out;
	}

	/**
	 * Return the number of the usage records matching the given query
	 * 
	 * 
	 * @param query
	 * @return
	 */
	public long getNumberOfRecords(String query) {
		service=client.resource(this.uri);
		//old query
		//RawUsageRecord[] records = service.path("usagerecords").path("raw").queryParam("query", query).get(RawUsageRecord[].class);
		String numberOfRecords = "";
		if(query == null || query.equals("")){
			numberOfRecords = service.path("usagerecords").path("records_size").get(String.class);
		}
		else{
			numberOfRecords  = service.path("usagerecords").path("records_size").queryParam(
					"query", query).get(String.class);
		}
		logger.debug(numberOfRecords + " usage records");
		return Long.valueOf(numberOfRecords);
	}

	/**
	 * Remove all elements in the Usage Tracker
	 * 
	 * @deprecated this operation is for development/testing purposes only. It
	 *             will be removed in future releases
	 */
	public void clear() {
		logger.debug("clearing the usage tracker...");
		service=client.resource(this.uri);
		service.path("usagerecords").delete();
		logger.debug("done");
	}

	//	/**
	//	 * Query the Usage Tracker for records with the given <key, value> pair
	//	 * 
	//	 * @param key
	//	 * @param value
	//	 * @return
	//	 * @deprecated use getUsageRecordsByQuery instead
	//	 */
	//	public Collection<UsageRecord> getUsageRecordByProperty(String key,
	//			String value) {
	//		logger.debug("retrieving usage records where '" + key + "' is '"
	//				+ value + "'...");
	//		return this.getUsageRecordsByQuery(key + "=" + value);
	//	}

	public void deleteUsageRecord(UsageRecord record) {
		service=client.resource(this.uri);
		logger.info("deleting usage record " + record.getId() + "...");
		service.path("usagerecords/"+record.getId()).delete();
		logger.info("deleted");
	}

	public void setUserToken(String username, String password) {
		client.addFilter(new HTTPBasicAuthentication(username,password));
	}

	public void setUserToken(String username, String password, String delegator) {
		client.addFilter(new HTTPBasicAuthentication(username,password,delegator));
	}




	//	public Collection<AggregatedResult> getReport(ReportType reportType, Period timePeriod, String aggregationGranule){
	//		service=client.resource(this.uri);
	//		AggregatedResult[] result = null;
	//		String query = "reportType="+reportType.toString()+"&timePeriod="+timePeriod.toString()+"&aggregationGranule="+aggregationGranule;
	//		System.out.println(query);
	//		result = service.path("reports").queryParam("query", query).get(AggregatedResult[].class);
	//
	//		Collection<AggregatedResult> out = new Vector<AggregatedResult>();
	//		for (AggregatedResult r : result) {
	//			out.add(r);
	//		}
	//		return out;		
	//	}

	private static void generateJobRecords (UsageTrackerRestAPI ut, int n){
		List<String> resourceScopes = Arrays.asList("/d4science.org","/d4science.org/vo1","/d4science.org/vo2","/d4science.org/vo1/vre1","/d4science.org/vo1/vre2");
		List<String> dataTypes = Arrays.asList("STORAGE", "DATABASE", "TREE", "GEO");
		List<String> consumerIds = Arrays.asList("luigi.fortunati","antonio.savini","ermanno.travaglino","pasquale.pagano","andrea.manzi","fabio.simeoni","john.gerbesiotis");
		List<String> resourceOwners = Arrays.asList("luigi.fortunati","antonio.savini","ermanno.travaglino","pasquale.pagano","andrea.manzi","fabio.simeoni","john.gerbesiotis");
		List<String> providerIds=Arrays.asList("cloud.eng.it","cnr.research-infrastructures.eu","fao.org","cern.ch");
		List<String> qualifiers=Arrays.asList("data-transformation","search","map-reduce");
		List<String> names=Arrays.asList("abc","acd","bca","bac");

		//insert storage usage records	
		for (int i = 0; i < n; i++) {

			RawUsageRecord t = new RawUsageRecord();

			//basic properties

			t.setResourceType(ResourceTypes.JOB.toString());	

			Collections.shuffle(consumerIds);
			t.setConsumerId("test");

			Collections.shuffle(resourceOwners);
			t.setCreatorId(resourceOwners.get(0));
			
			Collections.shuffle(resourceOwners);
			t.setResourceOwner(resourceOwners.get(0));

			Collections.shuffle(resourceScopes);
			t.setResourceScope("/testing/vo1");//resourceScopes.get(0));

			
			//specific properties
			t.setResourceSpecificProperty("jobId",UUID.randomUUID().toString());
			Collections.shuffle(qualifiers);
			t.setResourceSpecificProperty("jobQualifier",qualifiers.get(0));
			
			Collections.shuffle(names);
			t.setResourceSpecificProperty("jobName",names.get(0));
		
			
			t.setResourceSpecificProperty("jobStatus", "completed");
			
			t.setResourceSpecificProperty("vmsUsed", String.valueOf(i));

		



			long start2013 = 0;
			try {
				start2013 = new SimpleDateFormat("yyyy").parse("2012").getTime();
			} catch (ParseException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			}
			final long millisInYear2012 = 1000 * 60 * 60 * 24 * 365 + 1000;
			long millis = Math.round(millisInYear2012 * Math.random());
			Timestamp timeStamp = new Timestamp(start2013 + millis);
			Calendar startTime = randomDate();
			Calendar endTime =  randomDate();
			while(endTime.before(startTime)){
				startTime = randomDate();
				endTime = randomDate();
			}
			try {
				t.setStartTime(startTime.getTime());
				t.setEndTime(endTime.getTime());
				
				t.setResourceSpecificProperty("jobStart",String.valueOf(startTime.getTimeInMillis()));
				t.setResourceSpecificProperty("jobEnd",String.valueOf(endTime.getTimeInMillis()));
				long wallDuration = endTime.getTimeInMillis()-startTime.getTimeInMillis();
				t.setResourceSpecificProperty("wallDuration",String.valueOf(wallDuration));

			} catch (InvalidValueException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
			t.setCreateTime(randomDate().getTime());
			
			
			ut.insertUsageRecord(t);
		}
	}
	
	private static void generateTaskRecords (UsageTrackerRestAPI ut, int n){
		List<String> resourceScopes = Arrays.asList("/d4science.org","/d4science.org/vo1","/d4science.org/vo2","/d4science.org/vo1/vre1","/d4science.org/vo1/vre2");
		List<String> dataTypes = Arrays.asList("STORAGE", "DATABASE", "TREE", "GEO");
		List<String> consumerIds = Arrays.asList("luigi.fortunati","antonio.savini","ermanno.travaglino","pasquale.pagano","andrea.manzi","fabio.simeoni","john.gerbesiotis");
		List<String> resourceOwners = Arrays.asList("luigi.fortunati","antonio.savini","ermanno.travaglino","pasquale.pagano","andrea.manzi","fabio.simeoni","john.gerbesiotis");
		List<String> refHosts=Arrays.asList("cloud.eng.it","cnr.research-infrastructures.eu","fao.org","cern.ch");
		List<String> refVMs=Arrays.asList("vm1","vm2","vm3","vm4");
		List<String> qualifiers=Arrays.asList("data-transformation","search","map-reduce");
		List<String> names=Arrays.asList("abc","acd","bca","bac");

		
		//insert storage usage records	
		for (int i = 0; i < n; i++) {

			RawUsageRecord t = new RawUsageRecord();

			//basic properties

			t.setResourceType(ResourceTypes.TASK.toString());	

			Collections.shuffle(consumerIds);
			t.setConsumerId("test");

			Collections.shuffle(resourceOwners);
			t.setCreatorId(resourceOwners.get(0));
			
			Collections.shuffle(resourceOwners);
			t.setResourceOwner(resourceOwners.get(0));

			Collections.shuffle(resourceScopes);
			t.setResourceScope("/testing/vo1");//resourceScopes.get(0));

			
			//specific properties
			t.setResourceSpecificProperty("jobId",UUID.randomUUID().toString());
			
			
			Collections.shuffle(refHosts);
			t.setResourceSpecificProperty("refHost",refHosts.get(0));
			t.setResourceSpecificProperty("domain",refHosts.get(0));
			
			Collections.shuffle(refVMs);
			t.setResourceSpecificProperty("refVM",refVMs.get(0));
			
			Collections.shuffle(names);
			t.setResourceSpecificProperty("jobName",names.get(0));
		
			
			t.setResourceSpecificProperty("usagePhase", "completed");
			
			t.setResourceSpecificProperty("overallNetworkIn", String.valueOf(new Long(i*1110)));
			t.setResourceSpecificProperty("overallNetworkOut", String.valueOf(new Long(i*200)));
			t.setResourceSpecificProperty("cores", String.valueOf(new Integer(i*8)));
			t.setResourceSpecificProperty("processors", String.valueOf(new Integer(i)));
		



			long start2013 = 0;
			try {
				start2013 = new SimpleDateFormat("yyyy").parse("2012").getTime();
			} catch (ParseException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			}
			final long millisInYear2012 = 1000 * 60 * 60 * 24 * 365 + 1000;
			long millis = Math.round(millisInYear2012 * Math.random());
			Timestamp timeStamp = new Timestamp(start2013 + millis);
			Calendar startTime = randomDate();
			Calendar endTime =  randomDate();
			while(endTime.before(startTime)){
				startTime = randomDate();
				endTime = randomDate();
			}
			try {
				t.setStartTime(startTime.getTime());
				t.setEndTime(endTime.getTime());
				
				t.setResourceSpecificProperty("usageStart",String.valueOf(startTime.getTimeInMillis()));
				t.setResourceSpecificProperty("usageEnd",String.valueOf(endTime.getTimeInMillis()));
				long wallDuration = endTime.getTimeInMillis()-startTime.getTimeInMillis();
				t.setResourceSpecificProperty("wallDuration",String.valueOf(wallDuration));

			} catch (InvalidValueException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
			t.setCreateTime(randomDate().getTime());
			
			System.out.println(t);
			ut.insertUsageRecord(t);
		}
	}


	
	private static void generateStorageUsageRecords (UsageTrackerRestAPI ut, int n){
		List<String> operationTypes = Arrays.asList("GET","DELETE","UPDATE","PUT");
		List<String> resourceScopes = Arrays.asList("/d4science.org","/d4science.org/vo1","/d4science.org/vo2","/d4science.org/vo1/vre1","/d4science.org/vo1/vre2");
		List<String> dataTypes = Arrays.asList("STORAGE", "DATABASE", "TREE", "GEO");
		List<String> consumerIds = Arrays.asList("luigi.fortunati","antonio.savini","ermanno.travaglino","pasquale.pagano","andrea.manzi","fabio.simeoni","john.gerbesiotis");
		List<String> resourceOwners = Arrays.asList("luigi.fortunati","antonio.savini","ermanno.travaglino","pasquale.pagano","andrea.manzi","fabio.simeoni","john.gerbesiotis");
		List<String> providerIds=Arrays.asList("cloud.eng.it","cnr.research-infrastructures.eu","fao.org","cern.ch");
		List<String> qualifiers=Arrays.asList("application/xml","application/zip","image/jpeg","text/xml");
		List<String> callerIPs=Arrays.asList("etics.eng.it","statistical-manager.cnr.research-infrastructures.eu","sdmx.fao.org","data-transfer.cern.ch");
		List<String> objectURIs=Arrays.asList("http://example.org/absolute/URI/with/absolute/path/to/resource.txt");

		//insert storage usage records	
		for (int i = 0; i < n; i++) {

			RawUsageRecord t = new RawUsageRecord();

			
			//basic properties

			t.setResourceType(ResourceTypes.STORAGE_USAGE.toString());	

			Collections.shuffle(consumerIds);
			t.setConsumerId(consumerIds.get(0));

			Collections.shuffle(resourceOwners);
			t.setResourceOwner(resourceOwners.get(0));

			Collections.shuffle(resourceScopes);
			t.setResourceScope("/testing/vo1");

			
			//specific properties
			Collections.shuffle(providerIds);
			t.setResourceSpecificProperty("providerId",providerIds.get(0));
			
			t.setResourceSpecificProperty("objectURI", "http://example.org/absolute/URI/with/absolute/path/to/object"+i*2+".bulk");
			
			Collections.shuffle(operationTypes);
			t.setResourceSpecificProperty("operationType",operationTypes.get(0));
			
			Collections.shuffle(qualifiers);
			t.setResourceSpecificProperty("qualifier",qualifiers.get(0));
					

			Collections.shuffle(dataTypes);
			t.setResourceSpecificProperty("dataType",dataTypes.get(0));

			
			t.setResourceSpecificProperty("dataVolume", String.valueOf(10231089-i*2));
			t.setResourceSpecificProperty("dataCount", String.valueOf(1000-i*2));

			Collections.shuffle(callerIPs);
			t.setResourceSpecificProperty("callerIP", callerIPs.get(0));

			long start2013 = 0;
			try {
				start2013 = new SimpleDateFormat("yyyy").parse("2013").getTime();
			} catch (ParseException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			}
			final long millisInYear2012 = 1000 * 60 * 60 * 24 * 365 + 1000;
			long millis = Math.round(millisInYear2012 * Math.random());
			Timestamp timeStamp = new Timestamp(start2013 + millis);
			Calendar startTime = randomDate();
			Calendar endTime = randomDate();
			while(endTime.before(startTime)){
				startTime = randomDate();
				endTime = randomDate();
			}
			try {
				t.setStartTime(startTime.getTime());
				t.setEndTime(endTime.getTime());
			} catch (InvalidValueException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
			

			t.setCreateTime(randomDate().getTime());
//
//			try {
//
//				t.setStartTime(startTime.getTime());
//				t.setEndTime(endTime.getTime());
//
//			} catch (InvalidValueException e) {
//				// TODO Auto-generated catch block
//				e.printStackTrace();
//			}

			ut.insertUsageRecord(t);
		}
	}
	

	private static void generateStorageStatusRecords (UsageTrackerRestAPI ut, int n){
		List<String> resourceScopes = Arrays.asList("/d4science.org","/d4science.org/vo1","/d4science.org/vo2","/d4science.org/vo1/vre1","/d4science.org/vo1/vre2");
		List<String> dataTypes = Arrays.asList("STORAGE", "DATABASE", "TREE", "GEO");
		List<String> consumerIds = Arrays.asList("luigi.fortunati","antonio.savini","ermanno.travaglino","pasquale.pagano","andrea.manzi","fabio.simeoni","john.gerbesiotis");
		List<String> resourceOwners = Arrays.asList("luigi.fortunati","antonio.savini","ermanno.travaglino","pasquale.pagano","andrea.manzi","fabio.simeoni","john.gerbesiotis");
		List<String> providerIds=Arrays.asList("cloud.eng.it","cnr.research-infrastructures.eu","fao.org","cern.ch");
		List<String> qualifiers=Arrays.asList("application/xml","application/zip","image/jpeg","text/xml");

		//insert storage usage records	
		for (int i = 0; i < n; i++) {

			RawUsageRecord t = new RawUsageRecord();

			//basic properties

			t.setResourceType(ResourceTypes.STORAGE_STATUS.toString());	

			Collections.shuffle(consumerIds);
			t.setConsumerId(consumerIds.get(0));

			Collections.shuffle(resourceOwners);
			t.setCreatorId(resourceOwners.get(0));
			
			Collections.shuffle(resourceOwners);
			t.setResourceOwner(resourceOwners.get(0));

			Collections.shuffle(resourceScopes);
			t.setResourceScope("/testing/vo1");//resourceScopes.get(0));

			
			//specific properties
			Collections.shuffle(providerIds);
			t.setResourceSpecificProperty("providerId",providerIds.get(0));
			
			Collections.shuffle(qualifiers);
			t.setResourceSpecificProperty("qualifier",qualifiers.get(0));
					

			Collections.shuffle(dataTypes);
			t.setResourceSpecificProperty("dataType",dataTypes.get(0));

			int max = 10000;
			int min = 1;
			Random r = new Random();
			int random = (r.nextInt(max - min + 1) + min);
			while(random < 0)
				random = r.nextInt(max - min + 1) + min;
			t.setResourceSpecificProperty("dataVolume", String.valueOf(random));
			t.setResourceSpecificProperty("dataCount", String.valueOf(10100-i*2));


			long start2013 = 0;
			try {
				start2013 = new SimpleDateFormat("yyyy").parse("2012").getTime();
			} catch (ParseException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			}
			final long millisInYear2012 = 1000 * 60 * 60 * 24 * 365 + 1000;
			long millis = Math.round(millisInYear2012 * Math.random());
			Timestamp timeStamp = new Timestamp(start2013 + millis);
			Calendar startTime = randomDate();
			Calendar endTime =  randomDate();
			while(endTime.before(startTime)){
				startTime = randomDate();
				endTime = randomDate();
			}
			//t.setCreateTime(randomDate().getTime());
			
//			SimpleDateFormat isoFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
//		    isoFormat.setTimeZone(TimeZone.getTimeZone("GMT-8"));
//		    Date date = null;
//		    try {
//				date = isoFormat.parse("2013-07-19T09:30:00");
//			} catch (ParseException e1) {
//				// TODO Auto-generated catch block
//				e1.printStackTrace();
//			}
//		    System.out.println(new DateTime(2013,7,19,23,59,00,DateTimeZone.forID("America/New_York")).toString());
//				t.setCreateTime(new DateTime(2013,7,19,23,59,00,DateTimeZone.forID("America/New_York")).toDate());
			t.setCreateTime(randomDate().getTime());
				try {
					t.setStartTime(randomDate().getTime());//new DateTime(2013,7,19,23,59,00,DateTimeZone.forID("America/New_York")).toDate());
					t.setEndTime(randomDate().getTime());

				} catch (InvalidValueException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
//				System.out.println("createTime = "+t.getStartTime());
//				AggregatedResult re = new AggregatedResult();
//				DateTime dt = new DateTime(t.getCreateTime()).withTimeAtStartOfDay().withZone(DateTimeZone.forTimeZone(TimeZone.getTimeZone("PST")));
//				re.setTime(dt);
//				System.out.println("timeeee = "+re.getTime());
			System.out.println("record = "+t);
			ut.insertUsageRecord(t);
		}
	}

	private static void generateServiceRecords (UsageTrackerRestAPI ut, int n){
		List<String> resourceScopes = Arrays.asList("/d4science.org","/d4science.org/vo1","/d4science.org/vo2","/d4science.org/vo1/vre1","/d4science.org/vo1/vre2");
		List<String> consumerIds = Arrays.asList("luigi.fortunati","antonio.savini","ermanno.travaglino","pasquale.pagano","andrea.manzi","fabio.simeoni","john.gerbesiotis");
		List<String> resourceOwners = Arrays.asList("luigi.fortunati","antonio.savini","ermanno.travaglino","pasquale.pagano","andrea.manzi","fabio.simeoni","john.gerbesiotis");
			
		List<String> callerIPs=Arrays.asList("etics.eng.it","statistical-manager.cnr.research-infrastructures.eu","sdmx.fao.org","data-transfer.cern.ch");
		List<String> domains=Arrays.asList("cern.ch","fao.org","eng.it","cnr.it");
		List<String> serviceClasses=Arrays.asList("Personalisation","ContentManagement","ContentManagement","ContentManagement");
		List<String> serviceNames=Arrays.asList("GCubeModelLibrary","GCubeModelLibrary","StorageManagementService","ContentManager");
		
		List<String> refHosts=Arrays.asList("etics.eng.it","statistical-manager.cnr.research-infrastructures.eu","sdmx.fao.org","data-transfer.cern.ch");
		List<String> refVMs=Arrays.asList("etics.eng.it","statistical-manager.cnr.research-infrastructures.eu","sdmx.fao.org","data-transfer.cern.ch");

		//insert storage usage records	
		for (int i = 0; i < n; i++) {

			RawUsageRecord t = new RawUsageRecord();

			
			//basic properties

			t.setResourceType("service");	

			Collections.shuffle(consumerIds);
			t.setConsumerId(consumerIds.get(0));

			Collections.shuffle(resourceOwners);
			t.setResourceOwner(resourceOwners.get(0));

			Collections.shuffle(resourceScopes);
			t.setResourceScope("/testing/vo1");

			
			//specific properties
			Collections.shuffle(refHosts);
			t.setResourceSpecificProperty("refHost", refHosts.get(0));
			
			Collections.shuffle(refVMs);
			t.setResourceSpecificProperty("refVM", refVMs.get(0));
			
			Collections.shuffle(serviceClasses);
			t.setResourceSpecificProperty("serviceClass", serviceClasses.get(0));
			
			Collections.shuffle(serviceNames);
			t.setResourceSpecificProperty("serviceName", serviceNames.get(0));
			
			Collections.shuffle(domains);
			t.setResourceSpecificProperty("domain", domains.get(0));
			
			t.setResourceSpecificProperty("averageInvocationTime", String.valueOf(120-i*i));
			t.setResourceSpecificProperty("invocationCount", String.valueOf(120-i*2));

			Collections.shuffle(callerIPs);
			t.setResourceSpecificProperty("callerIP", callerIPs.get(0));

			long start2013 = 0;
			try {
				start2013 = new SimpleDateFormat("yyyy").parse("2013").getTime();
			} catch (ParseException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			}
			final long millisInYear2012 = 1000 * 60 * 60 * 24 * 365 + 1000;
			long millis = Math.round(millisInYear2012 * Math.random());
			Timestamp timeStamp = new Timestamp(start2013 + millis);
			Calendar startTime = randomDate();
			Calendar endTime = randomDate();
			while(endTime.before(startTime)){
				startTime = randomDate();
				endTime = randomDate();
			}
			
			try {
				t.setStartTime(startTime.getTime());
				t.setEndTime(endTime.getTime());

			} catch (InvalidValueException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}

			t.setCreateTime(randomDate().getTime());
//
//			try {
//
//				t.setStartTime(startTime.getTime());
//				t.setEndTime(endTime.getTime());
//
//			} catch (InvalidValueException e) {
//				// TODO Auto-generated catch block
//				e.printStackTrace();
//			}

			ut.insertUsageRecord(t);
		}
	}
	
	public Collection<AggregatedResult> getReport(Report r) {
		service=client.resource(this.uri);
		
		AggregatedResult[] results = service.path("reports").path(r.getResourceType()).path(r.getReportType()).post(AggregatedResult[].class,r);
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();
		for(AggregatedResult c : results)
			out.add(c);
		return out;
		
	}
	
	public FilterRating getFilterRating(Report r) {
		service=client.resource(this.uri);
		
		FilterRating filterRating = service.path("filters").path(r.getResourceType()).path(r.getReportType()).post(FilterRating.class,r);
		
		return filterRating;
		
	}
	
	public static void main(String[] args) {
		UsageTrackerRestAPI ut  = new UsageTrackerRestAPI("http://gcube-dev-accounting.res.eng.it:8080/usagetracker/rest");
		
		//generateStorageUsageRecords(ut,1000);
//		generateTaskRecords(ut, 100);
//		generateJobRecords(ut, 100);
//		generateServiceRecords(ut, 100);
//		generateStorageStatusRecords(ut, 100);
//		generateStorageUsageRecords(ut, 100);

		Report r = new Report();
		
		r.setResourceType("storage-status");
//		List<String> reportTypes = Arrays.asList("avg-volume","avg-entities");
//		Collections.shuffle(reportTypes);
//		
	
		r.setReportType("volume");
		r.setAggregationGranule("day");
		r.setDimension("resourceOwner");
		
		
		Map<String,List<String>> genericFilters = new HashMap<String, List<String>>();
		ArrayList<String> genericValues = new ArrayList<String>();
		//genericValues.add("/d4science.research-infrastructures.eu/Ecosystem");
		//genericValues.add("cloud.eng.it");
//		genericValues.add("gianpaolo.coro");
//		genericValues.add("loredana.liccardo");
//
//		genericFilters.put("resourceOwner", genericValues);

		ArrayList<String> val = new ArrayList<String>();
//		val.add("/gcube");
//		genericFilters.put("resourceScope", val);
		
		
		r.setFilters(genericFilters);

		Calendar cal = Calendar.getInstance();
		cal.set(2014,9,1,0,0,0);
		r.setStartPeriod(cal.getTime());
		cal.set(2014,9,30,23,59,59);
		r.setEndPeriod(cal.getTime());

//		System.out.println(r.getStartPeriod());
//		System.out.println(r.getEndPeriod());
		
		System.out.println(r);
		
		Collection<AggregatedResult> results = ut.getReport(r);
		for (AggregatedResult aggregatedResult : results) {
			System.out.println(aggregatedResult.toString());
			//System.out.println(aggregatedResult.getTime().getYear()+" - "+aggregatedResult.getTime().getMonthOfYear() + "-"+aggregatedResult.getTime().getDayOfMonth()+" "+ aggregatedResult.getMetric() + " "+ aggregatedResult.getValue() + " "+aggregatedResult.getDimension());
		}
		
//		System.out.println("filtering....\n");
//		FilterRating rating = ut.getFilterRating(r);
//		for (EntryFilterRating entryFilterRating : rating.getRating()) {
//			System.out.println(entryFilterRating.getEntry() + " - " +entryFilterRating.getValue());
//		}
//		for (Map.Entry<String, Double> entry : rating.getRating()) {
//			if (entry!= null){
//				String filterName = entry.getKey();
//				Double filterValue = entry.getValue();
//				System.out.println("name = "+filterName+" - "+"value = "+filterValue);
//			}
//		}
//		
		//System.out.println(randomDate().getTimeInMillis());
		//generateStorageUsageRecords(ut);
		//generateStorageStatusRecords(ut);
		//generateServiceRecords(ut);
//		generateJobRecords(ut);
//		generateTaskRecords(ut);
		//		GenericReport g = new GenericReport();
		//		g.setConsumerId("noconsumer");
		//		g.setResourceOwner("noresourceowner");
		//		StorageUsageReport d = new StorageUsageReport();
		//		d.setResourceOwner("ciao");
		//		g.setReportType(d);




		//ut.getReport(reportType, timePeriod, aggregationGranule);
		//System.out.println(ut.getUsageRecords(""));
		//for (int i = 0; i < 100; i++) {
		//System.out.println("i = "+(550+i));
//		
//		long c = ut.getNumberOfRecords("startTime=2008-01-01,2015-11-31&resourceScope=/gcube&resourceType=storage-status");
//		System.out.println("#records = "+c);
//		Collection<UsageRecord> records = ut.getUsageRecords("startTime=2013-06-18,2013-07-18&resourceScope=/gcube&rsp:providerId=cern.ch,fao.org&resourceType=storage-status&sortField=startTime&sortOrder=desc");
//				int i=0;
//				for (Iterator<UsageRecord> iterator = records.iterator(); iterator.hasNext();) {
//					UsageRecord usageRecord = (UsageRecord) iterator.next();
//					System.out.println("\n"+ (++i) +" id = "+usageRecord.getId() + " - " + usageRecord.getStartTime() + "-" + usageRecord.getEndTime());
//					
//				}
//		



		



		//insert service records
		//		for (int i = 0; i < 1000; i++) {
		//			
		//			RawUsageRecord t = new RawUsageRecord();
		//			t.setAggregatedId("guest");
		//			t.setConsumerId("luigi.fortunati");
		//			t.setResourceScope("/vo1/testing");
		//			long start2012 = 0;
		//			try {
		//				start2012 = new SimpleDateFormat("yyyy").parse("2012").getTime();
		//			} catch (ParseException e1) {
		//				// TODO Auto-generated catch block
		//				e1.printStackTrace();
		//			}
		//			 final long millisInYear2012 = 1000 * 60 * 60 * 24 * 365 + 1000; // Have to account for the leap second!
		//			    long millis = Math.round(millisInYear2012 * Math.random());
		//			    Timestamp timeStamp = new Timestamp(start2012 + millis);
		//			t.setCreateTime(randomDate());
		//			try {
		//				t.setStartTime(randomDate());
		//			
		//			t.setCreatorId("test");
		//			t.setEndTime(randomDate());
		//			t.setResourceOwner("test");
		//			
		//			t.setResourceType("service");
		//			t.setResourceSpecificProperty("callerIP", "192.168.125."+(i+1)%254);
		//			t.setResourceSpecificProperty("invocationCount", String.valueOf(i*2+i*3));
		//			t.setResourceSpecificProperty("averageInvocationTime", String.valueOf(millisInYear2012));
		//			t.setResourceSpecificProperty("serviceClass", "service.class.x");
		//			t.setResourceSpecificProperty("serviceName", "service.name.y");
		//
		//			ut.insertUsageRecord(t);
		//			} catch (InvalidValueException e) {
		//				// TODO Auto-generated catch block
		//				e.printStackTrace();
		//			}
		//			}
		//		

		//		Collection<UsageRecord> c = ut.getUsageRecords("sortField=startTime&paginationStartPage=1&paginationRecords=6");
		//		for (Iterator iterator = c.iterator(); iterator.hasNext();) {
		//			UsageRecord usageRecord = (UsageRecord) iterator.next();
		//			System.out.println("\nid = "+usageRecord.getId() + " - startTime = "+TimeHelper.getISO8601(usageRecord.getStartTime().getTimeInMillis()) + "- endTime = "+TimeHelper.getISO8601(usageRecord.getEndTime().getTimeInMillis()));
		//			
		//		}

		//System.out.println(ut.getNumberOfRecords(""));


	}

//	public static void main(String[] args) {
//		RawUsageRecord t = new RawUsageRecord();
//		Calendar startTime = randomDate();
//		Calendar endTime =  randomDate();
//		while(endTime.before(startTime)){
//			startTime = randomDate();
//			endTime = randomDate();
//		}
//		try {
//			t.setStartTime(startTime.getTime());
//			t.setEndTime(endTime.getTime());
//			
//			System.out.println("usageStart(Date) = "+startTime.getTime());
//			System.out.println("usageEnd(Date) = "+endTime.getTime());
//			t.setResourceSpecificProperty("usageStart",String.valueOf(startTime.getTimeInMillis()));
//			t.setResourceSpecificProperty("usageEnd",String.valueOf(endTime.getTimeInMillis()));
//		}
//		catch(Exception e){}
//		
//		System.out.println("usageStart(long) = "+t.getResourceSpecificProperty("usageStart"));
//		System.out.println("usageEnd(long) = "+t.getResourceSpecificProperty("usageEnd"));
//		
//		long usageEnd = Long.valueOf(t.getResourceSpecificProperty("usageEnd"));
//		long usageStart = Long.valueOf(t.getResourceSpecificProperty("usageStart"));
//		
//		System.out.println("usageStart(longCast) = "+usageStart);
//		System.out.println("usageEnd(longCast) = "+usageEnd);
//		
//		System.out.println("(usageEnd - usageStart) long = " + (usageEnd-usageStart)/1000);
//		Date usageDiff = new Date(Math.abs(usageEnd-usageStart));
//		System.out.println("(usageEnd - usageStart) date = " + usageDiff);
//
//	}
	

	public static GregorianCalendar randomDate(){
		Calendar calendar = new GregorianCalendar();
		int currentYear = 2013;
		int max = 365;
		int min = 1;
		int randomNum = random.nextInt(max - min + 1) + min;
		calendar.set(Calendar.DAY_OF_YEAR-1, randomNum);
//		calendar.set(Calendar.DAY_OF_MONTH, 1);
//		calendar.set(Calendar.MONTH, 0);
		calendar.set(Calendar.YEAR, currentYear-random.nextInt(2));
		return (GregorianCalendar) calendar;
	}

	private static final Random random = new Random();

}
