/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.blob;

import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.base.StandardSystemProperty;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.io.Closeables;
import com.google.common.io.Files;
import com.google.common.util.concurrent.ListenableFutureTask;
import com.google.common.util.concurrent.MoreExecutors;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.sql.Timestamp;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.LineIterator;
import org.apache.jackrabbit.oak.commons.IOUtils;
import org.apache.jackrabbit.oak.plugins.blob.BlobGarbageCollector;
import org.apache.jackrabbit.oak.plugins.blob.BlobReferenceRetriever;
import org.apache.jackrabbit.oak.plugins.blob.GarbageCollectorFileState;
import org.apache.jackrabbit.oak.plugins.blob.ReferenceCollector;
import org.apache.jackrabbit.oak.spi.blob.GarbageCollectableBlobStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MarkSweepGarbageCollector
implements BlobGarbageCollector {
    public static final Logger LOG = LoggerFactory.getLogger(MarkSweepGarbageCollector.class);
    public static final String NEWLINE = StandardSystemProperty.LINE_SEPARATOR.value();
    public static final String TEMP_DIR = StandardSystemProperty.JAVA_IO_TMPDIR.value();
    public static final int DEFAULT_BATCH_COUNT = 2048;
    private final long maxLastModifiedInterval;
    private final boolean runConcurrently;
    private final GarbageCollectableBlobStore blobStore;
    private final BlobReferenceRetriever marker;
    private final GarbageCollectorFileState fs;
    private final Executor executor;
    private final int batchCount;
    private State state = State.NOT_RUNNING;

    public MarkSweepGarbageCollector(BlobReferenceRetriever marker, GarbageCollectableBlobStore blobStore, Executor executor, String root, int batchCount, boolean runBackendConcurrently, long maxLastModifiedInterval) throws IOException {
        this.executor = executor;
        this.blobStore = blobStore;
        this.marker = marker;
        this.batchCount = batchCount;
        this.runConcurrently = runBackendConcurrently;
        this.maxLastModifiedInterval = maxLastModifiedInterval;
        this.fs = new GarbageCollectorFileState(root);
    }

    public MarkSweepGarbageCollector(BlobReferenceRetriever marker, GarbageCollectableBlobStore blobStore, Executor executor) throws IOException {
        this(marker, blobStore, executor, TEMP_DIR, 2048, true, TimeUnit.HOURS.toMillis(24L));
    }

    public MarkSweepGarbageCollector(BlobReferenceRetriever marker, GarbageCollectableBlobStore blobStore, Executor executor, long maxLastModifiedInterval) throws IOException {
        this(marker, blobStore, executor, TEMP_DIR, 2048, true, maxLastModifiedInterval);
    }

    @Override
    public void collectGarbage() throws Exception {
        this.markAndSweep();
    }

    public State getState() {
        return this.state;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void markAndSweep() throws IOException, InterruptedException {
        boolean threw = true;
        try {
            Stopwatch sw = Stopwatch.createStarted();
            LOG.info("Starting Blob garbage collection");
            this.mark();
            int deleteCount = this.sweep();
            threw = false;
            LOG.info("Blob garbage collection completed in {}. Number of blobs deleted [{}]", (Object)sw.toString(), (Object)deleteCount);
        }
        finally {
            Closeables.close(this.fs, threw);
            this.state = State.NOT_RUNNING;
        }
    }

    private void mark() throws IOException, InterruptedException {
        this.state = State.MARKING;
        LOG.debug("Starting mark phase of the garbage collector");
        ListenableFutureTask<Integer> blobIdRetriever = ListenableFutureTask.create(new BlobIdRetriever());
        if (this.runConcurrently) {
            this.executor.execute(blobIdRetriever);
        } else {
            MoreExecutors.sameThreadExecutor().execute(blobIdRetriever);
        }
        this.iterateNodeTree();
        try {
            blobIdRetriever.get();
        }
        catch (ExecutionException e) {
            LOG.warn("Error occurred while fetching all the blobIds from the BlobStore. GC would continue with the blobIds retrieved so far", e.getCause());
        }
        this.difference();
        LOG.debug("Ending mark phase of the garbage collector");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void difference() throws IOException {
        LOG.debug("Starting difference phase of the garbage collector");
        FileLineDifferenceIterator iter = new FileLineDifferenceIterator(this.fs.getMarkedRefs(), this.fs.getAvailableRefs(), this.batchCount);
        BufferedWriter bufferWriter = null;
        try {
            bufferWriter = Files.newWriter(this.fs.getGcCandidates(), Charsets.UTF_8);
            ArrayList<String> expiredSet = Lists.newArrayList();
            int numCandidates = 0;
            while (iter.hasNext()) {
                expiredSet.add(iter.next());
                if (expiredSet.size() <= this.getBatchCount()) continue;
                numCandidates += expiredSet.size();
                MarkSweepGarbageCollector.saveBatchToFile(expiredSet, bufferWriter);
            }
            if (!expiredSet.isEmpty()) {
                numCandidates += expiredSet.size();
                MarkSweepGarbageCollector.saveBatchToFile(expiredSet, bufferWriter);
            }
            LOG.debug("Found GC candidates - " + numCandidates);
        }
        catch (Throwable throwable) {
            IOUtils.closeQuietly(bufferWriter);
            throw throwable;
        }
        IOUtils.closeQuietly(bufferWriter);
        LOG.debug("Ending difference phase of the garbage collector");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int sweep() throws IOException {
        int count = 0;
        this.state = State.SWEEPING;
        LOG.debug("Starting sweep phase of the garbage collector");
        ConcurrentLinkedQueue<String> exceptionQueue = new ConcurrentLinkedQueue<String>();
        LineIterator iterator = FileUtils.lineIterator(this.fs.getGcCandidates(), Charsets.UTF_8.name());
        ArrayList<String> ids = Lists.newArrayList();
        while (iterator.hasNext()) {
            ids.add((String)iterator.next());
            if (ids.size() <= this.getBatchCount()) continue;
            count += ids.size();
            this.executor.execute(new Sweeper(ids, exceptionQueue));
            ids = Lists.newArrayList();
        }
        if (!ids.isEmpty()) {
            count += ids.size();
            this.executor.execute(new Sweeper(ids, exceptionQueue));
        }
        count -= exceptionQueue.size();
        BufferedWriter writer = null;
        try {
            if (!exceptionQueue.isEmpty()) {
                writer = Files.newWriter(this.fs.getGarbage(), Charsets.UTF_8);
                MarkSweepGarbageCollector.saveBatchToFile(Lists.newArrayList(exceptionQueue), writer);
            }
        }
        finally {
            LineIterator.closeQuietly(iterator);
            IOUtils.closeQuietly(writer);
        }
        if (!exceptionQueue.isEmpty()) {
            LOG.warn("Unable to delete some blob entries from the blob store. Details around such blob entries can be found in [{}]", (Object)this.fs.getGarbage().getAbsolutePath());
        }
        LOG.debug("Ending sweep phase of the garbage collector");
        return count;
    }

    private int getBatchCount() {
        return this.batchCount;
    }

    private long getLastMaxModifiedTime() {
        return this.maxLastModifiedInterval > 0L ? System.currentTimeMillis() - this.maxLastModifiedInterval : 0L;
    }

    static void saveBatchToFile(List<String> ids, BufferedWriter writer) throws IOException {
        writer.append(Joiner.on(NEWLINE).join(ids));
        writer.append(NEWLINE);
        ids.clear();
        writer.flush();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void iterateNodeTree() throws IOException {
        final BufferedWriter writer = Files.newWriter(this.fs.getMarkedRefs(), Charsets.UTF_8);
        final AtomicInteger count = new AtomicInteger();
        try {
            this.marker.collectReferences(new ReferenceCollector(){
                private final List<String> idBatch;
                private final boolean debugMode;
                {
                    this.idBatch = Lists.newArrayListWithCapacity(MarkSweepGarbageCollector.this.getBatchCount());
                    this.debugMode = LOG.isTraceEnabled();
                }

                @Override
                public void addReference(String blobId) {
                    if (this.debugMode) {
                        LOG.trace("BlobId : {}", (Object)blobId);
                    }
                    try {
                        Iterator<String> idIter = MarkSweepGarbageCollector.this.blobStore.resolveChunks(blobId);
                        while (idIter.hasNext()) {
                            String id = idIter.next();
                            this.idBatch.add(id);
                            if (this.idBatch.size() >= MarkSweepGarbageCollector.this.getBatchCount()) {
                                MarkSweepGarbageCollector.saveBatchToFile(this.idBatch, writer);
                                this.idBatch.clear();
                            }
                            if (this.debugMode) {
                                LOG.trace("chunkId : {}", (Object)id);
                            }
                            count.getAndIncrement();
                        }
                        if (!this.idBatch.isEmpty()) {
                            MarkSweepGarbageCollector.saveBatchToFile(this.idBatch, writer);
                            this.idBatch.clear();
                        }
                    }
                    catch (Exception e) {
                        throw new RuntimeException("Error in retrieving references", e);
                    }
                }
            });
            LOG.info("Number of valid blob references marked under mark phase of Blob garbage collection [{}]", (Object)count.get());
            this.fs.sort(this.fs.getMarkedRefs());
        }
        finally {
            IOUtils.closeQuietly(writer);
        }
    }

    private static String timestampToString(long timestamp) {
        return (new Timestamp(timestamp) + "00").substring(0, 23);
    }

    static class FileLineDifferenceIterator
    implements Iterator<String> {
        private final LineIterator markedIter;
        private final LineIterator allIter;
        private final ArrayDeque<String> queue;
        private final int batchSize;
        private boolean done;
        private TreeSet<String> markedBuffer;

        public FileLineDifferenceIterator(File marked, File available, int batchSize) throws IOException {
            this.markedIter = FileUtils.lineIterator(marked);
            this.allIter = FileUtils.lineIterator(available);
            this.batchSize = batchSize;
            this.queue = new ArrayDeque(batchSize);
            this.markedBuffer = Sets.newTreeSet();
        }

        private void close() {
            LineIterator.closeQuietly(this.markedIter);
            LineIterator.closeQuietly(this.allIter);
        }

        @Override
        public boolean hasNext() {
            if (!this.queue.isEmpty()) {
                return true;
            }
            if (this.done) {
                return false;
            }
            if (!this.markedIter.hasNext() && !this.allIter.hasNext()) {
                this.done = true;
                this.close();
                return false;
            }
            this.queue.addAll(this.difference());
            if (!this.queue.isEmpty()) {
                return true;
            }
            this.done = true;
            this.close();
            return false;
        }

        @Override
        public String next() {
            return this.nextDifference();
        }

        public String nextDifference() {
            if (!this.hasNext()) {
                throw new NoSuchElementException("No more difference");
            }
            return this.queue.remove();
        }

        protected Set<String> difference() {
            TreeSet<Object> gcSet = new TreeSet<String>();
            while (this.allIter.hasNext() && gcSet.size() < this.batchSize) {
                Object stre;
                TreeSet<Object> allBuffer = new TreeSet<Object>();
                while (this.markedIter.hasNext() && this.markedBuffer.size() < this.batchSize) {
                    stre = this.markedIter.next();
                    this.markedBuffer.add((String)stre);
                }
                while (this.allIter.hasNext() && allBuffer.size() < this.batchSize) {
                    stre = this.allIter.next();
                    allBuffer.add(stre);
                }
                if (this.markedBuffer.isEmpty()) {
                    gcSet = allBuffer;
                    continue;
                }
                gcSet.addAll(Sets.difference(allBuffer, this.markedBuffer));
                if (((String)allBuffer.last()).compareTo(this.markedBuffer.last()) < 0) {
                    TreeSet<String> markedLeftoverBuffer = Sets.newTreeSet();
                    markedLeftoverBuffer.addAll(this.markedBuffer.tailSet((String)allBuffer.last(), false));
                    this.markedBuffer = markedLeftoverBuffer;
                    Object var3_3 = null;
                    continue;
                }
                this.markedBuffer.clear();
            }
            return gcSet;
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }

    private class BlobIdRetriever
    implements Callable<Integer> {
        private BlobIdRetriever() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public Integer call() throws Exception {
            LOG.debug("Starting retrieve of all blobs");
            BufferedWriter bufferWriter = null;
            int blobsCount = 0;
            try {
                bufferWriter = new BufferedWriter(new FileWriter(MarkSweepGarbageCollector.this.fs.getAvailableRefs()));
                Iterator<String> idsIter = MarkSweepGarbageCollector.this.blobStore.getAllChunkIds(MarkSweepGarbageCollector.this.getLastMaxModifiedTime());
                ArrayList<String> ids = Lists.newArrayList();
                while (idsIter.hasNext()) {
                    ids.add(idsIter.next());
                    if (ids.size() <= MarkSweepGarbageCollector.this.getBatchCount()) continue;
                    blobsCount += ids.size();
                    MarkSweepGarbageCollector.saveBatchToFile(ids, bufferWriter);
                }
                if (!ids.isEmpty()) {
                    blobsCount += ids.size();
                    MarkSweepGarbageCollector.saveBatchToFile(ids, bufferWriter);
                }
                MarkSweepGarbageCollector.this.fs.sort(MarkSweepGarbageCollector.this.fs.getAvailableRefs());
                LOG.debug("Number of blobs present in BlobStore : [{}] which have been last modified before [{}]", (Object)blobsCount, (Object)MarkSweepGarbageCollector.timestampToString(MarkSweepGarbageCollector.this.getLastMaxModifiedTime()));
            }
            catch (Throwable throwable) {
                IOUtils.closeQuietly(bufferWriter);
                throw throwable;
            }
            IOUtils.closeQuietly(bufferWriter);
            return blobsCount;
        }
    }

    class Sweeper
    implements Runnable {
        private final ConcurrentLinkedQueue<String> exceptionQueue;
        private final List<String> ids;

        public Sweeper(List<String> ids, ConcurrentLinkedQueue<String> exceptionQueue) {
            this.exceptionQueue = exceptionQueue;
            this.ids = ids;
        }

        @Override
        public void run() {
            try {
                LOG.debug("Blob ids to be deleted {}", this.ids);
                boolean deleted = MarkSweepGarbageCollector.this.blobStore.deleteChunks(this.ids, MarkSweepGarbageCollector.this.getLastMaxModifiedTime());
                if (!deleted) {
                    this.exceptionQueue.addAll(this.ids);
                }
            }
            catch (Exception e) {
                LOG.warn("Error occurred while deleting blob with ids [{}]", this.ids, (Object)e);
                this.exceptionQueue.addAll(this.ids);
            }
        }
    }

    public static enum State {
        NOT_RUNNING,
        MARKING,
        SWEEPING;

    }
}

