/**
 * 
 */
package org.gcube.accounting.persistence;

import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.ServiceLoader;

import org.gcube.accounting.aggregation.scheduler.AggregationScheduler;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.common.scope.impl.ScopeBean;
import org.gcube.common.scope.impl.ScopeBean.Type;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR) http://www.lucafrosini.com/
 *
 */
public abstract class PersistenceFactory {
	
	private static final Logger logger = LoggerFactory.getLogger(PersistenceFactory.class);
	
	public final static String HOME_SYSTEM_PROPERTY = "user.home";
	
	private static final String ACCOUTING_FALLBACK_FILENAME = "accountingFallback.log";
	
	private static String fallbackLocation;
	
	private static Map<String, Persistence> persistences;
	
	static {
		persistences = new HashMap<String, Persistence>();
	}
	
	private static File file(File file) throws IllegalArgumentException {
		
		if(!file.isDirectory()){
			file = file.getParentFile();
		}

		//create folder structure if not exist
		if (!file.exists())
			file.mkdirs();
		
		return file;
		
	}
	
	public synchronized static void setFallbackLocation(String path){
		if(fallbackLocation == null){
			if(path==null){
				path = System.getProperty(HOME_SYSTEM_PROPERTY);
			}
			file(new File(path));
			fallbackLocation = path;
		}
	}
	
	public static Persistence getPersistence() {
		String scope = ScopeProvider.instance.get();
		if(scope==null){
			logger.error("No Scope available. FallbackPersistence will be used");
			File fallbackFile = new File(fallbackLocation, ACCOUTING_FALLBACK_FILENAME);
			return new FallbackPersistence(fallbackFile);
		}
		
		Persistence persistence = persistences.get(scope);
		if(persistence==null){
		
			ScopeBean bean = new ScopeBean(scope);
			if(bean.is(Type.VRE)){ 
				bean = bean.enclosingScope(); 
			}
			String name = bean.name();
			
			File fallbackFile = new File(fallbackLocation, String.format("%s.%s", name, ACCOUTING_FALLBACK_FILENAME));
			FallbackPersistence fallbackPersistence = new FallbackPersistence(fallbackFile);
			try {
				ServiceLoader<Persistence> serviceLoader = ServiceLoader.load(Persistence.class);
				for (Persistence foundPersistence : serviceLoader) {
					if(foundPersistence.getClass().isInstance(FallbackPersistence.class)){
						continue;
					}
					try {
						String foundPersistenceClassName = foundPersistence.getClass().getSimpleName();
						logger.debug("Testing {}", foundPersistenceClassName);
						PersistenceConfiguration configuration = PersistenceConfiguration.getPersistenceConfiguration(foundPersistenceClassName);
						foundPersistence.prepareConnection(configuration);
						/*
						 * Uncomment the following line of code if you want to try 
						 * to create a test UsageRecord before setting the
						 * foundPersistence as default
						 * 
						 * foundPersistence.accountWithFallback(TestUsageRecord.createTestServiceUsageRecord());
						 */
						persistence = foundPersistence;
						break;
					} catch (Exception e) {
						logger.debug(String.format("%s not initialized correctly. It will not be used", foundPersistence.getClass().getSimpleName()));
					}
				}
				if(persistence==null){
					persistence = fallbackPersistence;
				}
			} catch(Exception e){
				logger.error("Unable to instance a Persistence Implementation. Using fallback as default", 
						e.getCause());
				persistence = fallbackPersistence;
			}
			persistence.setAggregationScheduler(AggregationScheduler.getInstance());
			persistence.setFallback(fallbackPersistence);
			persistences.put(scope, persistence);
		}
		
		return persistence;
	}
	
}
