package org.gcube.application.geoportalcommon;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Arrays;

import org.gcube.application.geoportalcommon.shared.GeoNaDataViewerProfile;
import org.gcube.application.geoportalcommon.shared.GeoNaItemRef;
import org.gcube.application.geoportalcommon.shared.PublicLink;
import org.gcube.portlets.user.urlshortener.UrlShortener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * The Class GeoportalCommon.
 *
 * @author Francesco Mangiacrapa at ISTI-CNR (francesco.mangiacrapa@isti.cnr.it)
 * 
 * Dec 1, 2020
 */
public class GeoportalCommon {

	
	/** The Constant LOG. */
	private static final Logger LOG = LoggerFactory.getLogger(GeoportalCommon.class);

	/**
	 * Gets the public links for.
	 *
	 * @param item the item
	 * @return the public links for
	 * @throws Exception the exception
	 */
	public GeoNaItemRef getPublicLinksFor(GeoNaItemRef item) throws Exception {
		LOG.info("getPublicLinksFor called for: " + item);

		try {

			if (item == null)
				throw new Exception("Bad request, the item is null");

			if (item.getItemId() == null)
				throw new Exception("Bad request, the item id is null");

			if (item.getItemType() == null)
				throw new Exception("Bad request, the item type is null");

			GeoNaDataViewerProfile geonaDataProfile = getGeoNaDataViewProfile(GeoportalCommonConstants.GEOPORTAL_DATA_VIEWER_APP);

			// Restricted Link
			String link = String.format("%s?%s=%s&%s=%s", geonaDataProfile.getRestrictedPortletURL(),
					GeoportalCommonConstants.GET_GEONA_ITEM_ID, item.getItemId(),
					GeoportalCommonConstants.GET_GEONA_ITEM_TYPE, item.getItemType());

			String shortUrl = getShortUrl(link);
			item.setRestrictedLink(new PublicLink(link, shortUrl));

			// Open Link
			link = String.format("%s?%s=%s&%s=%s", geonaDataProfile.getOpenPortletURL(),
					GeoportalCommonConstants.GET_GEONA_ITEM_ID, item.getItemId(),
					GeoportalCommonConstants.GET_GEONA_ITEM_TYPE, item.getItemType());

			shortUrl = getShortUrl(link);
			item.setOpenLink(new PublicLink(link, shortUrl));

			LOG.info("returning: " + item);
			return item;

		} catch (Exception e) {
			LOG.error("Error on getPublicLinksFor for: " + item, e);
			throw new Exception("Share link not available for this item. Try later or contact the support. Error: "
					+ e.getMessage());
		}
	}

	/**
	 * Gets the GeoNaData Viewer Profile
	 *
	 * @param appID if null or empty uses the default appID that is {@link GeoportalCommonConstants#GEOPORTAL_DATA_VIEWER_APP}
	 * @return the GeoNaData Viewer Profile
	 * @throws Exception the exception
	 */
	public GeoNaDataViewerProfile getGeoNaDataViewProfile(String appID) throws Exception {
		LOG.info("getGeoNaDataViewProfile called for: "+appID);
		
		if(appID==null || appID.isEmpty())
			appID = GeoportalCommonConstants.GEOPORTAL_DATA_VIEWER_APP;
		
		LOG.info("using AppId: "+appID);
		GeoNaDataViewerProfileReader gdvp = new GeoNaDataViewerProfileReader(appID);
		GeoNaDataViewerProfile profile = gdvp.readProfileFromInfrastrucure();
		LOG.info("Returning profile: " + profile);
		return profile;
	}


	/**
	 * Gets the short url.
	 *
	 * @param longUrl the long url
	 * @return the short url
	 * @throws Exception the exception
	 */
	public String getShortUrl(String longUrl) throws Exception {
		LOG.info("getShortUrl called for " + longUrl);

		if (longUrl == null)
			return longUrl;

		UrlShortener shortener = new UrlShortener();

		try {

			if (shortener != null && shortener.isAvailable()) {
				String toShort = longUrl;
				String[] splitted = toShort.split("\\?");
				LOG.debug("Splitted long URL is: " + Arrays.asList(splitted));
				String link = toShort;
				if (splitted.length > 1) {
					LOG.debug("Query string detected, encoding it...");
					String encodedQuery = splitted[1];
					try {
						encodedQuery = URLEncoder.encode(splitted[1], "UTF-8");
					} catch (UnsupportedEncodingException e) {
						e.printStackTrace();

					}
					LOG.debug("encodedQuery is: " + encodedQuery);
					link = String.format("%s?%s", splitted[0], encodedQuery);
				}
				return shortener.shorten(link);
			}
			return longUrl;

		} catch (Exception e) {
			LOG.error("Error on shortening the longURL " + longUrl, e);
			return null;
		}
	}
}
