package org.gcube.application.geoportal.model.concessioni;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.DiscriminatorValue;
import javax.persistence.ElementCollection;
import javax.persistence.Entity;
import javax.persistence.OneToMany;
import javax.persistence.OneToOne;

import org.gcube.application.geoportal.model.AccessPolicy;
import org.gcube.application.geoportal.model.Record;
import org.gcube.application.geoportal.model.RecordType;
import org.gcube.application.geoportal.model.content.OtherContent;
import org.gcube.application.geoportal.model.content.UploadedImage;
import org.gcube.application.geoportal.model.gis.BBOX;
import org.gcube.application.geoportal.model.report.Check;
import org.gcube.application.geoportal.model.report.ConstraintCheck;
import org.gcube.application.geoportal.model.report.ValidationReport;
import org.gcube.application.geoportal.model.report.ValidationReport.ValidationStatus;
import org.gcube.application.geoportal.utils.CollectionsUtils;
import org.gcube.application.geoportal.utils.Serialization;
import org.hibernate.annotations.LazyCollection;
import org.hibernate.annotations.LazyCollectionOption;
import org.json.JSONObject;

import com.fasterxml.jackson.core.JsonProcessingException;

import lombok.Getter;
import lombok.Setter;
import lombok.ToString;

@Getter
@Setter
@ToString(callSuper=true)

@Entity
@DiscriminatorValue("CONCESSIONE")
public class Concessione extends Record{

	
	//Introduzione (descrizione del progetto)
	private String introduzione;

	//Descrizione del contenuto
	private String descrizioneContenuto;
	
	//Autori	
	@LazyCollection(LazyCollectionOption.FALSE)
	@ElementCollection(targetClass=String.class)
	private List<String> authors;
	
	//Soggetto che materialmente invia i dati.	
	private String contributore;

	//Indicare il nome del titolare/i dei dati contenuti nel dataset e/o per sue specifiche parti.
	@LazyCollection(LazyCollectionOption.FALSE)
	@ElementCollection(targetClass=String.class)
	private List<String> titolari;
	
	private String responsabile;
	private String editore;
	
	@LazyCollection(LazyCollectionOption.FALSE)
	@ElementCollection(targetClass=String.class)
	private List<String> fontiFinanziamento;
	
	//Research Excavation; Archaeology  (valori di default)
	@LazyCollection(LazyCollectionOption.FALSE)
	@ElementCollection(targetClass=String.class)
	private List<String> soggetto;

	
	//Referenze bibliografiche, DOI (se esistenti) di risorse correlate all’indagine in oggetto
	@LazyCollection(LazyCollectionOption.FALSE)
	@ElementCollection(targetClass=String.class)
	private List<String> risorseCorrelate;

	
	private LocalDateTime dataInizioProgetto;
	private LocalDateTime dataFineProgetto;
	
	private String titolareLicenza;
	private String titolareCopyright;
	
	@LazyCollection(LazyCollectionOption.FALSE)
	@ElementCollection(targetClass=String.class)
	private List<String> paroleChiaveLibere;
	
	@LazyCollection(LazyCollectionOption.FALSE)
	@ElementCollection(targetClass=String.class)
	private List<String> paroleChiaveICCD;
	
	
	private Double centroidLat;
	private Double centroidLong;
	
	@LazyCollection(LazyCollectionOption.FALSE)
	@OneToOne(cascade = CascadeType.ALL)	
    private RelazioneScavo relazioneScavo;
	
	@LazyCollection(LazyCollectionOption.FALSE)
	@OneToMany(mappedBy = "record", cascade = CascadeType.ALL)
    private List<UploadedImage> immaginiRappresentative;
	
	@LazyCollection(LazyCollectionOption.FALSE)
	@OneToOne(cascade = CascadeType.ALL)
    private LayerConcessione posizionamentoScavo;
	
	@LazyCollection(LazyCollectionOption.FALSE)
	@OneToMany(mappedBy = "record", cascade = CascadeType.ALL)
    private List<LayerConcessione> pianteFineScavo;
	
	@LazyCollection(LazyCollectionOption.FALSE)
	@OneToMany(mappedBy = "record", cascade = CascadeType.ALL)
	private List<OtherContent> genericContent;
	
	
	public Concessione() {
		super();
		setRecordType(RecordType.CONCESSIONE);
	}
		
	
//	@Override
//	public String asJson() throws JsonProcessingException {
//		// Detaching Content ...		
//		if(relazioneScavo!=null) {
//			relazioneScavo.setRecord(null);
//			relazioneScavo.detachContent();
//		}
//		if(immaginiRappresentative!=null)
//			for(UploadedImage img:immaginiRappresentative) {
//				img.setRecord(null);
//				img.detachContent();
//			}
//		if(posizionamentoScavo!=null) {
//			posizionamentoScavo.setRecord(null);
//			posizionamentoScavo.detachContent();
//		}
//		if(pianteFineScavo!=null)
//			for(LayerConcessione l:pianteFineScavo) {
//				l.setRecord(null);
//				l.detachContent();
//			}
//		if(genericContent!=null)
//			for(OtherContent o:genericContent) {
//				o.setRecord(null);
//				o.detachContent();
//			}
//		
//		try {
//			return super.asJson();
//		}finally {
//			//Reattaching content
//			if(relazioneScavo!=null) {
//				relazioneScavo.setRecord(this);
//				relazioneScavo.reattachContent();
//			}
//			if(immaginiRappresentative!=null)
//				for(UploadedImage img:immaginiRappresentative) {
//					img.setRecord(this);
//					img.reattachContent();
//				}
//			if(posizionamentoScavo!=null) {
//				posizionamentoScavo.setRecord(this);
//				posizionamentoScavo.reattachContent();
//			}
//			if(pianteFineScavo!=null)
//				for(LayerConcessione l:pianteFineScavo) {
//					l.setRecord(this);		
//					l.reattachContent();
//				}
//			if(genericContent!=null)
//				for(OtherContent o:genericContent) {
//					o.setRecord(this);
//					o.reattachContent();
//				}
//		}
//	}
	
	@Override
	public ValidationReport validate() {		
		ValidationReport validator= super.validate();
		
		validator.setObjectName("Concessione");
		
		setPolicy(AccessPolicy.OPEN);
		
		
		
		validator.checkMandatory(authors, "Lista Autori");
		if(validator.checkMandatory(centroidLat, "Latitudine"))
			if(centroidLat>90||centroidLat<-90) validator.addMessage(ValidationStatus.ERROR, "Latitudine non valida : "+centroidLat);
		
		if(validator.checkMandatory(centroidLong, "Longitudine"))
			if(centroidLong>180||centroidLong<-180) validator.addMessage(ValidationStatus.ERROR, "Longitudine non valida : "+centroidLong);
		
		validator.checkMandatory(contributore, "Contributore");
		if(validator.checkMandatory(dataFineProgetto, "Data Fine Progetto") &&
				validator.checkMandatory(dataInizioProgetto, "Data Inizio Progetto"))
			if(dataFineProgetto.isBefore(dataInizioProgetto)) validator.addMessage(ValidationStatus.ERROR, "Data Fine Progetto non può esser prima di Data Inizio Progetto.");
		
		validator.checkMandatory(descrizioneContenuto, "Descrizione contenuto");
		validator.checkMandatory(editore, "Editore");
		validator.checkMandatory(fontiFinanziamento, "Fonti Finanaziamento");
		validator.checkMandatory(introduzione, "Introduzione");
		validator.checkMandatory(paroleChiaveICCD, "Parole chiave ICCD");
		validator.checkMandatory(paroleChiaveLibere, "Parole chiave libere");
		
		validator.checkMandatory(responsabile,"Responsabile");
	
		
		
		validator.checkMandatory(titolareCopyright, "Titolare Copyright");
		validator.checkMandatory(titolareLicenza,"Titolare licenza");
		validator.checkMandatory(titolari, "Titolari");
		
		
		
		if(validator.checkMandatory(relazioneScavo, "Relazione scavo")) {
			
			
			
			validator.addChild(relazioneScavo.validateForInsertion());
		}

//		if(immaginiRappresentative!=null)
//		
//			for(UploadedImage img : immaginiRappresentative) {
//				validator.setDefault(img.getSoggetto(),getSoggetto());
//				validator.setDefault(img.getCreationTime(),getCreationTime());
//				validator.setDefault(img.getPolicy(), getPolicy());
//				validator.setDefault(img.getLicenzaID(), getLicenzaID());
//
//				validator.addChild(img.validateForInsertion());
//			}
//		
		
		if(validator.checkMandatory(posizionamentoScavo, "Posizionamento scavo")) {
			
			ValidationReport posReport=posizionamentoScavo.validateForInsertion();
			posReport.setObjectName("Posizionamento scavo");
			validator.addChild(posReport);
		}

		
		if(genericContent!=null)
			for(OtherContent content:genericContent)
				validator.addChild(content.validateForInsertion());
		
		if(validator.checkMandatory(pianteFineScavo,"Piante fine scavo"))
			for(LayerConcessione l:pianteFineScavo) {
				validator.addChild(l.validateForInsertion());
			}
						
				
				
		
		
		return validator;
	}
	
	
	@Override
	public void setDefaults() {		
		super.setDefaults();
		setSoggetto(ConstraintCheck.defaultFor(soggetto, Arrays.asList(new String[] {"Research Excavation","Archaeology"}))
				.addChecks(Check.collectionSizeMin(2)).evaluate());
		
		
		
		if(relazioneScavo!=null) {
			relazioneScavo.setTitolo(ConstraintCheck.defaultFor(relazioneScavo.getTitolo(),getNome()+" relazione di scavo").evaluate());
			relazioneScavo.setSoggetto(ConstraintCheck.defaultFor(relazioneScavo.getSoggetto(),getSoggetto()).evaluate());
			relazioneScavo.setCreationTime(ConstraintCheck.defaultFor(relazioneScavo.getCreationTime(),getCreationTime()).evaluate());
			relazioneScavo.setLicenseID(getLicenzaID());
			relazioneScavo.setPolicy(getPolicy());
		}


		if(immaginiRappresentative!=null)
			for(UploadedImage img : immaginiRappresentative) {
				img.setSoggetto(ConstraintCheck.defaultFor(img.getSoggetto(),getSoggetto()).evaluate());
				img.setCreationTime(ConstraintCheck.defaultFor(img.getCreationTime(),getCreationTime()).evaluate());
				img.setPolicy(ConstraintCheck.defaultFor(img.getPolicy(),getPolicy()).evaluate());
				img.setLicenseID(ConstraintCheck.defaultFor(img.getLicenseID(),getLicenzaID()).evaluate());				
			}
		
		
		if(posizionamentoScavo!=null) {
			posizionamentoScavo.setTitolo(ConstraintCheck.defaultFor(posizionamentoScavo.getTitolo(), getNome()+" posizionamento scavo").evaluate());
			posizionamentoScavo.setAbstractSection(
					ConstraintCheck.defaultFor(posizionamentoScavo.getAbstractSection(),"Posizionamento topografico georeferenziato dell’area interessata dalle indagini").evaluate());
			
			posizionamentoScavo.setTopicCategory(ConstraintCheck.defaultFor(posizionamentoScavo.getTopicCategory(), "Society").evaluate());
			posizionamentoScavo.setSubTopic(ConstraintCheck.defaultFor(posizionamentoScavo.getSubTopic(), "Archeology").evaluate());

			posizionamentoScavo.setParoleChiaveLibere(ConstraintCheck.defaultFor(posizionamentoScavo.getParoleChiaveLibere(),getParoleChiaveLibere()).evaluate());
			posizionamentoScavo.setParoleChiaveICCD(ConstraintCheck.defaultFor(posizionamentoScavo.getParoleChiaveICCD(),getParoleChiaveICCD()).evaluate());
			
			//TODO Evaluate
			posizionamentoScavo.setBbox(ConstraintCheck.defaultFor(posizionamentoScavo.getBbox(), BBOX.WORLD_EXTENT).evaluate());
			
			posizionamentoScavo.setPolicy(ConstraintCheck.defaultFor(posizionamentoScavo.getPolicy(),getPolicy()).evaluate());;
			posizionamentoScavo.setLicenseID("CC-BY");
			posizionamentoScavo.setResponsabile(getResponsabile());
			posizionamentoScavo.setCreationTime(ConstraintCheck.defaultFor(posizionamentoScavo.getCreationTime(), getCreationTime()).evaluate());
			
			
		}

		
		
		if(pianteFineScavo!=null)
			for(LayerConcessione l:pianteFineScavo) {
				
				l.setTitolo(ConstraintCheck.defaultFor(l.getTitolo(), getNome()+" pianta fine scavo").evaluate());
				l.setAbstractSection(
						ConstraintCheck.defaultFor(l.getAbstractSection(),"Planimetria georeferenziata dell'area indagata al termine delle attività").evaluate());
				
				l.setTopicCategory(ConstraintCheck.defaultFor(l.getTopicCategory(), "Society").evaluate());
				l.setSubTopic(ConstraintCheck.defaultFor(l.getSubTopic(), "Archeology").evaluate());

				l.setParoleChiaveLibere(ConstraintCheck.defaultFor(l.getParoleChiaveLibere(),getParoleChiaveLibere()).evaluate());
				l.setParoleChiaveICCD(ConstraintCheck.defaultFor(l.getParoleChiaveICCD(),getParoleChiaveICCD()).evaluate());
				
				//TODO Evaluate
				l.setBbox(ConstraintCheck.defaultFor(l.getBbox(), BBOX.WORLD_EXTENT).evaluate());
				
				l.setPolicy(ConstraintCheck.defaultFor(l.getPolicy(),getPolicy()).evaluate());;
				l.setLicenseID(ConstraintCheck.defaultFor(l.getLicenseID(), "CC-BY").evaluate());
				l.setResponsabile(getResponsabile());
				l.setCreationTime(ConstraintCheck.defaultFor(l.getCreationTime(), getCreationTime()).evaluate());
			}
						
		
		
		
	}
	
//	@Override
//	public int hashCode() {
//		final int prime = 31;
//		int result = 1;
////		result = prime * result + ((authors == null) ? 0 : authors.hashCode());
//		result = prime * result + CollectionsUtils.hashCode(authors);
////		result = prime * result + ((fontiFinanaziamento == null) ? 0 : fontiFinanaziamento.hashCode());
//		result = prime * result + CollectionsUtils.hashCode(fontiFinanziamento);
////		result = prime * result + ((immaginiRappresentative == null) ? 0 : immaginiRappresentative.hashCode());
//		result = prime * result + CollectionsUtils.hashCode(immaginiRappresentative);
////		result = prime * result + ((paroleChiaveICCD == null) ? 0 : paroleChiaveICCD.hashCode());
//		result = prime * result + CollectionsUtils.hashCode(paroleChiaveICCD);
////		result = prime * result + ((paroleChiaveLibere == null) ? 0 : paroleChiaveLibere.hashCode());
//		result = prime * result + CollectionsUtils.hashCode(paroleChiaveLibere);
////		result = prime * result + ((pianteFineScavo == null) ? 0 : pianteFineScavo.hashCode());
//		result = prime * result + CollectionsUtils.hashCode(pianteFineScavo);
////		result = prime * result + ((risorseCorrelate == null) ? 0 : risorseCorrelate.hashCode());
//		result = prime * result + CollectionsUtils.hashCode(risorseCorrelate);
////		result = prime * result + ((soggetto == null) ? 0 : soggetto.hashCode());
//		result = prime * result + CollectionsUtils.hashCode(soggetto);		
////		result = prime * result + ((titolari == null) ? 0 : titolari.hashCode());
//		result = prime * result + CollectionsUtils.hashCode(titolari);
//		
//		result = prime * result + ((centroidLat == null) ? 0 : centroidLat.hashCode());
//		result = prime * result + ((centroidLong == null) ? 0 : centroidLong.hashCode());
//		result = prime * result + ((contributore == null) ? 0 : contributore.hashCode());
//		result = prime * result + ((dataFineProgetto == null) ? 0 : dataFineProgetto.hashCode());
//		result = prime * result + ((dataInizioProgetto == null) ? 0 : dataInizioProgetto.hashCode());
//		result = prime * result + ((descrizioneContenuto == null) ? 0 : descrizioneContenuto.hashCode());
//		result = prime * result + ((editore == null) ? 0 : editore.hashCode());
//		result = prime * result + ((genericContent == null) ? 0 : genericContent.hashCode());
//		result = prime * result + ((introduzione == null) ? 0 : introduzione.hashCode());
//		
//		result = prime * result + ((posizionamentoScavo == null) ? 0 : posizionamentoScavo.hashCode());
//		result = prime * result + ((relazioneScavo == null) ? 0 : relazioneScavo.hashCode());
//		result = prime * result + ((responsabile == null) ? 0 : responsabile.hashCode());
//		result = prime * result + ((titolareCopyright == null) ? 0 : titolareCopyright.hashCode());
//		result = prime * result + ((titolareLicenza == null) ? 0 : titolareLicenza.hashCode());
//		return result;
//	}


//	@Override
//	public boolean equals(Object obj) {
//		if (this == obj)
//			return true;
//		if (obj == null)
//			return false;
//		if (getClass() != obj.getClass())
//			return false;
//		Concessione other = (Concessione) obj;
////		if (authors == null) {
////			if (other.authors != null)
////				return false;
////		} else if (!authors.equals(other.authors))
////			return false;
//		if(!CollectionsUtils.equalsCollections(authors, other.authors)) return false;
//		
////		if (fontiFinanaziamento == null) {
////		if (other.fontiFinanaziamento != null)
////			return false;
////	} else if (!fontiFinanaziamento.equals(other.fontiFinanaziamento))
////		return false;
//	if (!CollectionsUtils.equalsCollections(fontiFinanziamento, other.fontiFinanziamento)) return false;
//	
////	if (immaginiRappresentative == null) {
////		if (other.immaginiRappresentative != null)
////			return false;
////	} else if (!immaginiRappresentative.equals(other.immaginiRappresentative))
////		return false;
//	if (!CollectionsUtils.equalsCollections(immaginiRappresentative, other.immaginiRappresentative)) return false;
//	
////	if (paroleChiaveICCD == null) {
////		if (other.paroleChiaveICCD != null)
////			return false;
////	} else if (!paroleChiaveICCD.equals(other.paroleChiaveICCD))
////		return false;
//	if (!CollectionsUtils.equalsCollections(paroleChiaveICCD, other.paroleChiaveICCD)) return false;
//	
////	if (paroleChiaveLibere == null) {
////		if (other.paroleChiaveLibere != null)
////			return false;
////	} else if (!paroleChiaveLibere.equals(other.paroleChiaveLibere))
////		return false;
//	if (!CollectionsUtils.equalsCollections(paroleChiaveLibere, other.paroleChiaveLibere)) return false;
//	
////	if (piantaFineScavo == null) {
////		if (other.piantaFineScavo != null)
////			return false;
////	} else if (!piantaFineScavo.equals(other.piantaFineScavo))
////		return false;
//	if (!CollectionsUtils.equalsCollections(pianteFineScavo, other.pianteFineScavo)) return false;
//	
////	if (risorseCorrelate == null) {
////		if (other.risorseCorrelate != null)
////			return false;
////	} else if (!risorseCorrelate.equals(other.risorseCorrelate))
////		return false;
//	if (!CollectionsUtils.equalsCollections(risorseCorrelate, other.risorseCorrelate)) return false;
//	
////	if (soggetto == null) {
////		if (other.soggetto != null)
////			return false;
////	} else if (!soggetto.equals(other.soggetto))
////		return false;
//	if (!CollectionsUtils.equalsCollections(soggetto, other.soggetto)) return false;
//	
////	if (titolari == null) {
////		if (other.titolari != null)
////			return false;
////	} else if (!titolari.equals(other.titolari))
////		return false;
//	if (!CollectionsUtils.equalsCollections(titolari, other.titolari)) return false;
//		
//		
//		if (centroidLat == null) {
//			if (other.centroidLat != null)
//				return false;
//		} else if (!centroidLat.equals(other.centroidLat))
//			return false;
//		if (centroidLong == null) {
//			if (other.centroidLong != null)
//				return false;
//		} else if (!centroidLong.equals(other.centroidLong))
//			return false;
//		if (contributore == null) {
//			if (other.contributore != null)
//				return false;
//		} else if (!contributore.equals(other.contributore))
//			return false;
//		if (dataFineProgetto == null) {
//			if (other.dataFineProgetto != null)
//				return false;
//		} else if (!dataFineProgetto.equals(other.dataFineProgetto))
//			return false;
//		if (dataInizioProgetto == null) {
//			if (other.dataInizioProgetto != null)
//				return false;
//		} else if (!dataInizioProgetto.equals(other.dataInizioProgetto))
//			return false;
//		if (descrizioneContenuto == null) {
//			if (other.descrizioneContenuto != null)
//				return false;
//		} else if (!descrizioneContenuto.equals(other.descrizioneContenuto))
//			return false;
//		if (editore == null) {
//			if (other.editore != null)
//				return false;
//		} else if (!editore.equals(other.editore))
//			return false;
//		if (genericContent == null) {
//			if (other.genericContent != null)
//				return false;
//		} else if (!genericContent.equals(other.genericContent))
//			return false;
//		if (introduzione == null) {
//			if (other.introduzione != null)
//				return false;
//		} else if (!introduzione.equals(other.introduzione))
//			return false;
//		
//		if (posizionamentoScavo == null) {
//			if (other.posizionamentoScavo != null)
//				return false;
//		} else if (!posizionamentoScavo.equals(other.posizionamentoScavo))
//			return false;
//		if (relazioneScavo == null) {
//			if (other.relazioneScavo != null)
//				return false;
//		} else if (!relazioneScavo.equals(other.relazioneScavo))
//			return false;
//		if (responsabile == null) {
//			if (other.responsabile != null)
//				return false;
//		} else if (!responsabile.equals(other.responsabile))
//			return false;
//		if (titolareCopyright == null) {
//			if (other.titolareCopyright != null)
//				return false;
//		} else if (!titolareCopyright.equals(other.titolareCopyright))
//			return false;
//		if (titolareLicenza == null) {
//			if (other.titolareLicenza != null)
//				return false;
//		} else if (!titolareLicenza.equals(other.titolareLicenza))
//			return false;
//		
//		return true;
//	}
//	
	
	
}
