package org.gcube.applicationsupportlayer.social;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import org.gcube.application.framework.core.session.ASLSession;
import org.gcube.applicationsupportlayer.social.mailing.EmailPlugin;
import org.gcube.portal.databook.shared.ApplicationProfile;
import org.gcube.portal.databook.shared.Notification;
import org.gcube.portal.databook.shared.NotificationChannelType;
import org.gcube.portal.databook.shared.NotificationType;
import org.gcube.portal.databook.shared.RunningJob;
import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.home.workspace.WorkspaceFolder;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItem;
import org.gcube.vomanagement.usermanagement.UserManager;
import org.gcube.vomanagement.usermanagement.exception.UserManagementPortalException;
import org.gcube.vomanagement.usermanagement.exception.UserManagementSystemException;
import org.gcube.vomanagement.usermanagement.exception.UserRetrievalFault;
import org.gcube.vomanagement.usermanagement.impl.liferay.LiferayUserManager;
import org.gcube.vomanagement.usermanagement.model.UserModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 
 * @author Massimiliano Assante, ISTI-CNR
 *
 * use to notify users from within your application 
 */
public class ApplicationNotificationsManager extends SocialPortalBridge implements NotificationsManager {
	private static final Logger _log = LoggerFactory.getLogger(ApplicationNotificationsManager.class);
	
	/**
	 * Use this constructor if you do not need notifications to point back to your applications
	 * @param aslSession the ASLSession instance
	 */
	public ApplicationNotificationsManager(ASLSession session) {
		super(session);
		_log.warn("Asked for Simple Notification (without redirect to creator)");
	}
	/**
	 * Use this constructor if you do need notifications to point back to your applications, 
	 * make sure you create your application profile on the infrastructure.
	 *    
	 * @see http://gcube.wiki.gcube-system.org/gcube/index.php/Social_Networking_Library#Create_Your_Application_Profile
	 * 
	 * @param aslSession the ASLSession instance
	 * @param portletClassName your portlet class name will be used ad unique identifier for your applicationProfile
	 */
	public ApplicationNotificationsManager(ASLSession session, String portletClassName) {
		super(session, portletClassName);
	}
	/**
	 * actually save the notification to the store
	 * @param notification2Save the notification instance to save
	 * @return true if the notification was sent ok
	 */
	private boolean saveNotification(Notification notification2Save) {
		_log.trace("Trying to send notification to: " + notification2Save.getUserid() +  " Type: " + notification2Save.getType());
		if (notification2Save.getSenderid().compareTo(notification2Save.getUserid()) == 0) {
			_log.trace("Sender and Receiver are the same " + notification2Save.getUserid() +  " Notification Stopped");
			return true;  //I'm not sending notifications to the person who triggered it, pretend I sent it though
		}
		List<NotificationChannelType> channels = null;
		try {
			channels = getStoreInstance().getUserNotificationChannels(notification2Save.getUserid(), notification2Save.getType());
		} catch (Exception e) {
			e.printStackTrace();
		}
		boolean result = false;
		if (channels.contains(NotificationChannelType.PORTAL)) {
			result = getStoreInstance().saveNotification(notification2Save);
			if (result)
				_log.trace("Notification Saved Successfully! ");
			else
				_log.error("Error While trying to save Notification");
		}
		if (channels.contains(NotificationChannelType.EMAIL))
			EmailPlugin.sendNotification(notification2Save, aslSession.getGroupName());

		if (channels.isEmpty()) {
			_log.info("Notification was not needed as "+ notification2Save.getUserid()  +" decided not to be notified for " + notification2Save.getType());
			result = true;
		}
		return result;

	}
	/**
	 * return the url of the application if exists in the profile
	 * @return .
	 */
	private String getApplicationUrl() {
		if (applicationProfile != null && applicationProfile.getUrl() != null)
			return applicationProfile.getUrl();
		else return "";
	}
	/**
	 * {@inheritDoc}
	 * @throws InternalErrorException 
	 */
	@Override
	public boolean notifyFolderSharing(String userIdToNotify, WorkspaceFolder sharedFolder) throws InternalErrorException {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_FOLDER_SHARE, 
				userIdToNotify, //user no notify
				sharedFolder.getId(),  //the 
				new Date(),
				getApplicationUrl()+"?itemid="+sharedFolder.getId(), 
				"shared the workspace folder "+ sharedFolder.getName() +" with you",
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 * @throws InternalErrorException 
	 */
	@Override
	public boolean notifyFolderRenaming(String userIdToNotify, String previousName, String newName, String renamedFolderId) throws InternalErrorException {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_FOLDER_RENAMED, 
				userIdToNotify, //user no notify
				renamedFolderId,  //the 
				new Date(),
				getApplicationUrl()+"?itemid="+renamedFolderId, 
				"renamed your shared folder "+ previousName +" as " + newName,
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 * @throws UserManagementPortalException 
	 * @throws UserRetrievalFault 
	 * @throws UserManagementSystemException 
	 */
	@Override
	public boolean notifyFolderAddedUser(String userIdToNotify, WorkspaceFolder sharedFolder, String newAddedUserId) throws InternalErrorException, UserManagementSystemException, UserRetrievalFault, UserManagementPortalException {
		UserManager um = new LiferayUserManager();
		UserModel user = um.getUserByScreenName(newAddedUserId);
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_FOLDER_ADDEDUSER, 
				userIdToNotify, //user no notify
				sharedFolder.getId(),  //the 
				new Date(),
				getApplicationUrl()+"?itemid="+sharedFolder.getId(),
				"added "+ user.getFullname() +" to your workspace shared folder "+ sharedFolder.getName(),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 * @throws UserManagementPortalException 
	 * @throws UserRetrievalFault 
	 * @throws UserManagementSystemException 
	 */
	@Override
	public boolean notifyFolderAddedUsers(String userIdToNotify, WorkspaceFolder sharedFolder, List<String> newAddedUserIds) throws InternalErrorException, UserManagementSystemException, UserRetrievalFault, UserManagementPortalException  {
		if (newAddedUserIds != null && newAddedUserIds.size() > 0) {
			if (newAddedUserIds.size() == 1)
				return notifyFolderAddedUser(userIdToNotify, sharedFolder, newAddedUserIds.get(0));
			StringBuilder addedUsersFullNames = new StringBuilder();
			UserManager um = new LiferayUserManager();
			for (String userId : newAddedUserIds) 
				addedUsersFullNames.append(um.getUserByScreenName(userId).getFullname()).append(" ");

			Notification not = new Notification(
					UUID.randomUUID().toString(), 
					NotificationType.WP_FOLDER_ADDEDUSER, 
					userIdToNotify, //user no notify
					sharedFolder.getId(),  //the 
					new Date(),
					getApplicationUrl()+"?itemid="+sharedFolder.getId(),
					"added "+ addedUsersFullNames +" to your workspace shared folder "+ sharedFolder.getName(),
					false, 
					aslSession.getUsername(),
					aslSession.getUserFullName(), 
					aslSession.getUserAvatarId());

			return saveNotification(not);
		}

		return false;
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyFolderRemovedUser(String userIdToNotify, WorkspaceFolder sharedFolder) throws InternalErrorException, UserManagementSystemException, UserRetrievalFault, UserManagementPortalException {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_FOLDER_REMOVEDUSER, 
				userIdToNotify, //user no notify
				sharedFolder.getId(),  //the 
				new Date(),
				getApplicationUrl()+"?itemid="+sharedFolder.getId(), 
				"unshared his shared folder "+ sharedFolder.getName() + " from your workspace",
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 * @throws InternalErrorException 
	 */
	@Override
	public boolean notifyAddedItem(String userIdToNotify, WorkspaceItem item, WorkspaceFolder sharedFolder) throws InternalErrorException {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_ITEM_NEW, 
				userIdToNotify, //user no notify
				item.getId(),  //the 
				new Date(),
				getApplicationUrl()+"?itemid="+item.getParent().getId(), 
				"added "+ item.getName() +" to your workspace shared folder "+ item.getPath().substring(0,item.getPath().lastIndexOf('/')),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 * @throws InternalErrorException 
	 */
	@Override
	public boolean notifyMovedItem(String userIdToNotify,	WorkspaceItem item, WorkspaceFolder sharedFolder) throws InternalErrorException {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_ITEM_DELETE, 
				userIdToNotify, //user no notify
				item.getId(),  //the 
				new Date(),
				getApplicationUrl()+"?itemid="+sharedFolder.getId(), 
				"removed item "+ item.getName() +" from your workspace shared folder "+ sharedFolder.getName(),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 * @throws InternalErrorException 
	 */
	@Override
	public boolean notifyRemovedItem(String userIdToNotify, String itemName, WorkspaceFolder sharedFolder) throws InternalErrorException {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_ITEM_DELETE, 
				userIdToNotify, //user no notify
				sharedFolder.getId(),  //the 
				new Date(),
				getApplicationUrl()+"?itemid="+sharedFolder.getId(), 
				"deleted item "+ itemName +" from your workspace shared folder "+ sharedFolder.getName(),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 * @throws InternalErrorException 
	 */
	@Override
	public boolean notifyUpdatedItem(String userIdToNotify,	WorkspaceItem item, WorkspaceFolder sharedFolder) throws InternalErrorException {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_ITEM_UPDATED, 
				userIdToNotify, //user no notify
				item.getId(),  //the 
				new Date(),
				getApplicationUrl()+"?itemid="+item.getParent().getId(), 
				" updated "+ item.getName() +" to your workspace shared folder "+ item.getPath().substring(0,item.getPath().lastIndexOf('/')),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 * @throws InternalErrorException 
	 */
	@Override
	public boolean notifyItemRenaming(String userIdToNotify, String previousName, WorkspaceItem renamedItem) throws InternalErrorException {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.WP_ITEM_RENAMED, 
				userIdToNotify, //user no notify
				renamedItem.getId(),  //the 
				new Date(),
				getApplicationUrl()+"?itemid="+renamedItem.getParent().getId(), 
				"renamed " + previousName +" as " + renamedItem.getName() +" in your shared folder " + renamedItem.getParent().getName(),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyMessageReceived(String userIdToNotify,	String subject) {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.MESSAGE, 
				userIdToNotify, //user no notify
				"messageid_not_provided",  //the 
				new Date(),
				"/group/data-e-infrastructure-gateway/messages", 
				"sent you a message with subject: " + escapeHtml(subject),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyOwnCommentReply(String userIdToNotify, String feedid, String feedText) {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.OWN_COMMENT, 
				userIdToNotify, //user no notify
				feedid,  //the post 
				new Date(),
				"/group/data-e-infrastructure-gateway?oid="+feedid, 
				"commented on your post: " + escapeHtml(feedText),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());
		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyCommentReply(String userIdToNotify, String feedid, String commentText, String feedOwnerFullName) {
		String notificationText =  (aslSession.getUserFullName().compareTo(feedOwnerFullName) == 0) ? 
				"also commented on his post: " + escapeHtml(commentText)
				: 
					"also commented on " + feedOwnerFullName + "'s post: " + escapeHtml(commentText);
				Notification not = new Notification(
						UUID.randomUUID().toString(), 
						NotificationType.COMMENT, 
						userIdToNotify, //user no notify
						feedid,  //the post 
						new Date(),
						"/group/data-e-infrastructure-gateway?oid="+feedid, 
						notificationText,
						false, 
						aslSession.getUsername(),
						aslSession.getUserFullName(), 
						aslSession.getUserAvatarId());
				return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyCommentOnFavorite(String userIdToNotify, String feedid, String commentText) {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.COMMENT, 
				userIdToNotify, //user no notify
				feedid,  //the post 
				new Date(),
				"/group/data-e-infrastructure-gateway?oid="+feedid, 
				"commented on one of your favorite posts: " + escapeHtml(commentText),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());
		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyUserTag(String userIdToNotify, String feedid, String feedText) {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.MENTION, 
				userIdToNotify, //user no notify
				feedid,  //the post 
				new Date(),
				"/group/data-e-infrastructure-gateway?oid="+feedid, 
				"mentioned you in his post: " + escapeHtml(feedText),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());
		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyLikedFeed(String userIdToNotify, String feedid, String feedText) {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.LIKE, 
				userIdToNotify, //user no notify
				feedid,  //the post 
				new Date(),
				"/group/data-e-infrastructure-gateway?oid="+feedid, 
				"likes your post: " + escapeHtml(feedText),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());
		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyJobStatus(String userIdToNotify, ApplicationProfile executingJobApId, RunningJob job) {
		//TODO: missing implementation 
		return false;
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyDocumentWorkflowView(String userIdToNotify, String documentWorkflowId, String documentName) {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.DOCUMENT_WORKFLOW_VIEW, 
				userIdToNotify, //user no notify
				documentWorkflowId,  //the workflowid 
				new Date(),
				getApplicationUrl()+"?oid="+documentWorkflowId, 
				"viewed document workflow " + escapeHtml(documentName),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());
		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyDocumentWorkflowUpdate(String userIdToNotify, String documentWorkflowId, String documentName) {

		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.DOCUMENT_WORKFLOW_EDIT, 
				userIdToNotify, //user no notify
				documentWorkflowId,  //the workflowid 
				new Date(),
				getApplicationUrl()+"?oid="+documentWorkflowId, 
				"updated document workflow " + escapeHtml(documentName),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());
		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyDocumentWorkflowTaskRequest(String userIdToNotify,	String documentWorkflowId, String documentName, String assignedRoleName) {

		String notificationText = "in " + aslSession.getGroupName() + " you are requested to perform a new task in the Document Workflow titled: "
				+  escapeHtml(documentName) + ". Your role is: " + assignedRoleName;

		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.DOCUMENT_WORKFLOW_STEP_REQUEST_TASK, 
				userIdToNotify, //user no notify
				documentWorkflowId,  //the workflowid 
				new Date(),
				getApplicationUrl()+"?oid="+documentWorkflowId, 
				notificationText,
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);

	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyDocumentWorkflowUserForward(String userIdToNotify,	String documentWorkflowId, String documentName, String fromStepName, String toStepName){
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.DOCUMENT_WORKFLOW_USER_FORWARD_TO_OWNER, 
				userIdToNotify, //user no notify
				documentWorkflowId,  //the workflowid 
				new Date(),
				getApplicationUrl()+"?oid="+documentWorkflowId, 

				"forwarded the Document Workflow titled: " + escapeHtml(documentName)  +
				" from status \"" + fromStepName + "\" to status \"" + toStepName +"\". In " + aslSession.getGroupName(),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyDocumentWorkflowStepForwardComplete(String userIdToNotify, String documentWorkflowId, String documentName, String fromStepName, String toStepName) {
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.DOCUMENT_WORKFLOW_FORWARD_STEP_COMPLETED_OWNER, 
				userIdToNotify, //user no notify
				documentWorkflowId,  //the workflowid 
				new Date(),
				getApplicationUrl()+"?oid="+documentWorkflowId, 
				"has performed the last needed forward on a Document Workflow titled: " + escapeHtml(documentName) + ". " + 
						"Step \"" + fromStepName + "\" is now complete. The next step is \"" + toStepName +"\". In " + aslSession.getGroupName(),
						false, 
						aslSession.getUsername(),
						aslSession.getUserFullName(), 
						aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyDocumentWorkflowFirstStepRequest(String userIdToNotify, String documentWorkflowId,	String documentName, String assignedRole) {
		String notificationText = "involved you in the Document Workflow titled: " + escapeHtml(documentName) + ". " +
				"You are requested to perform a task. Your role is: " + assignedRole+"";
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.DOCUMENT_WORKFLOW_FIRST_STEP_REQUEST_INVOLVMENT, 
				userIdToNotify, //user no notify
				documentWorkflowId,  //the workflowid 
				new Date(),
				getApplicationUrl()+"?oid="+documentWorkflowId, 
				notificationText,
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyNewCalendarEvent(String userIdToNotify, String eventTitle, String eventType, Date startDate, Date endingDate) {
		SimpleDateFormat spf = new SimpleDateFormat("EEE dd MMMMM, yyyy");
		
		String endDateToDisplay="";
		if (endingDate != null) {
			endDateToDisplay = " to " + spf.format(endingDate);
		}
	
		StringBuilder notificationText = new StringBuilder();
		notificationText.append("added the following event in the <b>").append(aslSession.getGroupName()).append("</b> shared calendar: ") // has done something
		.append("<strong>").append(eventTitle).append("</strong><br />")
		.append("<br /><strong> Time:</strong> ").append(spf.format(startDate)).append(endDateToDisplay).append("<br />")
		.append("<br /><strong> Category: </strong> ").append(eventType);
		
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.CALENDAR_ADDED_EVENT, 
				userIdToNotify, //user no notify
				"",  //
				new Date(),
				getApplicationUrl(), 
				notificationText.toString(),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyEditedCalendarEvent(String userIdToNotify, String eventTitle, String eventType, Date startDate, Date endingDate) {
		SimpleDateFormat spf = new SimpleDateFormat("EEE dd MMMMM, yyyy");
		
		String endDateToDisplay="";
		if (endingDate != null) {
			endDateToDisplay = " to " + spf.format(endingDate);
		}
	
		StringBuilder notificationText = new StringBuilder();
		notificationText.append("edited the following event in the <b>").append(aslSession.getGroupName()).append("</b> shared calendar: ") // has done something
		.append("<strong>").append(eventTitle).append("</strong><br />")
		.append("<br /><strong> Time:</strong> ").append(spf.format(startDate)).append(endDateToDisplay).append("<br />")
		.append("<br /><strong> Category: </strong> ").append(eventType);
		
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.CALENDAR_UPDATED_EVENT, 
				userIdToNotify, //user no notify
				"",  //
				new Date(),
				getApplicationUrl(), 
				notificationText.toString(),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notifyDeletedCalendarEvent(String userIdToNotify, String eventTitle, String eventType, Date startDate, Date endingDate) {
		SimpleDateFormat spf = new SimpleDateFormat("EEE dd MMMMM, yyyy");
		
		String endDateToDisplay="";
		if (endingDate != null) {
			endDateToDisplay = " to " + spf.format(endingDate);
		}
	
		StringBuilder notificationText = new StringBuilder();
		notificationText.append("deleted the following event in the <b>").append(aslSession.getGroupName()).append("</b> shared calendar: ") // has done something
		.append("<strong>").append(eventTitle).append("</strong><br />")
		.append("<br /><strong> Was planned:</strong> ").append(spf.format(startDate)).append(endDateToDisplay).append("<br />")
		.append("<br /><strong> Category: </strong> ").append(eventType);
		
		Notification not = new Notification(
				UUID.randomUUID().toString(), 
				NotificationType.CALENDAR_DELETED_EVENT, 
				userIdToNotify, //user no notify
				"",  //
				new Date(),
				getApplicationUrl(), 
				notificationText.toString(),
				false, 
				aslSession.getUsername(),
				aslSession.getUserFullName(), 
				aslSession.getUserAvatarId());

		return saveNotification(not);
	}
}
