package org.gcube.common.gxhttp.request;

import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.util.Map;
import java.util.Objects;

import org.gcube.common.gxhttp.reference.GXConnection;
import org.gcube.common.gxhttp.reference.GXHTTP;
import org.gcube.common.gxhttp.reference.GXConnection.HTTPMETHOD;

/**
 * A context-aware request to a web application.
 * It supports sending streams through Put/Post requests.
 *  
 * @author Manuele Simi (ISTI CNR)
 * @author Luca Frosini (ISTI-CNR)
 *
 */
public class GXHTTPStreamRequest extends GXHTTPCommonRequest
 implements GXHTTP<InputStream,HttpURLConnection> {

	/**
	 * A new request.
	 */
	private GXHTTPStreamRequest(String address) {
		builder.connection = new GXConnection(address);
	}

	/**
	 * Creates a new request.
	 * 
	 * @param address
	 *            the address of the web app to call
	 * @return the request
	 */
	public static GXHTTPStreamRequest newRequest(String address) {
		return new GXHTTPStreamRequest(address);
	}
	
	/* (non-Javadoc)
	 * @see org.gcube.common.gxhttp.reference.GXHTTP#put(java.lang.Object)
	 */
	@Override
	public HttpURLConnection put(InputStream body) throws Exception {
		if (Objects.nonNull(body))
			builder.connection.addBodyAsStream(body);
		logger.trace("Sending a PUT request...");
		return builder.connection.send(HTTPMETHOD.PUT);
	}

	/* (non-Javadoc)
	 * @see org.gcube.common.gxhttp.reference.GXHTTP#post(java.lang.Object)
	 */
	@Override
	public HttpURLConnection post(InputStream body) throws Exception {
		logger.trace("Sending a POST request...");
		if (Objects.nonNull(body))
			builder.connection.addBodyAsStream(body);
		return builder.connection.send(HTTPMETHOD.POST);
	}

	/* (non-Javadoc)
	 * @see org.gcube.common.gxhttp.reference.GXHTTP#patch(java.lang.Object)
	 */
	@Override
	public HttpURLConnection patch(InputStream body) throws Exception {
		logger.trace("Sending a PATCH request...");
		if (Objects.nonNull(body))
			builder.connection.addBodyAsStream(body);
		return builder.connection.send(HTTPMETHOD.PATCH);
	}

	/**
	 * Sets the User-Agent for the request
	 *
	 * @param agent the agent
	 * @return the request
	 */
	public GXHTTPStreamRequest from(String agent) {
		builder.from(agent)	;
		return this;
	}
	
	/**
	 * Adds a path to the request URL
	 *
	 * @param path the path to add
	 * @return the request
	 * @throws UnsupportedEncodingException if the encoding is not supported
	 *
	 */
	public GXHTTPStreamRequest path(String path) throws UnsupportedEncodingException {
		builder.path(path);
		return this;
	}

	/**
	 * Adds a header to the request
	 *
	 * @param name the name of the header
	 * @param value the value of the header
	 * @return the request
	 */
	public GXHTTPStreamRequest header(String name, String value) {
		builder.header(name, value);
		return this;
	}

	/**
	 * Adds query parameters to the request
	 *
	 * @param queryParams the query parameters
	 * @return the request
	 * @throws UnsupportedEncodingException if the encoding is not supported
	 */
	public GXHTTPStreamRequest queryParams(Map<String, String> queryParams) throws UnsupportedEncodingException {
		builder.queryParams(queryParams);
		return this;
	}
}
