package org.gcube.common.homelibrary.jcr.repository;

import static org.gcube.resources.discovery.icclient.ICFactory.clientFor;
import static org.gcube.resources.discovery.icclient.ICFactory.queryFor;

import java.security.Principal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;
import javax.jcr.Value;
import javax.jcr.security.AccessControlList;
import javax.jcr.security.AccessControlManager;
import javax.jcr.security.AccessControlPolicy;
import javax.jcr.security.AccessControlPolicyIterator;
import javax.jcr.security.Privilege;

import org.apache.jackrabbit.rmi.repository.URLRemoteRepository;
import org.gcube.common.encryption.StringEncrypter;
import org.gcube.common.homelibrary.home.User;
import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.jcr.workspace.usermanager.JCRUserManager;
import org.gcube.common.homelibrary.jcr.workspace.util.Utils;
import org.gcube.common.resources.gcore.ServiceEndpoint;
import org.gcube.common.resources.gcore.ServiceEndpoint.AccessPoint;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.resources.discovery.client.api.DiscoveryClient;
import org.gcube.resources.discovery.client.queries.api.SimpleQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;



public class JCRRepository {

	public static final String HL_NAMESPACE					= "hl:";
	public static final String JCR_NAMESPACE				= "jcr:";
	public static final String REP_NAMESPACE				= "rep:";

	public static final String PATH_SEPARATOR 				= "/";
	public static final String ROOT_WEBDAV					= "/repository/default/";

	public static final String HOME_FOLDER 					= "Home";
	public static final String SHARED_FOLDER				= "Share";

	private static final String DOWNLOADS					= "Downloads";
	private static final String SMART_FOLDER 				= "Folders";
	private static final String GCUBE_FOLDER 				= "GCube";
	private static final String IN_BOX_FOLDER 				= "InBox";
	private static final String OUT_BOX_FOLDER				= "OutBox";
	private static final String HIDDEN_FOLDER				= "HiddenFolder";


	private static final String NT_FOLDER 					= "nt:folder";
	private static final String NT_HOME						= "nthl:home";
	public static final String 	NT_WORKSPACE_FOLDER 		= "nthl:workspaceItem";
	private static final String NT_ROOT_ITEM_SENT  			= "nthl:rootItemSentRequest";
	private static final String NT_ROOT_FOLDER_BULK_CREATOR = "nthl:rootFolderBulkCreator";

	private static final String SCOPES						= "hl:scopes";

	private static final String nameResource 				= "HomeLibraryRepository";



	private static Repository repository;

	//root level
//	private static String homeFolderId;
	private static String gcubeRootId;
	private static String sharedRootId;

	//home level
	private String userHomeId;
	private String smartFoldersId;
	private String downloadsId;
	private String inBoxId;
	private String outBoxId;
	private String hiddenFolderId;


	private static String user;
	private static String pass;
	public static String url;
	private static String webdavUrl;


	private static Logger logger = LoggerFactory.getLogger(JCRRepository.class);


	private static synchronized void initializeRepository() throws InternalErrorException {

		if(repository != null)
			return;


		String callerScope = ScopeProvider.instance.get();

		try {

			if (callerScope==null) throw new IllegalArgumentException("scope is null");

			String rootScope = Utils.getRootScope(callerScope);

			logger.debug("scope for repository creation is "+rootScope+" caller scope is "+callerScope);

			ScopeProvider.instance.set(rootScope);

			SimpleQuery query = queryFor(ServiceEndpoint.class);

			query.addCondition("$resource/Profile/Category/text() eq 'Database' and $resource/Profile/Name eq '"+ nameResource + "' ");

			DiscoveryClient<ServiceEndpoint> client = clientFor(ServiceEndpoint.class);

			List<ServiceEndpoint> resources = client.submit(query);

			if(resources.size() != 0) {	   
				try {
					ServiceEndpoint resource = resources.get(0);

					for (AccessPoint ap:resource.profile().accessPoints()) {

						if (ap.name().equals("JCR")) {

							url = ap.address();
							user = ap.username();						
							pass = StringEncrypter.getEncrypter().decrypt(ap.password());
						}
						else if (ap.name().equals("WebDav")) {								
							webdavUrl = ap.address();				
						}
					}
				} catch (Throwable e) {
					logger.error("error decrypting resource",e);
				}
			}

			//			user = "social.isti";
			//			pass="kidiprep1";
			//			url = "http://social.isti.cnr.it:9000/jackrabbit-webapp-patched-2.4.3";
			if (user==null || pass==null) throw new InternalErrorException("cannot discover password and username in scope "+callerScope);

			repository = new URLRemoteRepository(url + "/rmi");

			logger.debug("user is "+user+" password is null?"+(pass==null)+" and repository is null?"+(repository==null));

		} catch (Exception e) {
			throw new InternalErrorException(e);
		}finally{
			if (callerScope!=null)
				ScopeProvider.instance.set(callerScope);
		}

	}

	public static List<String> getHomeNames() throws RepositoryException{
		List<String> homes = new ArrayList<String>();
		Session session = null;
		try{
			try {
				session = getSession();
			} catch (InternalErrorException e) {
				logger.error("error gettin session",e);
				throw new RepositoryException(e);
			}
			Node home = (Node)session.getItem(PATH_SEPARATOR+HOME_FOLDER);
			NodeIterator it = home.getNodes();
			while (it.hasNext()){
				Node node = it.nextNode();
				if (node.getPrimaryNodeType().getName().equals(NT_HOME))
					homes.add(node.getName());
			}
		}finally{
			if (session != null)
				session.logout();
		}
		return homes;
	}



	//	private static void addUserToJCRUserManager(String userId, String userHome) {
	//
	//		GetMethod getMethod = null;
	//		try {
	//
	//			HttpClient httpClient = new HttpClient();            
	//			//			 System.out.println(url);
	//			getMethod =  new GetMethod(url + "/PortalUserManager?userId=" + userId + "&userHome=" +userHome);
	//			httpClient.executeMethod(getMethod);
	//
	//			logger.debug("User set with status code " + getMethod.getResponseBodyAsString());
	//
	//		} catch (Exception e) {
	//			logger.error("User set with error ", e);
	//		} finally {
	//			if(getMethod != null)
	//				getMethod.releaseConnection();
	//		}
	//
	//	}



	public static Session getSession(String user) throws InternalErrorException  {

		//		initializeRepository();
		//		synchronized (repository) {
		try {
			logger.debug("session of " + user);
			Session session = repository.login( 
					new SimpleCredentials(user, JCRUserManager.getSecurePassword(user).toCharArray()));
			return session;
		} catch (Exception e) {
			throw new InternalErrorException(e);
		}
		//		}
	}

	public static Session getSession() throws InternalErrorException  {

		initializeRepository();
		synchronized (repository) {
			try {
				Session session = repository.login( 
						new SimpleCredentials(user, pass.toCharArray()));

//				System.out.println("********************* JACKRABBIT REPOSITORY *****************");
				return session;
			} catch (Exception e) {
				throw new InternalErrorException(e);
			}
		}
	}

	public synchronized static void initialize() throws InternalErrorException {

		logger.debug("Initialize repository");

		Session session = getSession();
		try {
			Node root = session.getRootNode();
			 String[] list = new String[] {HOME_FOLDER, GCUBE_FOLDER, SHARED_FOLDER};
 			NodeIterator nodes = root.getNodes(list);

			List<String> listNodes = new LinkedList<String>();
			while (nodes.hasNext()){
				Node myNode =nodes.nextNode();
				String nodeName = myNode.getName();
				listNodes.add(nodeName);
				
				switch (nodeName) {
				case HOME_FOLDER:
//					homeFolderId = myNode.getIdentifier();
					break;
				case GCUBE_FOLDER:
					gcubeRootId = myNode.getIdentifier();
					break;
				case SHARED_FOLDER:
					sharedRootId = myNode.getIdentifier();
					break;
				default:
					break;
				}
				
			}
			
			List<String> myList = new LinkedList<String>(Arrays.asList(list));
//			System.out.println("list: " + myList.toString());
			myList.removeAll(listNodes);
//			System.out.println("diff: " + myList.toString());
			
			if (myList.size()!=0){
				createMissingRootFolders(root, myList);
				session.save();
			}
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	private static void createMissingRootFolders(Node root, List<String> missingFolders) throws InternalErrorException {
		logger.trace("Creating Root folders: " + missingFolders.toString());
		try{
			for (String folder : missingFolders)
				switch (folder) {
				case HOME_FOLDER:
					Node homeFolder = root.addNode(HOME_FOLDER, NT_FOLDER);
//					homeFolderId = homeFolder.getIdentifier();
					break;
				case GCUBE_FOLDER:
					Node gCubeFolder = root.addNode(GCUBE_FOLDER, NT_FOLDER);
					gcubeRootId = gCubeFolder.getIdentifier();
					break;
				case SHARED_FOLDER:
					Node sharedFolder = root.addNode(SHARED_FOLDER, NT_FOLDER);
					sharedRootId = sharedFolder.getIdentifier();
					break;
				default:
					break;
				}
		} catch (RepositoryException e) {
			logger.error("Repository not instantiated", e);
			throw new InternalErrorException(e);
		}
		
	}

	public JCRRepository(final User user) throws InternalErrorException {

//		System.out.println("JCRRepository");
		Session session = getSession();
		try {
			Node root = session.getRootNode();
			Node home = root.getNode(HOME_FOLDER);

			//			System.out.println("get user home");
			Node userHome;
			if (!exist(home, user.getPortalLogin())) {
				userHome = home.addNode(user.getPortalLogin(), NT_HOME);
			} else {
				userHome = home.getNode(user.getPortalLogin());
				//				System.out.println("GETNODE");
			}

			//			session.save();
			userHomeId = userHome.getIdentifier();

			JCRUserManager um = new JCRUserManager();
			um.createUser(user.getPortalLogin());


			String[] list = new String[] {SMART_FOLDER, IN_BOX_FOLDER, OUT_BOX_FOLDER, HIDDEN_FOLDER, DOWNLOADS};

			NodeIterator node = userHome.getNodes(list);

			List<String> listNodes = new ArrayList<String>();
			
			while (node.hasNext()){
				Node myNode = node.nextNode();
				String nodeName = myNode.getName();
				listNodes.add(nodeName);

				switch (nodeName) {
				case SMART_FOLDER:
					smartFoldersId = myNode.getIdentifier();
					break;
				case IN_BOX_FOLDER:
					inBoxId = myNode.getIdentifier();
					break;
				case OUT_BOX_FOLDER:
					outBoxId = myNode.getIdentifier();
					break;
				case HIDDEN_FOLDER:
					hiddenFolderId = myNode.getIdentifier();
					break;
				case DOWNLOADS:
					downloadsId = myNode.getIdentifier();
					break;
				default:
					break;
				}
			}


			List<String> myList = new LinkedList<String>(Arrays.asList(list));

			myList.removeAll(listNodes);

			if (myList.size()!=0){
				createMissingFolders(userHome, myList);
				userHome.getSession().save();
			}


		} catch (RepositoryException e) {
			logger.error("Repository not instantiated", e);
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	
	/**
	 * Create missing folders in user workspace
	 * @param userHome
	 * @param missingFolders
	 * @throws InternalErrorException
	 */
	private void createMissingFolders(Node userHome, List<String> missingFolders) throws InternalErrorException {

		logger.trace("Creating folders: " + missingFolders.toString());
		try{
			for (String folder : missingFolders)
				switch (folder) {
				case SMART_FOLDER:
					Node smartFolders = userHome.addNode(SMART_FOLDER, NT_FOLDER);
					smartFoldersId = smartFolders.getIdentifier();
					break;
				case IN_BOX_FOLDER:
					Node inBoxNode = userHome.addNode(IN_BOX_FOLDER, NT_ROOT_ITEM_SENT);
					inBoxId = inBoxNode.getIdentifier();
					break;
				case OUT_BOX_FOLDER:
					Node outBoxNode = userHome.addNode(OUT_BOX_FOLDER, NT_ROOT_ITEM_SENT);
					outBoxId = outBoxNode.getIdentifier();
					break;
				case HIDDEN_FOLDER:
					Node hiddenFolder = userHome.addNode(HIDDEN_FOLDER, NT_FOLDER);
					hiddenFolderId = hiddenFolder.getIdentifier();
					break;
				case DOWNLOADS:
					Node downloads = userHome.addNode(DOWNLOADS, NT_ROOT_FOLDER_BULK_CREATOR);
					downloadsId = downloads.getIdentifier();
					break;
				default:
					break;
				}
		} catch (RepositoryException e) {
			logger.error("Repository not instantiated", e);
			throw new InternalErrorException(e);
		}


	}

	public static void removeUser(User user) throws Exception {

		Session session = getSession();
		Node node = session.getRootNode().getNode(HOME_FOLDER).getNode(user.getPortalLogin());
		node.remove();
		session.save();
	}


	public static void setACL(final String portalLogin , String userHome)
			throws InternalErrorException {


		Session session = null;
		try {

			Repository RMIrepository = new URLRemoteRepository(url + "/rmi");
			session = RMIrepository.login( 
					new SimpleCredentials(user, pass.toCharArray()));

			AccessControlManager accessControlManager = session.getAccessControlManager();
			AccessControlPolicyIterator acpIterator = accessControlManager.getApplicablePolicies(userHome);
			if (acpIterator.hasNext()) {
				logger.debug(" ------------ ACL Present ");
				AccessControlPolicy policy = acpIterator.nextAccessControlPolicy();
				AccessControlList list = (AccessControlList)policy;
				list.addAccessControlEntry(new Principal() {

					@Override
					public String getName() {
						return portalLogin;
					}
				}, new Privilege[]{accessControlManager.privilegeFromName(Privilege.JCR_READ)});

				accessControlManager.setPolicy(userHome, list);
			}

			session.save();
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			if(session != null)
				session.logout();
		}
	}


	public void setScope(Session session, String scope) throws RepositoryException {

		Node userHome = getUserHome(session);

		Property scopes;
		try {
			scopes = userHome.getProperty(SCOPES);
		} catch (PathNotFoundException e) {
			String[] values = {scope};
			userHome.setProperty(SCOPES, values);
			return;
		}

		for (Value value : scopes.getValues()) {
			if(value.getString().equals(scope))
				return;
		}

		String[] newScopes = new String[scopes.getValues().length + 1];
		newScopes[0] = scope;
		for (int i = 1; i < newScopes.length; i++) {
			newScopes[i] = scopes.getValues()[i - 1].getString();
		}
		userHome.setProperty(SCOPES, newScopes);
	}


	private static boolean exist(Node parent, String childName) throws RepositoryException {

		try {
			parent.getNode(childName);
		} catch (PathNotFoundException e) {
			return false;
		} 
		return true;
	}

	/*
	public boolean existChild(Node father, String child){

		try {
			father.getNode(child);
		} catch (RepositoryException e) {
			return false;
		}
		return true;
	}*/

	public List<String> listScopes() throws RepositoryException, InternalErrorException {

		List<String> list = new LinkedList<String>();

		Session session = getSession();
		try {
			Node userHome = session.getNodeByIdentifier(userHomeId);
			Property scopes = userHome.getProperty(SCOPES);

			for (Value value  : scopes.getValues()) {
				list.add(value.getString());
			}
			return list;

		} catch (PathNotFoundException e) {
			return new LinkedList<String>();
		} finally {
			if (session != null)
				session.logout();
		}
	}

	public static Node getGCubeRoot(Session session) throws RepositoryException {
		return session.getNodeByIdentifier(gcubeRootId);
	}

	public static Node getSharedRoot(Session session) throws RepositoryException {
		return session.getNodeByIdentifier(sharedRootId);
	}

	public Node getUserHome(Session session) throws RepositoryException {		
		return  session.getNodeByIdentifier(userHomeId); 
	}

	public Node getRootSmartFolders(Session session) throws RepositoryException {
		return  session.getNodeByIdentifier(smartFoldersId);
	}

	public Node getOwnInBoxFolder(Session session) throws RepositoryException {
		return  session.getNodeByIdentifier(inBoxId);
	}

	public Node getRootFolderBulkCreators(Session session) throws RepositoryException {
		return session.getNodeByIdentifier(downloadsId);
	}

	public Node getOutBoxFolder(Session session) throws RepositoryException {
		return session.getNodeByIdentifier(outBoxId);
	}

	public Node getHiddenFolder(Session session) throws RepositoryException {
		return session.getNodeByIdentifier(hiddenFolderId);
	}


	public Node getInBoxFolder(Session session, String user) throws RepositoryException,
	InternalErrorException  {
		return session.getRootNode().getNode(HOME_FOLDER + PATH_SEPARATOR 
				+ user + PATH_SEPARATOR + IN_BOX_FOLDER);
	}

	public String getUserHomeUrl(String portalLogin) {
		return url + ROOT_WEBDAV + HOME_FOLDER + PATH_SEPARATOR + portalLogin;     
	}

	public String getWebDavUrl(String portalLogin) {
		return webdavUrl + PATH_SEPARATOR + portalLogin;      
	}


}
