package org.gcube.common.homelibrary.jcr.workspace.trash;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItem;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItemType;
import org.gcube.common.homelibrary.home.workspace.exceptions.InsufficientPrivilegesException;
import org.gcube.common.homelibrary.home.workspace.trash.WorkspaceTrashItem;
import org.gcube.common.homelibrary.jcr.repository.JCRRepository;
import org.gcube.common.homelibrary.jcr.repository.external.GCUBEStorage;
import org.gcube.common.homelibrary.jcr.workspace.JCRWorkspace;
import org.gcube.common.homelibrary.jcr.workspace.JCRWorkspaceFolder;
import org.gcube.contentmanagement.blobstorage.transport.backend.RemoteBackendException;

public class JCRWorkspaceTrashItem extends JCRWorkspaceFolder implements WorkspaceTrashItem {

	protected static final String CONTENT				= "jcr:content";
	protected static final String DELETE_DATE			= "hl:deletedTime";
	protected static final String DELETE_USER			= "hl:deletedBy";
	protected static final String ORIGINAL_PARENT_ID	= "hl:originalParentId";
	protected static final String ORIGINAL_PATH			= "hl:deletedFrom";
	protected static final String NAME					= "hl:name";
	protected static final String MIME_TYPE				= "hl:mimeType";
	protected static final String LENGTH				= "hl:length";
	protected static final String IS_FOLDER				= "hl:isFolder";


	private final JCRWorkspace workspace;

	private String identifier;
	private Calendar deletedTime;
	//	private String name;
	private String originalParentId;
	private String deletedBy;
	private String mimeType;
	private long length;
	private boolean isFolder;
	private String deletedFrom;
	private String trashId;




	public JCRWorkspaceTrashItem(JCRWorkspace workspace, Node node) throws RepositoryException {
		super(workspace, node);
		
		this.identifier = node.getIdentifier();
		this.workspace = workspace;
		this.trashId = workspace.getTrashId();
		
		PropertyIterator properties = node.getProperties(new String[]{"hl:*"});
		while (properties.hasNext()){
			Property property = properties.nextProperty();
			String propertyName = property.getName();
			switch (propertyName) {
			case DELETE_DATE: 
				this.deletedTime = property.getDate();
			break;
			case DELETE_USER: 
				this.deletedBy = property.getString();
			break;
			case ORIGINAL_PATH: 
				this.deletedFrom = property.getString();
			break;
			case ORIGINAL_PARENT_ID: 
				this.originalParentId = property.getString();
			break;
			case IS_FOLDER: 
				this.isFolder = property.getBoolean();
			break;
			case MIME_TYPE: 
				this.mimeType = property.getString();
			break;
			case LENGTH: 
				this.length = property.getLong();
			break;
			default: ;
			break;
			}

		}
		
//		try{
//			this.deletedTime = node.getProperty(DELETE_DATE).getDate();
//			this.deletedBy = node.getProperty(DELETE_USER).getString();
//			this.deletedFrom = node.getProperty(ORIGINAL_PATH).getString();
//		}catch (Exception e) {
//			// TODO: handle exception
//		}
//
//		this.originalParentId = node.getProperty(ORIGINAL_PARENT_ID).getString();
//		this.isFolder = node.getProperty(IS_FOLDER).getBoolean();
//
//		try{
//			this.mimeType = node.getProperty(MIME_TYPE).getString();
//		}catch (Exception e) {}
//		try{
//			this.length = node.getProperty(LENGTH).getLong();
//		}catch (Exception e) {}

	}



	public JCRWorkspaceTrashItem(JCRWorkspace workspace, Node node,
			String name, String description, Calendar date, String portalLogin,
			String originalParentId, String mimeType, long length, boolean isFolder, String originalPath) throws RepositoryException {
		super(workspace, node, name, description);
		this.identifier = node.getIdentifier();
		this.workspace = workspace;

		this.deletedBy = portalLogin;
		this.deletedTime = date;
		this.originalParentId = originalParentId;
		this.deletedFrom = originalPath;

		this.mimeType = mimeType;
		this.length = length;
		this.isFolder = isFolder;

		node.setProperty(NAME, name);
		node.setProperty(DELETE_DATE,Calendar.getInstance());
		node.setProperty(DELETE_USER, portalLogin);
		node.setProperty(ORIGINAL_PARENT_ID, originalParentId);
		node.setProperty(ORIGINAL_PATH, originalPath);

		node.setProperty(MIME_TYPE, mimeType);
		node.setProperty(LENGTH,length);
		node.setProperty(IS_FOLDER,isFolder);


	}
	
	@Override
	public void restore() throws InternalErrorException {
		Session session = JCRRepository.getSession();		
		Node node = null;
		try {
			//			System.out.println("getName() "+ getName());
			node = session.getNodeByIdentifier(identifier).getNode(getName());
			//			node = session.getNodeByIdentifier(identifier);
			//			System.out.println("node " + node.getPath());
			Node originalParentNode = null;
			try{
				originalParentNode = session.getNodeByIdentifier(originalParentId);
			}catch (Exception e) {
				logger.warn("The original Parent Node does not exist anymore");
			}
			String parentId = null;
			if (originalParentNode != null)
				parentId = originalParentNode.getParent().getIdentifier();
			if ((originalParentNode != null) && (!parentId.equals(trashId))) {
				//			WorkspaceItem item = workspace.getWorkspaceItem(node);
				//			System.out.println("originalPath " + originalParentId +  " - path: " + originalParentNode.getPath());

				//restore to the original parent with the original name
				try{
					workspace.moveItem(node.getIdentifier(), originalParentId);
				}catch (Exception e) {

					//restore using the original name + time stamp
					try{
						String time = timeStampRename();
						//						es. "2012_04_05_11400029_MyAwesomeFile.txt"
						setName(node, time + "_" +getName());
						//						workspace.renameItem(node.getIdentifier(), time + "_" +getName());
						workspace.moveItem(node.getIdentifier(), originalParentId);
					}catch (Exception e1) {
						throw new InternalErrorException(e1);
					}

				}
			}else {
				//restore to the workspace root (the original parent does not exists anymore)
				try{
					workspace.moveItem(node.getIdentifier(), workspace.getRoot().getId());
				}catch (Exception e1) {
					throw new InternalErrorException("The file has been restored in the root");
				}
			}


			//remove trash item (just the folder)
			session.getNodeByIdentifier(identifier).remove();
			session.save();

		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

	}


	/**
	 * Get Time Stamp to rename a file
	 * @return
	 */
	private static String timeStampRename() {
		//		"2012_04_05_11400029_MyAwesomeFile.txt"
		Date date = new Date();
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy_MM_dd_HHmmssss_");
		String formattedDate = sdf.format(date);
		//		System.out.println(formattedDate);
		return formattedDate;

	}



	@Override
	public void deletePermanently() throws InternalErrorException {

		Session session = null;
		Node node = null;
		try {
			session = JCRRepository.getSession();
			node = session.getNodeByIdentifier(identifier);
			WorkspaceItem item = workspace.getWorkspaceItem(node);
			logger.debug("deletePermanently node: " + item.getName());

			//remove the content of the trash item
			try{
				GCUBEStorage.removeRemoteFolder(node.getPath(), workspace.getOwner().getPortalLogin());
				//				removeRemoteContent(node.getNode(item.getName()), node.getNode(item.getName()).getType());
			}catch (Exception e) {
				logger.warn("Error removing " + node.getPath() + " from storage", e);
			}

			//remove the trash item
			node.remove();
			session.save();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}


	@Override
	public WorkspaceItemType getType() {
		return WorkspaceItemType.TRASH_ITEM;
	}



	public long getLength() throws InternalErrorException {
		return length;
	}

	@Override
	public String getMimeType() throws InternalErrorException {
		return mimeType;
	}


	@Override
	public String getDeletedFrom() {
		return deletedFrom;
	}


	//getter methods
	@Override
	public String getOriginalParentId() {
		return originalParentId;
	}

	@Override
	public Calendar getDeletedTime() {
		return deletedTime;
	}

	@Override
	public String getDeletedBy() {
		return deletedBy;
	}

	@Override
	public boolean isFolder() {
		return isFolder;
	}


	@Override
	public String getName() throws InternalErrorException {	

		String name = null;
		Session session = null;
		Node node = null;
		try {
			session = JCRRepository.getSession();
			node = session.getNodeByIdentifier(identifier);
			name =	node.getProperty(NAME).getString();
		} catch (RepositoryException e) {
//			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

		return name;
	}


	public void setName(Node node, String name) throws RepositoryException {

		try {
			node.setProperty(NAME, name);
			node.getSession().save();
		} catch (Exception e) {
			logger.error("Error setting a new name to WorkspaceTrashItem: " + e);
		} 

	}



	@Override
	public void removeChild(WorkspaceItem child) throws InternalErrorException,
	InsufficientPrivilegesException {}





}
